/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/State.h>

namespace Lum {

  State::State(bool active)
  : boolModel(NULL),intModel(NULL),index(1),active(active)
  {
    SetCanFocus(active);
    if (active) {
      RequestFocus();
      SetRedrawOnMouseActive(true);
    }
    SetRequestLabelFocus(true);
  }

  State::~State()
  {
    // no code
  }

  bool State::IsIn(const OS::MouseEvent& event) const
  {
    return PointIsIn(event) || (label!=NULL && label->PointIsIn(event));
  }

  void State::SetIndex(int index)
  {
    this->index=index;
  }

  bool State::SetModel(Base::Model* model)
  {
    boolModel=dynamic_cast<Model::Boolean*>(model);
    intModel=dynamic_cast<Model::Int*>(model);

    if (boolModel.Valid()) {
      Control::SetModel(boolModel);
    }
    else {
      Control::SetModel(intModel);
    }

    return (GetModel()!=NULL);
  }

  bool State::HandleMouseEvent(const OS::MouseEvent& event)
  {
    if (!active || !visible || GetModel()==NULL || !GetModel()->IsEnabled() ||
        (intModel.Valid() && intModel->IsNull()) ||
        (boolModel.Valid() && boolModel->IsNull())) {
      return false;
    }

    if (event.type==OS::MouseEvent::down && IsIn(event) && event.button==OS::MouseEvent::button1) {
      if (boolModel.Valid()) {
        orgState=boolModel->Get();
        boolModel->Set(!orgState);
      }
      else {
        orgState=intModel->Get();
        intModel->Set(index);
      }

      return true;
    }
    else if (event.type==OS::MouseEvent::move && event.IsGrabed()) {
      if (IsIn(event)) {
        if (boolModel.Valid()) {
          boolModel->Set(orgState==0);
        }
        else {
          intModel->Set(index);
        }
      }
      else {
        if (boolModel.Valid()) {
          boolModel->Set(orgState!=0);
        }
        else {
          intModel->Set(orgState);
        }
      }
      return true;
    }

    return false;
  }


  bool State::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (Control::HandleKeyEvent(event)) {
      return true;
    }

    if (!active) {
      return false;
    }

    if (event.type==OS::KeyEvent::down && event.key==OS::keySpace) {
      if (boolModel.Valid()) {
        boolModel->Toggle();
        return true;
      }
      else if (intModel.Valid()) {
        intModel->Set(index);
        return true;
      }
    }

    return false;
  }

  void State::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    if (IsMouseActive()) {
      draw->activated=true;
    }

    draw->focused=HasFocus();

    Control::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      draw->activated=false;
      draw->focused=false;
      return;
    }

    /* --- */

    if (boolModel.Valid() && !boolModel->IsNull() && boolModel->Get()) {
      draw->selected=true;
    }
    else if (intModel.Valid() && !intModel->IsNull() && intModel->Get()==index) {
      draw->selected=true;
    }

    DrawContent(x,y,w,h);

    draw->selected=false;
    draw->activated=false;
    draw->focused=false;
  }

  void State::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (visible && (model==boolModel || model==intModel)) {
      Redraw();
    }

    Control::Resync(model,msg);
  }

  ImageState::ImageState(OS::Theme::ObjectType objectType, bool active)
  : State(active),
    object(objectType)
  {
    // no code
  }

  void ImageState::CalcSize()
  {
    OS::display->GetTheme()->GetObjectSize(object,minWidth,minHeight);
    width=minWidth;
    height=minHeight;

    State::CalcSize();
  }

  void ImageState::DrawContent(int x, int y, size_t w, size_t h)
  {
    if (OS::display->GetTheme()->NeedsObjectParentDraw(object)) {
      DrawParentBackground(x,y,width,height);
    }

    OS::display->GetTheme()->DrawObject(GetDrawInfo(),object,
                                        this->x,this->y,this->width,this->height,
                                        x,y,w,h);
  }

}
