#ifndef LUM_COMBO_H
#define LUM_COMBO_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/Size.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Image.h>

#include <Lum/Dialog.h>
#include <Lum/Image.h>
#include <Lum/Object.h>
#include <Lum/Popup.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/TableView.h>

namespace Lum {
  /**
    Abstract base class for combo box like objects.
    A combo box is a object that shows one currently selected
    value (which depending on implementation might even be directly
    editable) and offers a popup window to offers a list
    of alternative values.

    The baseclass offers a framework for such functionality.
    It defines the needed models, the necessary drawing
    methods, the handling for opening and closing the
    popup window, callbacks for setting the new value on
    selection etc..
  */
  class LUMAPI Combo : public Control
  {
  private:
    class Table : public ::Lum::TableView
    {
    public:
      bool             success;

    public:
      Table();
      ~Table();

      void Resync(Base::Model* model, const Base::ResyncMsg& msg);
    };

    class ComboPopup : public Popup
    {
    public:
      Combo *combo;

    public:
      ComboPopup(Combo *combo);
      ~ComboPopup();
      void PreInit();
      void Resync(Base::Model* model, const Base::ResyncMsg& msg);
    };

  private:
    bool                          imageRight;
    bool                          editable;
    ::Lum::Table                  *table;
    Popup*                        popup;

  protected:
    Model::ActionRef              prevAction;
    Model::ActionRef              nextAction;
    Model::TableRef               tableModel; /** The table model used for the popup menu */
    Model::SingleLineSelectionRef selection;
    Object                        *value; /**
                                           The object used to display the current
                                           value. This might even allow implace
                                           editing like for example @otype{S.String}.
                                          */

  private:
    void OpenPopup();
    void SetTableRow(size_t row);

  protected:
    Combo(Object* value, bool editable);
    virtual ~Combo();

    virtual void CopySelection(size_t row) = 0;

  public:
    bool HasBaseline() const;
    size_t GetBaseline() const;

    Object* GetFocusObject();

    bool VisitChildren(Visitor &visitor, bool onlyVisible);

    void SetTableModel(Model::Table* model);

    void CalcSize();
    void Layout();

    virtual void InitTable(::Lum::Table* table);

    bool HandleMouseEvent(const OS::MouseEvent& event);
    bool HandleKeyEvent(const OS::KeyEvent& event);

    void Draw(int x, int y, size_t w, size_t h);
    void Hide();

    void DrawFocus();
    void HideFocus();

    void Resync(Base::Model* model, const Base::ResyncMsg& msg);
  };

  class LUMAPI TextCombo : public Combo
  {
  private:
    Model::StringRef model; /** The model for the current value */

  protected:
    void CopySelection(size_t row);

  public:
    TextCombo();
    ~TextCombo();

    bool SetModel(Base::Model* model);

    void Resync(Base::Model* model, const Base::ResyncMsg& msg);
  };

  class LUMAPI IndexCombo : public Combo
  {
  private:
    Model::SizeTRef model; /** The model for the current value */

  protected:
    void CopySelection(size_t row);

  public:
    IndexCombo();
    ~IndexCombo();

    bool SetModel(Base::Model* model);

    void Resync(Base::Model* model, const Base::ResyncMsg& msg);
  };

  class LUMAPI TextEditCombo : public Combo
  {
  private:
    String           *string;
    Model::StringRef model; /** The model for the current value */

  protected:
    void CopySelection(size_t row);

  public:
    TextEditCombo();
    ~TextEditCombo();

    bool SetModel(Base::Model* model);

    String* GetString() const;
  };
}

#endif
