/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Tab.h>

#include <Lum/OS/Theme.h>

#include <Lum/Dialog.h>
#include <Lum/Text.h>

namespace Lum {

  /* ----------------------- */

  class TabButton : public Control
  {
  private:
    Object          *image;      //! The label of the button
    Model::SizeTRef model;
    size_t          index;
    size_t          orgIndex;

  public:
    TabButton(size_t index, Object* object, const Model::SizeTRef& model);
    ~TabButton();

    void CalcSize();
    void Layout();
    void Draw(int x, int y, size_t w, size_t h);
    void Hide();

    bool HandleMouseEvent(const OS::MouseEvent& event);
    bool HandleKeyEvent(const OS::KeyEvent& event);

    OS::Color GetTextColor(OS::DrawInfo *draw) const;

    void Resync(Base::Model* model, const Base::ResyncMsg& msg);
  };

  TabButton::TabButton(size_t index, Object* object, const Model::SizeTRef& model)
  : image(NULL),model(model),index(index)
  {
    SetBackground(OS::display->GetFill(OS::Display::tabRiderBackgroundFillIndex));

    this->model=model;
    SetModel(model.Get());

    SetCanFocus(true);
    RequestFocus();

    this->image=object;
    this->image->SetParent(this);
  }

  TabButton::~TabButton()
  {
    delete image;
  }

  void TabButton::CalcSize()
  {
    /* Let the frame calculate its size */
    width=2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
    height=2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);

    /* Our minimal size is equal to the normal size */
    minWidth=width;
    minHeight=height;

    image->CalcSize();
    width+=image->GetOWidth();
    height+=image->GetOHeight();
    minWidth+=image->GetOMinWidth();
    minHeight+=image->GetOMinHeight();

    /* We *must* call CalcSize of our superclass! */
    Control::CalcSize();
  }

  bool TabButton::HandleMouseEvent(const OS::MouseEvent& event)
  {
    /* It makes no sense to get the focus if we are currently not visible */
    if (!visible || !model.Valid() || !model->IsEnabled()) {
      return false;
    }

    /*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
    */

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      orgIndex=model->Get();
      model->Set(index);
      return true;
    }
    else if (event.IsGrabEnd()) {
      /*
        If the users released the left mousebutton over our bounds we really
        got selected.
      */
      if (PointIsIn(event)) {
        model->Set(index);
      }
      else {
        model->Set(orgIndex);
      }
    }
    else if (event.type==OS::MouseEvent::move && event.IsGrabed()) {
      if (PointIsIn(event)) {
        model->Set(index);
      }
      else {
        model->Set(orgIndex);
      }

      return true;
    }

    return false;
  }

  bool TabButton::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (event.type==OS::KeyEvent::down) {
      if (event.key==OS::keySpace) {
        model->Set(index);

        return true;
      }
      else if (event.key==OS::keyLeft) {
        dynamic_cast<Tab*>(parent)->ActivatePreviousTab();
      }
      else if (event.key==OS::keyRight) {
        dynamic_cast<Tab*>(parent)->ActivateNextTab();
      }
    }

    return false;
  }

  OS::Color TabButton::GetTextColor(OS::DrawInfo *draw) const
  {
    if (draw->selected) {
      return OS::display->GetColor(OS::Display::tabTextSelectColor);
    }
    else {
      return OS::display->GetColor(OS::Display::tabTextColor);
    }
  }

  /*
     We tell the image to resize themself to
     our current bounds. Our bounds could have changed
     because Resize may have been called by some layout-objects
     between TabButton.CalcSize and TabButton.Draw.
   */
  void TabButton::Layout()
  {
    image->Resize(width-2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder),
                  height-2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));
    image->Move(x+(width-image->GetOWidth()) / 2,
                y+(height-image->GetOHeight()) / 2);

    Control::Layout();
  }

  void TabButton::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    if (IsMouseActive()) {
      draw->activated=true;
    }

    if (!model->IsNull() && model->Get()==index) {
      draw->selected=true;
    }

    draw->focused=HasFocus();

    Control::Draw(x,y,w,h); /* We must call Draw of our superclass */

    if (!OIntersect(x,y,w,h)) {
      draw->activated=false;
      draw->selected=false;
      draw->focused=false;
      return;
    }

    /* --- */

    draw->PushClipBegin(x,y,w,h);
    draw->SubClipRegion(image->GetOX(),image->GetOY(),
                        image->GetOWidth(),image->GetOHeight());
    draw->PushClipEnd();
    DrawBackground(x,y,w,h);
    draw->PopClip();

    image->Draw(x,y,w,h);

    draw->activated=false;
    draw->selected=false;
    draw->focused=false;
  }

  void TabButton::Hide()
  {
    if (visible) {
      /* Hide the image */
      image->Hide();

      /* hide the frame */
      Control::Hide();
    }
  }

  void TabButton::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model && visible) {
      Redraw();
    }

    Control::Resync(model,msg);
  }

  /* ----------------------- */

  Tab::Tab()
  : model(new Model::SizeT(0)),mw(0),mh(0),lastDrawn(NULL)
  {
    SetBackground(OS::display->GetFill(OS::Display::tabBackgroundFillIndex));

    SetCanFocus(true);

    AttachModel(model);
  }

  Tab::~Tab()
  {
    UnattachModel(model);

    std::list<Entry>::iterator iter;

    iter=list.begin();
    while (iter!=list.end()) {
      delete iter->rider;
      delete iter->object;

      ++iter;
    }
  }

  bool Tab::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    for (std::list<Entry>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
      if (!visitor.Visit(iter->rider)) {
        return false;
      }

      if (onlyVisible) {
        if (iter->object->IsVisible() && !visitor.Visit(iter->object)) {
          return false;
        }
      }
      else {
        if (!visitor.Visit(iter->object)) {
          return false;
        }
      }
    }

    return true;
  }

  void Tab::Add(Object* label, Object* object)
  {
    assert(label!=NULL && object!=NULL);

    Entry entry;

    entry.rider=new TabButton(list.size(),label,model);
    entry.rider->SetParent(this);
    entry.rider->SetFlex(true,true);

    entry.object=object;
    entry.object->SetParent(this);

    list.push_back(entry);
  }

  void Tab::Add(const std::wstring& label, Object* object)
  {
    Add(new Text(label),object);
  }

  void Tab::ActivatePreviousTab()
  {
    if (list.size()<=1) {
      return;
    }

    if (model->Get()==0) {
      model->Set(list.size()-1);
    }
    else {
      model->Dec();
    }
  }

  void Tab::ActivateNextTab()
  {
    if (list.size()<=1) {
      return;
    }

    if (model->Get()==list.size()-1) {
      model->Set(0);
    }
    else {
      model->Inc();
    }
  }

  bool Tab::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::SizeT*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void Tab::CalcSize()
  {
    std::list<Entry>::iterator iter;
    size_t                     ow,oh;

    width=0;
    height=0;
    mw=0;
    mh=0;
    ow=0;
    oh=0;

    iter=list.begin();
    while (iter!=list.end()) {
      iter->rider->CalcSize();
      mw=std::max(mw,(size_t)iter->rider->GetOWidth());
      mh=std::max(mh,(size_t)iter->rider->GetOHeight());

      iter->object->CalcSize();
      ow=std::max(ow,(size_t)iter->object->GetOWidth());
      oh=std::max(oh,(size_t)iter->object->GetOHeight());

      ++iter;
    }

    OS::FrameRef tabFrame=OS::display->GetFrame(OS::Display::tabFrameIndex);

    mh=std::max(mh,tabFrame->topBorder);

    ow+=2*OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    oh+=2*OS::display->GetSpaceVertical(OS::Display::spaceGroupIndent);

    height=mh+oh+tabFrame->minHeight;
    width=std::max(ow+tabFrame->minWidth,
                   list.size()*mw+
                   OS::display->GetTheme()->GetFirstTabOffset()+
                   OS::display->GetTheme()->GetLastTabOffset());

    minWidth=width;
    minHeight=height;

    Control::CalcSize();
  }

  Object* Tab::GetObject() const
  {
    size_t pos=0;

    for (std::list<Entry>::const_iterator iter=list.begin(); iter!=list.end(); ++iter) {
      if (pos==model->Get()) {
        return iter->object;
      }

      pos++;
    }

    return NULL;
  }

  void Tab::Layout()
  {
    int     xPos,yPos;
    size_t  width,height;
    size_t  pos;

    OS::FrameRef tabFrame=OS::display->GetFrame(OS::Display::tabFrameIndex);

    xPos=x+tabFrame->leftBorder;
    yPos=y+mh+tabFrame->topBorder;
    width=this->width-tabFrame->minWidth;
    height=this->height-mh-tabFrame->minHeight;

    pos=x+OS::display->GetTheme()->GetFirstTabOffset();
    for (std::list<Entry>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
      // Position rider relative to last rider
      iter->rider->MoveResize(pos,y,mw,mh);
      pos+=mw;

      // Center object in object area
      iter->object->Resize(width-2*OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent),
                           height-2*OS::display->GetSpaceVertical(OS::Display::spaceGroupIndent));

      iter->object->Move(xPos+(width-iter->object->GetOWidth()) / 2,
                         yPos+(height-iter->object->GetOHeight()) / 2);
    }

    Control::Layout();
  }

  void Tab::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();
    OS::FrameRef tabFrame=OS::display->GetFrame(OS::Display::tabFrameIndex);

    lastDrawn=GetObject();

    draw->PushClipBegin(x,y,w,h);
    // Object
    draw->SubClipRegion(lastDrawn->GetOX(),lastDrawn->GetOY(),
                        lastDrawn->GetOWidth(),lastDrawn->GetOHeight());
    draw->PushClipEnd();

    // Fill background of object with tab control background color
    DrawBackground(this->x+tabFrame->leftBorder,
                   this->y+mh+tabFrame->topBorder,
                   this->width-tabFrame->minWidth,
                   this->height-mh-tabFrame->minHeight);

    draw->PopClip();

    // Draw all riders
    for (std::list<Entry>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
      iter->rider->Draw(x,y,w,h);
    }

    // Area before the first tab rider
    if (OS::display->GetTheme()->GetFirstTabOffset()>0) {
      DrawParentBackground(this->x,
                           this->y,
                           OS::display->GetTheme()->GetFirstTabOffset(),
                           this->mh);
    }

    // Fill area behind last tab rider with background color of parent control.
    if (OS::display->GetTheme()->GetFirstTabOffset()+list.size()*mw<this->width) {
      DrawParentBackground(this->x+OS::display->GetTheme()->GetFirstTabOffset()+list.size()*mw,
                           this->y,
                           this->width-list.size()*mw-OS::display->GetTheme()->GetFirstTabOffset(),
                           this->mh);
    }

    if (tabFrame->alpha) {
      draw->PushClipBegin(this->x,this->y+mh,this->width,this->height-mh);
      draw->SubClipRegion(this->x+tabFrame->leftBorder,
                          this->y+mh+tabFrame->topBorder,
                          this->width-tabFrame->minWidth,
                          this->height-mh-tabFrame->minHeight);
      draw->PushClipEnd();
      DrawParentBackground(this->x,this->y+mh,this->width,this->height-mh);
      draw->PopClip();
    }

    // Draw frame around object area
    tabFrame->SetGap(OS::display->GetTheme()->GetFirstTabOffset()+model->Get()*mw,mw,0);

    tabFrame->Draw(draw,
                   this->x,
                   this->y+mh,
                   this->width,
                   this->height-mh);

    // Draw currently visible object
    lastDrawn->Draw(x,y,w,h);
  }

  void Tab::Hide()
  {
    if (visible) {
      for (std::list<Entry>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
        iter->rider->Hide();
      }

      lastDrawn->Hide();
      lastDrawn=NULL;

      Control::Hide();
    }
  }

  void Tab::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model && visible) {
      size_t                     pos=this->model->Get();
      std::list<Entry>::iterator iter=list.begin();
      Dialog                     *dialog=dynamic_cast<Dialog*>(GetWindow()->GetMaster());

      if (lastDrawn!=NULL) {
        OS::DrawInfo *draw=GetDrawInfo();

       /*
          This is a little trick:
          We must hide the old object, but we do not want it to clean up the
          covered area, since it will be overwritten by the new object anyway.
          So we create a clipping region that completely covers the to be drawn
          area :-)
        */
        draw->PushClipBegin(lastDrawn->GetOX(),lastDrawn->GetOY(),
                            lastDrawn->GetOWidth(),lastDrawn->GetOHeight());
        draw->SubClipRegion(lastDrawn->GetOX(),lastDrawn->GetOY(),
                            lastDrawn->GetOWidth(),lastDrawn->GetOHeight());
        draw->PushClipEnd();
        lastDrawn->Hide();
        draw->PopClip();
        lastDrawn=NULL;
      }

      Redraw();

      if (dialog!=NULL) {
        while (pos>0) {
          ++iter;
          --pos;
        }

        dialog->SetFocus(iter->rider);
      }
    }

    Control::Resync(model,msg);
  }
}
