/*
  This source is part of the Illumination library
  Copyright (C) 2006  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Dlg/Progress.h>

#include <Lum/FuelGauge.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Text.h>
#include <Lum/WindowGroup.h>

namespace Lum {
  namespace Dlg {

    class Progress : public Dialog
    {
    private:
      std::wstring     caption;
      std::wstring     action;
      Model::DoubleRef percent;
      Text             *captionText;
      Text             *actionText;

    public:
      Progress(const std::wstring& caption,
               const std::wstring& action,
               Model::DoubleRef& percent)
      : caption(caption),
        action(action),
        percent(percent),
        captionText(NULL),
        actionText(NULL)
      {
      }

      ~Progress()
      {
      }

      void SetCaption(const std::wstring& caption)
      {
        this->caption=caption;
        if (captionText!=NULL) {
          captionText->SetText(caption,
                               OS::Font::italic|OS::Font::bold,Text::left,
                               OS::display->GetFont(OS::Display::fontTypeProportional,
                                                    OS::Display::fontScaleCaption1));
        }
      }

      void SetAction(const std::wstring& action)
      {
        this->action=action;
        if (actionText!=NULL) {
          actionText->SetText(action);
        }
      }

      void PreInit()
      {
        FuelGauge   *fuelGauge;
        Panel       *vPanel;
        WindowGroup *wGroup;

        vPanel=new VPanel();

        captionText=new Text(caption,
                             OS::Font::italic|OS::Font::bold,Text::left,
                             OS::display->GetFont(OS::Display::fontTypeProportional,
                                                  OS::Display::fontScaleCaption1));
        captionText->SetFlex(true,false);
        captionText->SetMinWidth(Lum::Base::Size::modePixel,
                                 Lum::Base::GetSize(Lum::Base::Size::stdCharWidth,30,
                                                    OS::display->GetFont(OS::Display::fontTypeProportional,
                                                                         OS::Display::fontScaleCaption1)));
        vPanel->Add(captionText);

        vPanel->Add(new VSpace());

        actionText=new Text();
        actionText->SetFlex(true,false);
        actionText->SetMinWidth(Lum::Base::Size::stdCharWidth,60);
        actionText->SetText(action);
        vPanel->Add(actionText);

        vPanel->Add(new VSpace());

        fuelGauge=new FuelGauge();
        fuelGauge->SetFlex(true,false);
        fuelGauge->SetModel(percent);
        vPanel->Add(fuelGauge);
    /*
        row=new ButtonRow();
        row->SetFlex(true,false);

        for (size_t x=0; x<list.size(); x++) {
          button=new Button();
          button->SetFlex(true,true);
          button->RequestFocus();
          button->SetType(list[x].type);
          button->SetText(list[x].label);
          button->SetModel(list[x].action);
          row->Add(button);
        }
        vPanel->Add(row);
    */
        wGroup=new WindowGroup();
        wGroup->SetFlex(true,false);
        wGroup->SetMain(vPanel);

        SetTop(wGroup);

        Dialog::PreInit();
      }

      void Resync(Base::Model* model, const Base::ResyncMsg& msg)
      {
        if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
          /*
          resultAction=GetClosedAction();
          Exit();
          return;*/
        }

        Dialog::Resync(model,msg);
      }
    };

    ProgressTask::ProgressTask()
    : needsJoin(false),
      finished(true),
      progress(NULL),
      timer(new Model::Action),
      openTimer(new Model::Action),
      finishedAction(new Model::Action),
      parent(NULL),
      updateCaption(false),
      updateAction(false),
      percent(0),
      percentModel(new Model::Double)
    {
      AttachModel(timer);
      AttachModel(openTimer);

      percentModel->SetRange(0,100);
    }

    ProgressTask::~ProgressTask()
    {
      OS::MutexGuard guard(mutex);

      Cleanup();

      UnattachModel(openTimer);
      UnattachModel(timer);
    }

    void ProgressTask::Cleanup()
    {
      if (needsJoin) {
        Join();
        needsJoin=false;
      }

      if (progress!=NULL) {
        progress->Close();
        delete progress;
        progress=NULL;
      }

      if (parent!=NULL && !parent->IsEnabled()) {
        parent->Enable();
      }
    }

    void ProgressTask::Resync(Base::Model* model, const Base::ResyncMsg& msg)
    {
      if (model==timer && timer->IsFinished()) {
        OS::MutexGuard guard(mutex);

        if (needsJoin) {
          OS::display->RemoveTimer(openTimer);

          Cleanup();

          finishedAction->Trigger();
        }
        else {
          percentModel->Set(percent);
          if (updateCaption) {
            if (progress!=NULL) {
              progress->SetCaption(caption);
            }
            updateCaption=false;
          }
          if (updateAction) {
            if (progress!=NULL) {
              progress->SetAction(action);
            }
            updateAction=false;
          }
          OS::display->AddTimer(0,100000,timer);
        }
      }
      else if (model==openTimer && openTimer->IsFinished()) {
        progress=new Progress(caption,action,percentModel);
        progress->SetParent(parent);
        progress->Open();
      }
    }

    void ProgressTask::Start() throw (OS::ThreadException)
    {
      finished=false;
      needsJoin=false;

      percent=0;

      OS::display->AddTimer(0,100000,timer);
      OS::display->AddTimer(0,500000,openTimer);

      if (parent!=NULL) {
        parent->Disable();
      }

      Thread::Start();
    }

    void ProgressTask::Finished()
    {
      OS::MutexGuard guard(mutex);

      finished=true;
      needsJoin=true;
    }

    bool ProgressTask::IsFinished() const
    {
      OS::MutexGuard guard(mutex);

      return finished;
    }

    void ProgressTask::SetParent(OS::Window *window)
    {
      parent=window;
    }

    void ProgressTask::SetParent(Dialog *dialog)
    {
      parent=dialog->GetWindow();
    }

    void ProgressTask::SetCaption(const std::wstring& caption)
    {
      OS::MutexGuard guard(mutex);

      this->caption=caption;
      updateCaption=true;
    }

    void ProgressTask::SetAction(const std::wstring& action)
    {
      OS::MutexGuard guard(mutex);

      this->action=action;
      updateAction=true;
    }

    void ProgressTask::SetProgress(double progress)
    {
      OS::MutexGuard guard(mutex);

      percent=progress;
    }

    Model::Action* ProgressTask::GetFinishedAction() const
    {
      return finishedAction.Get();
    }
  }
}
