/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Dialog.h>

#include <cstdlib>

#include <Lum/Base/String.h>
#include <Lum/Base/Util.h>

#include <Lum/OS/Driver.h>
#include <Lum/OS/KeyboardHandler.h>
#include <Lum/OS/Theme.h>

#include <Lum/Panel.h>

#define MENU_TIMEOUT 2

namespace Lum {
  /**
    Checks if the given shortcut entry matches the given key.
  */
  bool Dialog::Shortcut::Match(OS::KeyEvent* event) const
  {
    std::wstring  name;
    unsigned long reqQual;
    unsigned long evQual;

    event->GetName(name);
    Base::ToUpper(name);

    reqQual=this->qualifier;
    evQual=event->qualifier & ~OS::qualifierCapsLock;

    // If qualifierXXX is set, it matches both qualifierXXXLeft and qualifierXXXRight.
    // In this case just remove them on both sides of the match and check if the rest
    // matches, too. If we don't have a match in this case, the match belowe wil not succeed,
    // either.

    if ((reqQual & OS::qualifierShift) && (evQual & OS::qualifierShift)) {
      reqQual=reqQual & ~OS::qualifierShift;
      evQual=evQual & ~(OS::qualifierShiftLeft|OS::qualifierShiftRight|OS::qualifierShift);
    }

    if ((reqQual & OS::qualifierControl) && (evQual & OS::qualifierControl)) {
      reqQual=reqQual & ~OS::qualifierControl;
      evQual=evQual & ~(OS::qualifierControlLeft|OS::qualifierControlRight|OS::qualifierControl);
    }

    if ((reqQual & OS::qualifierAlt) && (evQual & OS::qualifierAlt)) {
      reqQual=reqQual & ~OS::qualifierAlt;
      evQual=evQual & ~(OS::qualifierAltLeft|OS::qualifierAltRight|OS::qualifierAlt);
    }

    if ((reqQual & OS::qualifierMeta) && (evQual & OS::qualifierMeta)) {
      reqQual=reqQual & ~OS::qualifierMeta;
      evQual=evQual & ~(OS::qualifierMetaLeft|OS::qualifierMetaRight|OS::qualifierMeta);
    }

    if ((reqQual & OS::qualifierSuper) && (evQual & OS::qualifierSuper)) {
      reqQual=reqQual & ~OS::qualifierSuper;
      evQual=evQual & ~(OS::qualifierSuperLeft|OS::qualifierSuperRight|OS::qualifierSuper);
    }

    if ((reqQual & OS::qualifierHyper) && (evQual & OS::qualifierHyper)) {
      reqQual=reqQual & ~OS::qualifierHyper;
      evQual=evQual & ~(OS::qualifierHyperLeft|OS::qualifierHyperRight|OS::qualifierHyper);
    }

    // TODO:
    // Check case insensitive

    return reqQual==evQual && key==name && (object->IsVisible() || object->ShortcutAlways());
  }

  Dialog::KeyHandler::KeyHandler(OS::Window* window)
  : window(window),
    top(NULL),current(NULL),sCurrent(NULL)
  {
    // no code
  }

  Dialog::KeyHandler::~KeyHandler()
  {
    std::list<Shortcut*>::iterator iter;

    iter=shortcuts.begin();
    while (iter!=shortcuts.end()) {
      delete *iter;

      ++iter;
    }
  }

  void Dialog::KeyHandler::SetTop(Lum::Object* top)
  {
    this->top=top;
  }

  void Dialog::KeyHandler::SetFocus(Lum::Object* gadget)
  {
    //assert(gadget==NULL || gadget->HandlesFocus());

    if (OS::display->GetKeyboardHandler()!=NULL) {
      OS::display->GetKeyboardHandler()->OnFocusChange(window,current,gadget);
    }

    if (current==gadget) {
      return;
    }

    if (current!=NULL) {
      current->LostFocus();
    }

    current=gadget;

    if (current!=NULL) {
      current->CatchedFocus();
    }
  }

  Lum::Object* Dialog::KeyHandler::GetCurrentFocusObject() const
  {
    return current;
  }

  void Dialog::KeyHandler::SetFocusFirst()
  {
    SetFocus(top->GetFocusFirst());
  }

  void Dialog::KeyHandler::SetFocusNext()
  {
    Lum::Object *object;

    if (current!=NULL) {
      object=current->GetFocusNext(current);
      if (object!=NULL) {
        SetFocus(object);
        return;
      }
    }

    object=top->GetFocusFirst();
    if (object!=NULL && object!=current) {
      SetFocus(object);
    }
  }

  void Dialog::KeyHandler::SetFocusLast()
  {
    Lum::Object *object;

    if (current!=NULL) {
      object=current->GetFocusPrevious(current);
      if (object!=NULL) {
        SetFocus(object);
        return;
      }
    }

    object=top->GetFocusFirst();
    if (object!=NULL) {
      SetFocus(object);
    }
    else {
      SetFocus(NULL);
    }
  }

  void Dialog::KeyHandler::SetFocusNextInGroup()
  {
  }

  void Dialog::KeyHandler::SetFocusLastInGroup()
  {
  }

  void Dialog::KeyHandler::Activate()
  {
    if (current!=NULL) {
      if (OS::display->GetKeyboardHandler()!=NULL) {
        OS::display->GetKeyboardHandler()->OnFocusChange(window,current,current);
      }
      current->RecatchedFocus();
    }
    else {
      SetFocusFirst();
    }
  }

  void Dialog::KeyHandler::Deactivate()
  {
    if (current!=NULL) {
      current->LostFocus();
    }
  }

  /**
    Adds a shortcut with the given @oparam{qualifier} and character for
    @oparam{object}. An object must be visible to get notified, so shortcuts
    can be shared as long as only one object at the time is visible.

    If a shortcut gets triggered, a @otype{ShortcutMsgDesc} will be send to
    @oparam{target} or @oparam{object}, if @oparam{target} is @code{NIL}.
  */
  void Dialog::KeyHandler::AddShortcut(Lum::Object* object,
                                       unsigned long qualifier,
                                       const std::wstring& key,
                                       Model::Action* action)
  {
    Shortcut     *shortcut;
    std::wstring tmp;

    shortcut=new Shortcut();

    tmp=key;
    Base::ToUpper(tmp);

    shortcut->object=object;
    shortcut->qualifier=qualifier;
    shortcut->key=tmp;
    shortcut->action=action;

    shortcuts.push_back(shortcut);
  }

  /**
    Returns the shortcut matching or NIL.
  */
  Dialog::Shortcut* Dialog::KeyHandler::GetShortcut(OS::KeyEvent* event) const
  {
    std::list<Shortcut*>::const_iterator iter;

    iter=shortcuts.begin();
    while (iter!=shortcuts.end()) {
      if ((*iter)->Match(event)) {
          return (*iter);
      }
      ++iter;
    }

    return NULL;
  }

  void Dialog::KeyHandler::CancelCurrent()
  {
    if (sCurrent!=NULL) {
      sCurrent->action->Cancel();
    }
    sCurrent=NULL;
  }

  bool Dialog::KeyHandler::HandleEvent(OS::KeyEvent* event)
  {
    Shortcut * sc;

    if (current!=NULL && current->HandleKeyEvent(*event)) {
      return true;
    }

    switch (event->type) {
    case OS::KeyEvent::down:
      sc=GetShortcut(event);
      if (sc!=NULL) {
        if (sc!=sCurrent) {
          CancelCurrent();
          sCurrent=sc;
          sc->action->Start();
        }
      }
      else {
        CancelCurrent();
      }
      break;
    case OS::KeyEvent::up:
      if (sCurrent!=NULL) {
        if (!sCurrent->Match(event)) {
          CancelCurrent();
        }
        else {
          sCurrent->action->Finish();
          sCurrent=NULL;
        }
      }
      break;
    }

    if (event->type==OS::KeyEvent::down) {
      switch (event->key) {
      case OS::keyTab:
        if (event->qualifier==0) {
          SetFocusNext();
          return true;
        }
        break;
      case OS::keyLeftTab:
        SetFocusLast();
        return true;
        break;
      case OS::keyLeft:
      case OS::keyUp:
        if (event->qualifier==0) {
          SetFocusLastInGroup();
        }
        break;
      case OS::keyRight:
      case OS::keyDown:
        if (event->qualifier==0) {
          SetFocusNextInGroup();
        }
        break;
      default:
        break;
      }
    }

    return false;
  }

  Dialog::Dialog()
  : top(NULL),
    mouseGrab(NULL),
    mouseActive(NULL),
    reference(NULL),
    frame(NULL),
    window(OS::driver->CreateWindow()),
    keyHandler(window),
    mouseGrabHunting(false),
    deleted(false),
    menuAction(new Model::Action()),
    mouse1DownX(0),
    mouse1DownY(0)
  {
    assert(window!=NULL);

    window->SetMaster(this);

    AttachModel(window->GetUnmapedAction());
    AttachModel(window->GetFocusInAction());
    AttachModel(window->GetFocusOutAction());
    AttachModel(window->GetMouseInAction());
    AttachModel(window->GetMouseOutAction());
    AttachModel(window->GetRedrawAction());
    AttachModel(window->GetResizeAction());
    AttachModel(window->GetPreInitAction());
    AttachModel(window->GetOpenedAction());
    AttachModel(window->GetEventAction());
    AttachModel(window->GetGrabCancelAction());
    AttachModel(menuAction);
  }

  Dialog::~Dialog()
  {
    deleted=true;

    delete frame;

    UnattachModel(menuAction);
    UnattachModel(window->GetGrabCancelAction());
    UnattachModel(window->GetEventAction());
    UnattachModel(window->GetOpenedAction());
    UnattachModel(window->GetPreInitAction());
    UnattachModel(window->GetResizeAction());
    UnattachModel(window->GetRedrawAction());
    UnattachModel(window->GetMouseOutAction());
    UnattachModel(window->GetMouseInAction());
    UnattachModel(window->GetFocusOutAction());
    UnattachModel(window->GetFocusInAction());
    UnattachModel(window->GetUnmapedAction());

    delete window;
  }

  OS::Window* Dialog::GetWindow() const
  {
    return window;
  }

  /**
    Returns the object under the cursor to supports the given mode.
  */
  Lum::Object* Dialog::GetPosObject(PosMode mode) const
  {
    class GetPosVisitor : public Visitor
    {
    public:
      int         x,y;
      PosMode     mode;
      Lum::Object *object;

    public:
      bool Visit(Lum::Object* object)
      {
        if (object->IsVisible() && object->PointIsIn(x,y)) {
          if (object->VisitChildren(*this,true)) {
            switch (mode) {
            case posModeMenu:
              if (this->object==NULL && object->GetMenu()!=NULL) {
                this->object=object;
                return false;
              }
              break;
            case posModeAny:
              if (this->object==NULL) {
                this->object=object;
                return false;
              }
              break;
            }
          }
        }

        return true;
      }
    };

    int           x,y;
    GetPosVisitor visitor;

    window->GetMousePos(x,y);

    visitor.x=x;
    visitor.y=y;
    visitor.mode=mode;
    visitor.object=NULL;
    /* ignore */ frame->VisitChildren(visitor,true);

    return visitor.object;
  }

  void Dialog::EvaluateMouseActive()
  {
    Lum::Object *newActive;

    if (!window->HasFocus()) {
      newActive=NULL;
    }
    else if (mouseGrab!=NULL) {
      newActive=mouseGrab;
    }
    else {
      newActive=GetPosObject(posModeAny);
    }

    if (mouseActive!=newActive) {
      if (mouseActive!=NULL) {
        mouseActive->MouseInactive();
      }

      mouseActive=newActive;

      if (mouseActive!=NULL) {
        mouseActive->MouseActive();
      }
    }
  }

  /**
    Set the top object for this window.
  */
  void Dialog::SetTop(Lum::Object* object)
  {
    top=object;
    keyHandler.SetTop(top);
  }

  void Dialog::SetReference(Lum::Object* object)
  {
    reference=object;
  }

  Lum::Object* Dialog::GetReference() const
  {
    return reference;
  }

  void Dialog::CalcSize()
  {
    Panel *panel; // overloading member variable

    panel=new HPanel();

    switch (window->GetType()) {
    case OS::Window::typeTooltip:
      panel->SetBackground(OS::display->GetFill(OS::Display::tooltipWindowBackgroundFillIndex));
      break;
    case OS::Window::typeMenu:
      panel->SetBackground(OS::display->GetFill(OS::Display::menuWindowBackgroundFillIndex));
      break;
    case OS::Window::typePopup:
    case OS::Window::typeSplash:
      panel->SetBackground(OS::display->GetFill(OS::Display::popupWindowBackgroundFillIndex));
      break;
    case OS::Window::typeMain:
      panel->SetBackground(OS::display->GetFill(OS::Display::dialogWindowBackgroundFillIndex));
      break;
    default:
      //panel->SetBackground(OS::display->GetFill(OS::Display::dialogWindowBackgroundFillIndex));
      break;
    }

    panel->SetFlex(top->HorizontalFlex(),top->VerticalFlex());
    panel->Add(top);
    panel->SetWindow(window);

    panel->CalcSize();

    if (window->GetType()==OS::Window::typePopup && reference!=NULL) {
      // Is is nice, when reference and popup have the same width
      if (reference->GetOWidth()>panel->GetOWidth()) {
        panel->ResizeWidth(reference->GetOWidth());
      }
    }
    else if (OS::display->GetType()==OS::Display::typeTextual && window->GetType()==OS::Window::typeMain) {
      panel->Resize(OS::display->GetWorkAreaWidth(),OS::display->GetWorkAreaHeight());
    }

    window->SetSize(panel->GetOWidth(),panel->GetOHeight());

    if (window->GetX()+window->GetWidth()-1>OS::display->GetScreenWidth()
        && window->GetWidth()<OS::display->GetScreenWidth()) {
      window->SetPos(OS::display->GetScreenWidth()-window->GetWidth(),
                     window->GetY());
    }
    if (window->GetY()+window->GetHeight()-1>OS::display->GetScreenHeight()
        && window->GetHeight()<OS::display->GetScreenHeight()) {
      window->SetPos(window->GetX(),
                     OS::display->GetScreenHeight()-window->GetHeight());
    }

    if (panel->HorizontalFlex()) {
      window->SetMinWidth(panel->GetOMinWidth());
      window->SetMaxWidth(panel->GetOMaxWidth());
    }
    else {
      window->SetMinWidth(window->GetWidth());
      window->SetMaxWidth(window->GetWidth());
    }

    if (panel->VerticalFlex()) {
      window->SetMinHeight(panel->GetOMinHeight());
      window->SetMaxHeight(panel->GetOMaxHeight());
    }
    else {
      window->SetMinHeight(window->GetHeight());
      window->SetMaxHeight(window->GetHeight());
    }

    this->frame=panel;
  }


  /**
     Creates a window on the display with top as top object
     and with title as title.
  */
  void Dialog::PreInit()
  {
    // no code
  }

  void Dialog::ReinitWindow()
  {
    if (IsShown()) {
      frame->Hide();
    }

    CalcSize();
    window->Resize(frame->GetOWidth(),frame->GetOHeight());

    if (IsShown()) {
      frame->Move(0,0);
      frame->Draw(0,0,window->GetWidth(),window->GetHeight());
      keyHandler.SetFocusFirst();
    }
    else {
      frame->Resize(window->GetWidth(),window->GetHeight());
    }
  }

  void Dialog::OnContextHelp()
  {
    // TODO
  }

  /**
    This method gets called, when the window things you should
    open a context sensitiv menu.

    RESULT
    Return TRUE if you have opened a context menu, else FALSE.

    NOTE
    If this method returns FALSE, the display will propagate
    the corresponding event that started contextsensitive help
    to the window.
  */
  bool Dialog::OnContextMenu()
  {
    Lum::Object *object;

    object=GetPosObject(posModeMenu);
    if (object!=NULL) {
      OS::Window* menu;

      menu=object->GetMenu();
      if (menu!=NULL) {
        object->PrepareMenu();
        /* ignore */ menu->Open(true);
        return true;
      }
    }

    return false;
  }

  void Dialog::FocusFirst()
  {
    if (deleted) {
      // Do reassign focus if our destructor has ben called
      return;
    }

    keyHandler.SetFocusFirst();
  }

  void Dialog::FocusNext()
  {
    if (deleted) {
      // Do reassign focus if our destructor has ben called
      return;
    }

    keyHandler.SetFocusNext();
  }

  void Dialog::SetFocus(Lum::Object* gadget)
  {
    //assert(gadget==NULL || gadget->HandlesFocus());
    if (deleted) {
      // Do reassign focus if our destructor has ben called
      return;
    }

    keyHandler.SetFocus(gadget);
  }

  Lum::Object* Dialog::GetFocus() const
  {
    return keyHandler.current;
  }

  void Dialog::ObjectHides(Lum::Object* object)
  {
    if (object==mouseActive) {
      mouseActive=NULL;
    }

    if (object==keyHandler.current && window->IsShown()) {
      FocusNext();
    }
  }

  /**
    The defaulthandler ask all members of the layout object for the focus.
  */
  Lum::Object* Dialog::HandleMouseEvent(const OS::MouseEvent& event)
  {
    class MouseEventVisitor : public Visitor
    {
    public:
      const OS::MouseEvent& event;
      Lum::Object           *grab;

    public:
      MouseEventVisitor(const OS::MouseEvent& event) : event(event)
      {
        // no code
      }

      bool Visit(Lum::Object* object)
      {
        object->VisitChildren(*this,true);

        if (grab!=NULL) {
          return false;
        }

        if (object->HandleMouseEvent(event)) {
          grab=object;
        }

        return grab==NULL;
      }
    };

    MouseEventVisitor visitor(event);

    visitor.grab=NULL;

    if (top!=NULL && top->IsVisible()) {
      top->VisitChildren(visitor,true);

      if (visitor.grab==NULL && top->HandleMouseEvent(event)) {
        visitor.grab=top;
      }
    }

    return visitor.grab;
  }

  bool Dialog::HandleEvent(OS::Event* event)
  {
    OS::MouseEvent *mouseEvent;
    OS::KeyEvent   *keyEvent;

    /*
      We close the window if we are popup and the user clicks
      outside the window.
    */
    switch (window->GetType()) {
    case OS::Window::typeMenu:
    case OS::Window::typePopup:
    case OS::Window::typeSplash:
      /* Close the window, if it gets deactivated */
      if ((mouseEvent=dynamic_cast<OS::MouseEvent*>(event))!=NULL) {
        if (mouseEvent->type==OS::MouseEvent::down && !window->CursorIsIn()) {
          window->Exit();
          return true;
        }
      }
      else if ((keyEvent=dynamic_cast<OS::KeyEvent*>(event))!=NULL) {
        if (keyEvent->type==OS::KeyEvent::down &&
            keyEvent->qualifier==0 &&
            keyEvent->key==OS::keyEscape) {
          window->Exit();
          return true;
        }
      }

      break;
    default:
      break;
    }

    /*
      now delegate the event to the current or if no current object
      exists, to the top most object
    */
    if ((mouseEvent=dynamic_cast<OS::MouseEvent*>(event))!=NULL) {
      if (mouseEvent->type==OS::MouseEvent::down && mouseEvent->button==Lum::OS::MouseEvent::button3) {
        if (OnContextMenu()) {
          // menu mouse button up might have been lost, we kill the current mouse grab
          // to be on the save side.
          mouseGrab=NULL;
        }
      }
      else if (mouseEvent->type==OS::MouseEvent::down &&
               mouseEvent->button==Lum::OS::MouseEvent::button1) {
        Lum::Object *object;

        object=GetPosObject(posModeMenu);
        if (object!=NULL && object->GetMenu()!=NULL) {
          mouse1DownX=mouseEvent->x;
          mouse1DownY=mouseEvent->y;
          OS::display->AddTimer(MENU_TIMEOUT,0,menuAction);
        }
      }
      else if (mouseEvent->type==OS::MouseEvent::up &&
               mouseEvent->button==Lum::OS::MouseEvent::button1 &&
               mouseEvent->qualifier==Lum::OS::MouseEvent::button1) {
        OS::display->RemoveTimer(menuAction);
      }
      else if (mouseEvent->type==OS::MouseEvent::move &&
               mouseEvent->qualifier & Lum::OS::MouseEvent::button1 &&
               (std::abs(mouseEvent->x-mouse1DownX)>(int)OS::display->GetTheme()->GetMouseClickHoldSensitivity() ||
                std::abs(mouseEvent->y-mouse1DownY)>(int)OS::display->GetTheme()->GetMouseClickHoldSensitivity())) {
        OS::display->RemoveTimer(menuAction);
      }

      if (mouseGrab!=NULL) {
        mouseEvent->SetGrabed(true);
        mouseGrab->HandleMouseEvent(*mouseEvent);

        if (mouseEvent->IsGrabEnd()) {
          mouseGrab=NULL;
        }
      }
      else if (mouseEvent->IsGrabStart()) {
        mouseGrabHunting=true;
        mouseEvent->SetGrabed(false);
        mouseGrab=HandleMouseEvent(*mouseEvent);
        if (mouseGrab!=NULL) {
          if (!mouseGrabHunting) {
            mouseGrab=NULL;
          }
        }

        mouseGrabHunting=false;

        if (mouseGrab!=NULL && mouseGrab->HandlesFocus()) {
          SetFocus(dynamic_cast<Lum::Object*>(mouseGrab));
        }
      }
      else if (mouseEvent->type==OS::MouseEvent::move) {
        // We need that (currently) to get menu strip selection working
        // TODO: Try to atrivate this only if a OS mouse grab is active!?
        mouseEvent->SetGrabed(false);
        HandleMouseEvent(*mouseEvent);
      }

      EvaluateMouseActive();
    }
    else if ((keyEvent=dynamic_cast<OS::KeyEvent*>(event))!=NULL) {
      keyHandler.HandleEvent(keyEvent);
    }

    // Current object that grabed the mouse is not visible anymore => kill the grab
    if (mouseGrab!=NULL && !mouseGrab->IsVisible()) {
      mouseGrab=NULL;
    }

    return true;
  }

  void Dialog::RegisterShortcut(Lum::Object* object,
                                unsigned long qualifier,
                                const std::wstring& key,
                                Model::Action* action)
  {
    keyHandler.AddShortcut(object,qualifier,key,action);
  }

  void Dialog::RegisterCommitShortcut(Lum::Object* object,
                                      Model::Action* action)
  {
    keyHandler.AddShortcut(object,0,L"Return",action);
  }

  void Dialog::RegisterCancelShortcut(Lum::Object* object,
                                      Model::Action* action)
  {
    keyHandler.AddShortcut(object,0,L"Escape",action);
  }

  void Dialog::RegisterDefaultShortcut(Lum::Object* object,
                                       Model::Action* action)
  {
    keyHandler.AddShortcut(object,0,L"Return",action);
    keyHandler.AddShortcut(object,0,L"Escape",action);
  }

  void Dialog::Resync(::Lum::Base::Model* model, const ::Lum::Base::ResyncMsg& msg)
  {
    if (model==window->GetUnmapedAction() && window->GetUnmapedAction()->IsFinished()) {
      OS::display->RemoveTimer(menuAction);

      if (mouseActive!=NULL) {
        mouseActive->MouseInactive();
        mouseActive=NULL;
      }

      mouseGrab=NULL;
      frame->Hide();
    }
    else if (model==window->GetFocusInAction() && window->GetFocusInAction()->IsFinished()) {
      mouseGrab=NULL;
      keyHandler.Activate();
      EvaluateMouseActive();
    }
    else if (model==window->GetFocusOutAction() && window->GetFocusOutAction()->IsFinished()) {
      OS::display->RemoveTimer(menuAction);

      mouseGrab=NULL;
      mouseGrabHunting=false;
      keyHandler.Deactivate();
      EvaluateMouseActive();

      switch (window->GetType()) {
      case OS::Window::typeMenu:
      case OS::Window::typePopup:
      case OS::Window::typeSplash:
        /* Close the window, if it gets deactivated */
        if (window->IsInEventLoop()) {
        window->Close();
        }
        break;
      default:
        break;
      }
    }
    else if (model==window->GetMouseInAction() && window->GetMouseInAction()->IsFinished()) {
      EvaluateMouseActive();
    }
    else if (model==window->GetMouseOutAction() && window->GetMouseOutAction()->IsFinished()) {
      OS::display->RemoveTimer(menuAction);

      if (mouseActive!=NULL) {
        mouseActive->MouseInactive();
        mouseActive=NULL;
      }
    }
    else if (model==window->GetResizeAction() && window->GetResizeAction()->IsFinished()) {
      const OS::Window::ResizeMsg *resizeMsg=dynamic_cast<const OS::Window::ResizeMsg*>(&msg);

      if (window->GetType()==OS::Window::typeMain) {
        if (window->IsFullScreen()) {
          frame->SetBackground(OS::display->GetFill(OS::Display::backgroundFillIndex));
        }
        else {
          frame->SetBackground(OS::display->GetFill(OS::Display::dialogWindowBackgroundFillIndex));
        }
      }

      frame->Resize(resizeMsg->width,resizeMsg->height);
      EvaluateMouseActive();
    }
    else if (model==window->GetRedrawAction() && window->GetRedrawAction()->IsFinished()) {
      const OS::Window::RedrawMsg *redrawMsg=dynamic_cast<const OS::Window::RedrawMsg*>(&msg);

      frame->Move(0,0);
      frame->Draw(redrawMsg->x,redrawMsg->y,redrawMsg->width,redrawMsg->height);
    }
    else if (model==window->GetPreInitAction() && window->GetPreInitAction()->IsFinished()) {
      PreInit();

      CalcSize();

      if ((window->GetType()==OS::Window::typePopup) && (reference!=NULL)) {
        OS::Window* parent;
        int         x,y;

        parent=reference->GetWindow();
        x=parent->GetX()+reference->GetOX();
        y=parent->GetY()+reference->GetOY()+reference->GetOHeight();

        x=Base::RoundRange(x,0,OS::display->GetScreenWidth()-1-window->GetWidth());
        y=Base::RoundRange(y,0,OS::display->GetScreenHeight()-1-window->GetHeight());

        window->SetPos(x,y);
        SetPosition(OS::Window::positionManual,OS::Window::positionManual);
      }
    }
    else if (model==window->GetOpenedAction() && window->GetOpenedAction()->IsFinished()) {
      EvaluateMouseActive();
    }
    else if (model==window->GetEventAction() && window->GetEventAction()->IsFinished()) {
      const OS::Window::EventMsg *eventMsg=dynamic_cast<const OS::Window::EventMsg*>(&msg);

      HandleEvent(eventMsg->event);
    }
    else if (model==menuAction && menuAction->IsFinished()) {
      if (OnContextMenu()) {
        // menu mouse button up might have been lost, we kill the current mouse grab
        // to be on the save side.
        mouseGrab=NULL;
      }
    }
    else if (model==window->GetGrabCancelAction() && window->GetGrabCancelAction()->IsFinished()) {
      // A child popup opened while we were handling mouse button grab events.
      OS::display->RemoveTimer(menuAction);

      mouseGrab=NULL;
      mouseGrabHunting=false;
      if (mouseActive!=NULL) {
        mouseActive->MouseInactive();
        mouseActive=NULL;
      }
    }
  }

  bool Dialog::IsOpen() const
  {
    return window->IsOpen();
  }

  bool Dialog::IsShown() const
  {
    return window->IsShown();
  }

  bool Dialog::Show(bool activate)
  {
    return window->Show(activate);
  }

  bool Dialog::Hide()
  {
    return window->Hide();
  }

  void Dialog::SetParent(Dialog* parent)
  {
    if (parent!=NULL) {
      window->SetParent(parent->GetWindow());
    }
    else {
      window->SetParent(NULL);
    }
  }

  void Dialog::SetParent(OS::Window* parent)
  {
    window->SetParent(parent);
  }

  void Dialog::SetType(OS::Window::Type type)
  {
    window->SetType(type);
  }

  void Dialog::SetPosition(int x, int y)
  {
    window->SetPos(x,y);
    SetPosition(OS::Window::positionManual,OS::Window::positionManual);
  }

  void Dialog::SetPosition(OS::Window::Position horiz, OS::Window::Position vert)
  {
    window->SetPosition(horiz,vert);
  }

  void Dialog::SetTitle(const std::wstring& name)
  {
    window->SetTitle(name);
  }

  bool Dialog::Open(bool activate)
  {
    return window->Open(activate);
  }

  void Dialog::Close()
  {
    window->Close();
  }

  void Dialog::Exit()
  {
    window->Exit();
  }

  void Dialog::EventLoop()
  {
    window->EventLoop();
  }

  void Dialog::SetExitAction(::Lum::Model::Action* action)
  {
    window->SetExitAction(action);
  }

  ::Lum::Model::Action* Dialog::GetPreOpenedAction() const
  {
    return window->GetPreOpenedAction();
  }

  ::Lum::Model::Action* Dialog::GetOpenedAction() const
  {
    return window->GetOpenedAction();
  }

  ::Lum::Model::Action* Dialog::GetClosedAction() const
  {
    return window->GetClosedAction();
  }

  ::Lum::Model::Action* Dialog::GetExitAction() const
  {
    return window->GetExitAction();
  }
}

