#include <Lum/DatePicker.h>

/*
  This source is part of the Illumination library
  Copyright (C) 2005  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/String.h>

#include <time.h>

namespace Lum {

  DatePicker::DatePicker()
  : prevYear(new Model::Action()),
    prevMonth(new Model::Action()),
    nextMonth(new Model::Action()),
    nextYear(new Model::Action()),
    select(new Model::Action())
  {
    SetBackground(OS::display->GetFill(OS::Display::tableBackgroundFillIndex));

    SetCanFocus(true);
    RequestFocus();

    weekStart=1;

    AttachModel(prevYear);
    AttachModel(prevMonth);
    AttachModel(nextMonth);
    AttachModel(nextYear);

    prevYearButton=new Button();
    prevYearButton->SetFlex(true,true);
    prevYearButton->SetText(L"<<");
    prevYearButton->SetPulse(true);
    prevYearButton->SetModel(prevYear);
    prevMonthButton=new Button();
    prevMonthButton->SetFlex(true,true);
    prevMonthButton->SetText(L"<");
    prevMonthButton->SetPulse(true);
    prevMonthButton->SetModel(prevMonth);
    nextMonthButton=new Button();
    nextMonthButton->SetFlex(true,true);
    nextMonthButton->SetText(L">");
    nextMonthButton->SetPulse(true);
    nextMonthButton->SetModel(nextMonth);
    nextYearButton=new Button();
    nextYearButton->SetFlex(true,true);
    nextYearButton->SetText(L">>");
    nextYearButton->SetPulse(true);
    nextYearButton->SetModel(nextYear);

    current=new Text(L"",OS::Font::normal,Text::centered);
    current->SetFlex(true,true);
    current->SetParent(this);

    /* Initialize days texts */
    for (size_t x=0; x<31; x++) {
      texts[x]=new Text(Base::NumberToWString(x+1),
                             OS::Font::normal,Text::centered);
    }

    for (size_t x=0; x<7; x++) {
      texts[x+31]=new Text(Base::Calendar::GetWeekDayNameShort((x+1)%7),
                                OS::Font::normal,Text::centered);
    }

    for (size_t x=0; x<31+7; x++) {
      texts[x]->SetFlex(true,true);
      texts[x]->SetParent(this);
    }

    /* Initial first row with weekdays texts */
    for (size_t x=0; x<7; x++) {
      area[x][0]=x+31;
    }

    /* Initialize the rest to empty string */
    for (size_t y=1; y<7; y++) {
      for (size_t x=0; x<7; x++) {
        area[x][y]=(size_t)-1;
      }
    }

    HandleDateChange();
  }

  DatePicker::~DatePicker()
  {
    for (size_t x=0; x<31+7; x++) {
      delete texts[x];
    }

    delete current;

    delete prevYearButton;
    delete prevMonthButton;
    delete nextMonthButton;
    delete nextYearButton;

    UnattachModel(prevYear);
    UnattachModel(prevMonth);
    UnattachModel(nextMonth);
    UnattachModel(nextYear);

  }

  void DatePicker::HandleDateChange()
  {
    Base::Calendar current;
    size_t         day,week;

    current.SetDate(1,local.GetMonth(),local.GetYear());

    for (size_t y=1; y<7; y++) {
      for (size_t x=0; x<7; x++) {
        area[x][y]=(size_t)-1;
      }
    }

    week=1;
    cS=(size_t)-1;
    for (size_t x=1; x<=(size_t)current.GetDaysOfCurrentMonth(); x++) {
      current.SetDate(x,current.GetMonth(),current.GetYear());
      day=current.GetDayOfWeek();
      area[(day+7-weekStart) % 7][week]=x-1;

      if (model.Valid()) {
        if (current.GetYear()==model->Get().GetYear() &&
            current.GetMonth()==model->Get().GetMonth() &&
            current.GetDayOfMonth()==model->Get().GetDayOfMonth()) {
          cS=x-1;
        }
      }

      if (day==((weekStart-1) % 7)) {
        week++;
      }
    }

    this->current->SetText(Base::Calendar::GetMonthNameShort(current.GetMonth())+L" "+Base::NumberToWString(current.GetYear()),
                           OS::Font::normal,Text::centered);

    if (visible) {
      Redraw();
    }
  }

  bool DatePicker::VisitChildren(Visitor &visitor, bool /*onlyVisible*/)
  {
    if (!visitor.Visit(prevYearButton)) {
      return false;
    }
    if (!visitor.Visit(prevMonthButton)) {
      return false;
    }
    if (!visitor.Visit(nextMonthButton)) {
      return false;
    }
    if (!visitor.Visit(nextYearButton)) {
      return false;
    }

    return true;
  }

  bool DatePicker::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Calendar*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  Model::Action* DatePicker::GetSelectAction() const
  {
    return select.Get();
  }

  void DatePicker::CalcSize()
  {
    cw=0;
    ch=0;

    prevYearButton->SetParent(this);
    prevMonthButton->SetParent(this);
    nextMonthButton->SetParent(this);
    nextYearButton->SetParent(this);

    prevYearButton->CalcSize();
    prevMonthButton->CalcSize();
    nextMonthButton->CalcSize();
    nextYearButton->CalcSize();

    for (size_t x=0; x<31+7; x++) {
      texts[x]->CalcSize();
      cw=std::max(cw,(size_t)texts[x]->GetOMinWidth());
      ch=std::max(ch,(size_t)texts[x]->GetOMinHeight());
    }

    cw+=OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    ch+=OS::display->GetSpaceVertical(OS::Display::spaceIntraObject);

    cw=std::max(cw,prevYearButton->GetOMinWidth());
    cw=std::max(cw,prevMonthButton->GetOMinWidth());
    cw=std::max(cw,nextMonthButton->GetOMinWidth());
    cw=std::max(cw,nextYearButton->GetOMinWidth());

    ch=std::max(ch,prevYearButton->GetOMinHeight());
    ch=std::max(ch,prevMonthButton->GetOMinHeight());
    ch=std::max(ch,nextMonthButton->GetOMinHeight());
    ch=std::max(ch,nextYearButton->GetOMinHeight());

    current->CalcSize();

    cw=std::max(cw,current->GetOMinWidth()/3);
    ch=std::max(ch,current->GetOMinHeight());

    minWidth=7*cw;
    minHeight=8*ch;

    width=minWidth;
    height=minHeight;

    Control::CalcSize();
  }

  void DatePicker::Layout()
  {
    cw=width / 7;
    ch=height / 8;

    current->MoveResize(x+2*cw,y,3*cw,ch);

    prevYearButton->MoveResize(x+0*cw,y,cw,ch);
    prevMonthButton->MoveResize(x+1*cw,y,cw,ch);
    nextMonthButton->MoveResize(x+5*cw,y,cw,ch);
    nextYearButton->MoveResize(x+6*cw,y,cw,ch);

    for (size_t Y=0; Y<7; Y++) {
      for (size_t X=0; X<7; X++) {
        size_t text=area[X][Y];

        if (text!=(size_t)-1) {
          texts[text]->MoveResize(x+X*cw,y+(Y+1)*ch,cw,ch);
        }
      }
    }
  }

  void DatePicker::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    current->Draw(x,y,w,h);

    prevYearButton->Draw(x,y,w,h);
    prevMonthButton->Draw(x,y,w,h);
    nextMonthButton->Draw(x,y,w,h);
    nextYearButton->Draw(x,y,w,h);

    for (size_t Y=0; Y<7; Y++) {
      for (size_t X=0; X<7; X++) {
        size_t text=area[X][Y];

        if (text!=(size_t)-1) {
          if (text==cS) {
            draw->selected=true;
          }
          texts[text]->Draw(x,y,w,h);
          draw->selected=false;
        }
        else {
          DrawBackground(this->x+X*cw,this->y+(Y+1)*ch,cw,ch);
        }
      }
    }

    /* TODO: Fill right, bottom border */
    if (width>7*cw) {
      DrawBackground(this->x+7*cw,this->y,width-7*cw,8*ch);
    }
    if (height>8*ch) {
      DrawBackground(this->x,this->y+8*ch,7*cw,height-8*ch);
    }
    if (width>7*cw && height>8*ch) {
      DrawBackground(this->x+7*cw,this->y+8*ch,width-7*cw,height-8*ch);
    }
  }

  bool DatePicker::HandleMouseEvent(const OS::MouseEvent& event)
  {
    /* It makes no sense to get the focus if we are currently not visible */
    if (!visible || !model.Valid() || !model->IsEnabled()) {
      return false;
    }

    /*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
    */

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      for (size_t x=0; x<31+7; x++) {
        if (texts[x]->PointIsIn(event.x,event.y)) {
          if (x<=30) {
            local.SetDate(x+1,local.GetMonth(),local.GetYear());
            model->SetDate(local.GetDayOfMonth(),local.GetMonth(),local.GetYear());
            HandleDateChange();

            select->Trigger();

            return true;
          }
        }
      }

      if (current->PointIsIn(event.x,event.y)) {
        if (event.qualifier==0) {
          local.SetDate(model->Get().GetDayOfMonth(),
                        model->Get().GetMonth(),
                        model->Get().GetYear());
        }
        else {
          local.SetToCurrent();
          model->SetDate(local.GetDayOfMonth(),local.GetMonth(),local.GetYear());
        }
        HandleDateChange();
        return true;
      }
    }

    return false;
  }

  bool DatePicker::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (event.type==OS::KeyEvent::down) {

      if ((event.qualifier & OS::qualifierAlt) || (event.qualifier & OS::qualifierControl)) {
        if (event.key==OS::keyLeft || event.key==OS::keyUp) {
          Base::Calendar date=model->Get();

          date.AddYears(-1);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyRight || event.key==OS::keyDown) {
          Base::Calendar date=model->Get();

          date.AddYears(1);
          model->Set(date);

          return true;
        }
      }
      else if (event.qualifier & OS::qualifierShift) {
        if (event.key==OS::keyLeft || event.key==OS::keyUp) {
          Base::Calendar date=model->Get();

          date.AddMonths(-1);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyRight || event.key==OS::keyDown) {
          Base::Calendar date=model->Get();

          date.AddMonths(1);
          model->Set(date);

          return true;
        }
      }
      else if (event.qualifier==0) {
        if (event.key==OS::keyLeft) {
          Base::Calendar date=model->Get();

          date.AddDays(-1);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyRight) {
          Base::Calendar date=model->Get();

          date.AddDays(1);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyUp) {
          Base::Calendar date=model->Get();

          date.AddDays(-7);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyDown) {
          Base::Calendar date=model->Get();

          date.AddDays(7);
          model->Set(date);

          return true;
        }
        else if (event.key==OS::keyReturn) {
          select->Trigger();

          return true;
        }
      }
    }

    return false;
  }

  void DatePicker::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model) {
      local.SetDate(this->model->Get().GetDayOfMonth(),
                    this->model->Get().GetMonth(),
                    this->model->Get().GetYear());

      HandleDateChange();
    }
    else if (model==prevYear && prevYear->IsFinished()) {
      local.AddYears(-1);
      HandleDateChange();
    }
    else if (model==prevMonth && prevMonth->IsFinished()) {
      local.AddMonths(-1);
      HandleDateChange();
    }
    else if (model==nextMonth && nextMonth->IsFinished()) {
      local.AddMonths(1);
      HandleDateChange();
    }
    else if (model==nextYear && nextYear->IsFinished()) {
      local.AddYears(1);
      HandleDateChange();
    }

    Control::Resync(model,msg);
  }
}
