/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Button.h>

#include <Lum/Base/DateTime.h>

#include <Lum/OS/Theme.h>

#include <Lum/Dialog.h>
#include <Lum/Text.h>

namespace Lum {

  static size_t pulseSec  = 0;      // Make this a global preference value
  static size_t pulseMSec = 100000;

  Button::Button()
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetBackground(OS::display->GetFill(OS::Display::buttonBackgroundFillIndex));

    SetCanFocus(true);
    SetRedrawOnMouseActive(true);
  }

  Button::Button(Object* object)
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetBackground(OS::display->GetFill(OS::Display::buttonBackgroundFillIndex));

    SetCanFocus(true);
    SetRedrawOnMouseActive(true);

    SetLabel(object);
  }

  Button::Button(const std::wstring& text)
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetBackground(OS::display->GetFill(OS::Display::buttonBackgroundFillIndex));

    SetCanFocus(true);
    SetRedrawOnMouseActive(true);

    SetText(text);
  }

  Button::~Button()
  {
    delete object;
  }

  bool Button::HasBaseline() const
  {
    return object!=NULL && object->HasBaseline();
  }

  size_t Button::GetBaseline() const
  {
    assert(inited && object!=NULL && object->HasBaseline());

    return background->topBorder+object->GetBaseline()+OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
  }

  /**
    Set a new font to be used by the button gadget.
  */
  void Button::SetFont(OS::Font *font)
  {
    this->font=font;
  }

  /**
    Use this method if you do not want text displayed in the button but
    want to use some other Object as label
  */
  void Button::SetLabel(Object* label)
  {
    delete this->object;

    this->object=label;
    this->object->SetParent(this);
  }

  /**
    Assign an additional image. Some buttons type will automatically assign an image.
  */
  void Button::SetImage(OS::ImageRef& image)
  {
    this->image=image;
  }

  /**
    Call this method if you want the given text to be displayed in
    the button.

    This creates an instance of Text using the given text
    and sets it as button label.

    use '_' to mark the next character as key shortcut.
  */
  void Button::SetText(const std::wstring& string)
  {
    Text   *text;
    size_t pos;

    text=new Text();
    text->SetParent(this);
    text->SetFlex(true,true);
    text->SetAlignment(Text::centered);

    pos=string.find('_');
    if (pos!=std::wstring::npos && pos+1<string.length()) {
      shortCut=string[pos+1];
      text->AddText(string.substr(0,pos));
      text->AddText(string.substr(pos+1,1),OS::Font::underlined);
      text->AddText(string.substr(pos+2));
    }
    else {
      text->AddText(string);
    }

    delete object;

    object=text;
  }

  /**
    We can define special types of buttons.

    \see Type.
  */
  void Button::SetType(Type type)
  {
    this->type=type;
  }

  /**
    Is pulsemode is true, the button sends permanent events on
    mouse button down and none on the final button up.

    This is usefull for buttons in a scroller or similar.
  */
  void Button::SetPulse(bool pulse)
  {
    this->pulse=pulse;
  }

  bool Button::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Action*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void Button::CalcSize()
  {
    /* Let the frame calculate its size */
    switch (type) {
      case typeNormal:
      SetBackground(OS::display->GetFill(OS::Display::buttonBackgroundFillIndex));
      RequestFocus();
      break;
    case typeDefault:
      SetBackground(OS::display->GetFill(OS::Display::defaultButtonBackgroundFillIndex));
      image=OS::display->GetImage(OS::Display::defaultImageIndex);
      RequestFocus();
      break;
    case typeCommit:
      SetBackground(OS::display->GetFill(OS::Display::positiveButtonBackgroundFillIndex));
      image=OS::display->GetImage(OS::Display::positiveImageIndex);
      RequestFocus();
      break;
    case typeCancel:
      SetBackground(OS::display->GetFill(OS::Display::negativeButtonBackgroundFillIndex));
      image=OS::display->GetImage(OS::Display::negativeImageIndex);
      RequestFocus();
      break;
    case typeImage:
      SetBackground(OS::display->GetFill(OS::Display::buttonBackgroundFillIndex));
      RequestFocus();
      break;
    case typeToolBar:
      SetBackground(OS::display->GetFill(OS::Display::toolbarButtonBackgroundFillIndex));
      SetCanFocus(false);
      break;
    case typeScroll:
      SetBackground(OS::display->GetFill(OS::Display::scrollButtonBackgroundFillIndex));
      SetCanFocus(false);
      break;
    }

    width=0;
    height=0;
    minWidth=0;
    minHeight=0;

    if (object!=NULL) {
      /*
        Now we let the image calculate its bounds and simply add its size
        to the size of the button.
      */
      object->CalcSize();
      width=object->GetOWidth();
      height=object->GetOHeight();
      minWidth=object->GetOMinWidth();
      minHeight=object->GetOMinHeight();
    }

    if (image.Valid() && OS::display->GetTheme()->ShowButtonImages()) {

      minWidth+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      width+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      minHeight=std::max(minHeight,image->GetHeight());
      height=std::max(height,image->GetHeight());
    }

    switch (type) {
    case typeImage:
      minWidth+=Base::GetSize(Base::Size::softUnitP,25)*2;
      width+=Base::GetSize(Base::Size::softUnitP,25)*2;
      minHeight+=Base::GetSize(Base::Size::softUnitP,25)*2;
      height+=Base::GetSize(Base::Size::softUnitP,25)*2;
      break;
    case typeScroll:
      // No space around object
      break;
    default:
      break;
    }

    maxWidth=30000;
    maxHeight=30000;

    /* We *must* call CalcSize of our superclass! */
    Control::CalcSize();
  }

  bool Button::HandleMouseEvent(const OS::MouseEvent& event)
  {
    /* It makes no sense to get the focus if we are currently not visible */
    if (!visible || !model.Valid() || !model->IsEnabled()) {
      return false;
    }

    /*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
    */

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      if (!model->IsStarted()) {
        /* We change our state to pressed and redisplay ourself */
        model->Start();

        if (pulse) {
          model->Trigger();
          OS::display->AddTimer(pulseSec,pulseMSec,model);
        }

        /*
          Since we want the focus for waiting for buttonup we return
          a pointer to ourself.
        */
        return true;
      }
    }
    else if (event.IsGrabEnd()) {
      if (model->IsStarted()) {
        /*
          If the users released the left mousebutton over our bounds we really
          got selected.
        */
        if (PointIsIn(event)) {
          if (pulse) {
            model->Cancel();
          }
          else {
            model->Finish();
          }
        }
        else {
          model->Cancel();
        }
      }

      // Clean up and remove possibly remaining timer event.
      if (pulse) {
        OS::display->RemoveTimer(model);
      }
    }
    else if (event.type==OS::MouseEvent::move && event.IsGrabed()) {
      if (PointIsIn(event)) {
        if (!model->IsStarted()) {
          model->Start();
        }
      }
      else {
        if (model->IsStarted()) {
          model->Cancel();
        }
      }

      return true;
    }

    return false;
  }

  bool Button::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (event.type==OS::KeyEvent::down) {
      if (event.key==OS::keySpace && model->IsInactive()) {
        model->Start();

        return true;
      }
    }
    else if (event.type==OS::KeyEvent::up) {
      if (event.key==OS::keySpace) {
        if (event.qualifier==0 && model->IsStarted()) {
          model->Finish();
        }
        else if (event.qualifier!=0 && model->IsStarted()) {
          model->Cancel();
        }
      }
    }

    return false;
  }

  /*
     We tell the image to resize themself to
     our current bounds. Our bounds could have changed
     because Resize may have been called by some layout-objects
     between Button.CalcSize and Button.Draw.
   */
  void Button::Layout()
  {
    int    x,y;
    size_t width,height;

    x=this->x;
    y=this->y;
    width=this->width;
    height=this->height;

    switch (type) {
    case typeImage:
      x+=Base::GetSize(Base::Size::softUnitP,25);
      y+=Base::GetSize(Base::Size::softUnitP,25);
      width-=2*Base::GetSize(Base::Size::softUnitP,25);
      height-=2*Base::GetSize(Base::Size::softUnitP,25);
      break;
    case typeScroll:
      break;
    default:
      break;
    }

    if (image.Valid() && OS::display->GetTheme()->ShowButtonImages()) {
      imageX=x;
      imageY=y+(height-image->GetHeight())/2;
      x+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      width-=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    }

    if (object!=NULL) {
      object->Resize(width,height);
      object->Move(x+(width-object->GetOWidth()) / 2,
                 y+(height-object->GetOHeight()) / 2);
    }

    Control::Layout();
  }

  void Button::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h); /* We must call Draw of our superclass */

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    OS::DrawInfo *draw=GetDrawInfo();

    /*
      Set the correct draw mode before calling the baseclass,
      since the baseclass draw the object frame.
    */
    if (model.Valid() && model->IsEnabled()) {
      if (IsMouseActive() && !model->IsStarted()) {
        draw->activated=true;
      }
      if (model->IsStarted()) {
        draw->selected=true;
      }
    }
    else {
      draw->disabled=true;
    }

    draw->focused=HasFocus();

    DrawBackground(x,y,w,h);

    // Draw object
    if (object!=NULL) {
      object->Draw(x,y,w,h);
    }

    // Draw optional image
    if (image.Valid() && OS::display->GetTheme()->ShowButtonImages()) {
      image->Draw(draw,imageX,imageY,image->GetWidth(),image->GetHeight());
    }

    draw->activated=false;
    draw->selected=false;
    draw->disabled=false;
    draw->focused=false;

    if (!scAssigned) {
      Dialog* window;

      window=dynamic_cast<Dialog*>(GetWindow()->GetMaster());
      if (window!=NULL) {
        if (shortCut!='\0') {
          window->RegisterShortcut(this,0,std::wstring(1,shortCut),model);
          window->RegisterShortcut(this,OS::qualifierAlt,std::wstring(1,shortCut),model);
        }

        switch (type) {
        case typeCommit:
          window->RegisterCommitShortcut(this,model);
          break;
        case typeCancel:
          window->RegisterCancelShortcut(this,model);
          break;
        case typeDefault:
          window->RegisterDefaultShortcut(this,model);
          break;
        default:
          break;
        }
      }
      scAssigned=true;
    }
 }

  void Button::Hide()
  {
    if (visible) {
      if (object!=NULL) {
        /* Hide the image */
        object->Hide();
      }
      /* hide the frame */
      Control::Hide();
    }
  }

  void Button::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model && visible) {
      if (this->model->IsInactive()) {
        Redraw();
      }
      else if (this->model->IsStarted()) {
        Redraw();
      }
      else if (this->model->IsFinished()) {
        if (pulse) {
          OS::display->RemoveTimer(this->model);
          OS::display->AddTimer(pulseSec,pulseMSec,this->model);
        }
      }
    }

    Control::Resync(model,msg);
  }
}
