#ifndef LUM_OS_THEME_H
#define LUM_OS_THEME_H

/*
  This source is part of the Illumination library
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <vector>

#include <Lum/Private/ImportExport.h>

#include <Lum/OS/Color.h>
#include <Lum/OS/Display.h>
#include <Lum/OS/DrawInfo.h>
#include <Lum/OS/Fill.h>
#include <Lum/OS/Frame.h>
#include <Lum/OS/Image.h>

namespace Lum {
  namespace OS {

    /**
      A abstract base class for holding all information for theming a user interface.
    */
    class LUMAPI Theme
    {
    public:
      enum ObjectType {
        // State widgets
        objRadio,
        objCheck,
        objLED
      };

      enum ScrollKnobMode {
        scrollKnobModeSingle,
        scrollKnobModeDouble
      };

      enum ToolbarPosition {
        toolbarTop,
        toolbarBottom
      };

    public:
      Theme(Display *display);
      virtual ~Theme();

      /**
        @name Fonthandling
        Information about the to be used fonts for this theme.
      */
      //@{
      virtual bool HasFontSettings() const = 0;

      virtual std::wstring GetProportionalFontName() const = 0;
      virtual size_t GetProportionalFontSize() const = 0;

      virtual std::wstring GetFixedFontName() const = 0;
      virtual size_t GetFixedFontSize() const = 0;
      //@}

      /**
        @name Theme specific GUI primitives
        Colors, fills, frames and images for themes.
      */
      //@{
      virtual Color GetColor(Display::ColorIndex color) const = 0;
      virtual Fill* GetFill(Display::FillIndex fill) const = 0;
      virtual Frame* GetFrame(Display::FrameIndex frame) const = 0;
      virtual Image* GetImage(Display::ImageIndex image) const = 0;

      virtual void PushUniqueFill(DrawInfo* draw, size_t index, size_t maximum) = 0;
      virtual void PopUniqueFill(DrawInfo* draw, size_t index, size_t maximum) = 0;
      //@}

      /**
        Return the size of an object with out content.
      */
      virtual void GetObjectSize(ObjectType object,
                                 size_t& width,
                                 size_t& height) const = 0;

      /**
        Returns true, if the objects needs the parent to fill
        the content area first.
       */
      virtual bool NeedsObjectParentDraw(ObjectType object) const = 0;

      /**
        Draws the object itself, you have to draw childs
        onto the area afterwards.
       */
      virtual void DrawObject(DrawInfo* draw,
                              ObjectType object,
                              int x, int y, size_t width, size_t height,
                              int cX, int cY, size_t cWidth, size_t cHeight) = 0;

      /**
        @name Spaces
        Theme specific spacing.
      */
      //@{
      virtual size_t GetSpaceHorizontal(Display::Space space) const = 0;
      virtual size_t GetSpaceVertical(Display::Space space) const = 0;

      virtual size_t GetMinimalButtonWidth() const = 0;
      virtual size_t GetFirstTabOffset() const = 0;
      virtual size_t GetLastTabOffset() const = 0;
      virtual size_t GetEntryHorizTextOffset() const = 0;
      virtual size_t GetEntryVertTextOffset() const = 0;
      /**
        How many pixels must the mouse move before this is recognized as drag start?
      */
      virtual size_t GetDragStartSensitivity() const = 0;
      /**
        How many pixels may the move move after click (and while holding) to get this
        recognizes as "click and hold" (right mouse button replacement)?
      */
      virtual size_t GetMouseClickHoldSensitivity() const = 0;
      //@}

      /**
        @name Variants and behaviours
        Choosing between display variants and different behaviours.
      */
      //@{
      virtual ScrollKnobMode GetScrollKnobMode() const = 0;
      virtual ToolbarPosition GetToolbarPosition() const = 0;
      virtual bool ShowButtonImages() const = 0;
      //@}
    };
  }
}

#endif
