/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Scroller.h>

#include <Lum/OS/Theme.h>

#include <Lum/Button.h>
#include <Lum/Image.h>

namespace Lum {
  Scroller::Prefs *prefs=new Scroller::Prefs(L"Scroller");

  Scroller::Prefs::Prefs(const std::wstring& name)
    : Object::Prefs(name)
  {
    // no code
  }

  void Scroller::Prefs::Initialize()
  {
    Object::Prefs::Initialize();

    up=OS::display->GetImage(OS::Display::arrowScrollUpImageIndex);
    down=OS::display->GetImage(OS::Display::arrowScrollDownImageIndex);
    left=OS::display->GetImage(OS::Display::arrowScrollLeftImageIndex);
    right=OS::display->GetImage(OS::Display::arrowScrollRightImageIndex);
  }

  Scroller::Scroller()
  : panel(NULL),knob(new Knob),vert(true)
  {
    SetPrefs(::Lum::prefs);

    SetCanFocus(true);

    knob->SetFlex(true,true);
  }

  Scroller::~Scroller()
  {
    delete panel;
  }

  bool Scroller::VisitChildren(Visitor &visitor, bool /*onlyVisible*/)
  {
    if (visible && GetModel()!=NULL && GetModel()->IsEnabled() && panel!=NULL) {
      return visitor.Visit(panel);
    }

    return false;
  }

  void Scroller::Set(bool vert)
  {
    this->vert=vert;

    knob->Set(vert);
  }

  void Scroller::SetOffset(int offset)
  {
    knob->SetOffset(offset);
  }

  bool Scroller::SetModel(Base::Model* model)
  {
    bool res;

    res=knob->SetModel(model);
    Control::SetModel(knob->GetModel());

    return res;
  }

  Button* Scroller::CreateButton(bool vert, bool dec)
  {
    Button *button;
    Image  *image;

    image=new Image();
    image->SetFlex(true,true);
    if (dec) {
      if (vert) {
        image->SetImage(dynamic_cast<Prefs*>(prefs)->up);
      }
      else {
        image->SetImage(dynamic_cast<Prefs*>(prefs)->left);
      }
    }
    else {
      if (vert) {
        image->SetImage(dynamic_cast<Prefs*>(prefs)->down);
      }
      else {
        image->SetImage(dynamic_cast<Prefs*>(prefs)->right);
      }
    }

    button=new Button(image);
    button->SetType(Button::typeScroll);
    if (vert) {
      button->SetFlex(true,false);
    }
    else {
      button->SetFlex(false,true);
    }
    button->SetPulse(true);

    button->SetModel(dec ? knob->GetDecAction() : knob->GetIncAction());

    return button;
  }

  void Scroller::CalcSize()
  {
    if (!CanFocus()) {
      knob->SetCanFocus(false);
    }

    if (vert) {
      SetFrame(OS::display->GetFrame(OS::Display::vscrollFrameIndex));

      panel=new VPanel();
      panel->SetFlex(false,true);
    }
    else {
      SetFrame(OS::display->GetFrame(OS::Display::hscrollFrameIndex));

      panel=new HPanel();
      panel->SetFlex(true,false);
    }
    panel->SetParent(this);

    panel->Add(CreateButton(vert,true));
    if (OS::display->GetTheme()->GetScrollKnobMode()==OS::Theme::scrollKnobModeDouble) {
      panel->Add(CreateButton(vert,false));
    }

    panel->Add(knob);

    if (OS::display->GetTheme()->GetScrollKnobMode()==OS::Theme::scrollKnobModeDouble) {
      panel->Add(CreateButton(vert,true));
    }
    panel->Add(CreateButton(vert,false));

    panel->CalcSize();

    minWidth=panel->GetOMinWidth();
    minHeight=panel->GetOMinHeight();

    width=panel->GetOWidth();
    height=panel->GetOHeight();

    Control::CalcSize();
  }

  void Scroller::Layout()
  {
    panel->MoveResize(x,y,width,height);

    Control::Layout();
  }

  void Scroller::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    panel->Draw(x,y,w,h);
  }

  void Scroller::Hide()
  {
    if (visible) {
      panel->Hide();
      Control::Hide();
    }
  }
}
