/*
  This source is part of the Illumination library
  Copyright (C) 2006  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/Curses/Theme.h>

namespace Lum {
  namespace OS {
    namespace Curses {

      TextImage::TextImage(Display::ImageIndex image)
        : image(image)
      {
        switch (image) {
        case Display::hknobImageIndex:
        case Display::vknobImageIndex:
        case Display::arrowLeftImageIndex:
        case Display::arrowRightImageIndex:
        case Display::arrowUpImageIndex:
        case Display::arrowDownImageIndex:
        case Display::arrowScrollLeftImageIndex:
        case Display::arrowScrollRightImageIndex:
        case Display::arrowScrollUpImageIndex:
        case Display::arrowScrollDownImageIndex:
        case Display::comboImageIndex:
        case Display::treeExpanderImageIndex:
        case Display::hscaleKnobImageIndex:
        case Display::vscaleKnobImageIndex:
        case Display::closeImageIndex:
        case Display::menuSubImageIndex:
          width=1;
          height=1;
          break;
        case Display::comboEditButtonImageIndex:
          width=3;
          height=1;
          break;
        case Display::comboDividerImageIndex:
        case Display::leftSliderImageIndex:
        case Display::rightSliderImageIndex:
        case Display::topSliderImageIndex:
        case Display::bottomSliderImageIndex:
        case Display::menuDividerImageIndex:
        case Display::positiveImageIndex:
        case Display::negativeImageIndex:
        case Display::defaultImageIndex:
        case Display::helpImageIndex:
          width=0;
          height=0;
          break;
        }
      }

      size_t TextImage::GetWidth() const
      {
        return width;
      }

      size_t TextImage::GetHeight() const
      {
        return height;
      }

      bool TextImage::GetAlpha() const
      {
        return true;
      }

      unsigned long TextImage::GetDrawCap() const
      {
        return 0;//drawsDisabled|drawsFocused|drawsActivated;
      }

      void TextImage::Draw(OS::DrawInfo *draw, int x, int y, size_t w, size_t h)
      {
        draw->PushFont(OS::display->GetFont());
        draw->PushForeground(Display::textColor);

        for (int i=x; i<x+w; ++i) {
          for (int j=y; j<y+h; ++j) {

            switch (image) {
            case Display::menuCheckImageIndex:
              draw->PushForeground(Display::textColor);
              if (draw->selected) {
                draw->DrawString(i,j,L"X");
              }
              else {
                draw->DrawString(i,j,L"o");
              }
              draw->PopForeground();
              break;
            case Display::menuRadioImageIndex:
              draw->PushForeground(Display::textColor);
              if (draw->selected) {
                draw->DrawString(i,j,L"*");
              }
              else {
                draw->DrawString(i,j,L"o");
              }
              draw->PopForeground();
              break;
            case Display::hknobImageIndex:
            case Display::vknobImageIndex:
              draw->PushForeground(dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
              draw->FillRectangle(i,j,1,1);
              draw->PopForeground();
              draw->PushForeground(Display::textColor);
              draw->DrawString(i,j,L"#");
              draw->PopForeground();
              break;
            case Display::arrowLeftImageIndex:
            case Display::arrowScrollLeftImageIndex:
              draw->PushForeground(Display::textColor);
              draw->DrawString(i,j,L"<");
              draw->PopForeground();
              break;
            case Display::arrowRightImageIndex:
            case Display::arrowScrollRightImageIndex:
            case Display::menuSubImageIndex:
              draw->PushForeground(Display::textColor);
              draw->DrawString(i,j,L">");
              draw->PopForeground();
              break;
            case Display::arrowUpImageIndex:
            case Display::arrowScrollUpImageIndex:
              draw->PushForeground(Display::textColor);
              draw->DrawString(i,j,L"^");
              draw->PopForeground();
              break;
            case Display::arrowDownImageIndex:
            case Display::arrowScrollDownImageIndex:
            case Display::comboImageIndex:
              draw->PushForeground(Display::textColor);
              draw->DrawString(i,j,L"v");
              draw->PopForeground();
              break;
            case Display::comboEditButtonImageIndex:
              if (i==x) {
                draw->DrawFillString(i,j,L"[",OS::display->GetColor(Display::backgroundColor));
              }
              else if (i==x+w-1) {
                draw->DrawFillString(i,j,L"]",OS::display->GetColor(Display::backgroundColor));
              }
              else {
                if (draw->selected) {
                  draw->PushForeground(Display::fillTextColor);
                }
                draw->DrawFillString(i,j,L"v",OS::display->GetColor(Display::backgroundColor));
                if (draw->selected) {
                  draw->PopForeground();
                }
              }
              break;
            case Display::treeExpanderImageIndex:
              if (draw->selected) {
                draw->DrawString(i,j,L"-");
              }
              else {
                draw->DrawString(i,j,L"+");
              }
              break;
            case Display::hscaleKnobImageIndex:
            case Display::vscaleKnobImageIndex:
              if (draw->selected) {
                draw->PushForeground(Display::fillTextColor);
              }
              draw->DrawString(i,j,L"*");
              if (draw->selected) {
                draw->PopForeground();
              }
              break;
            case Display::leftSliderImageIndex:
            case Display::bottomSliderImageIndex:
              draw->DrawString(i,j,L"#");
              break;
            case Display::rightSliderImageIndex:
            case Display::topSliderImageIndex:
              draw->DrawString(i,j,L" ");
              break;
            case Display::closeImageIndex:
              draw->DrawString(i,j,L"X");
              break;
            case Display::comboDividerImageIndex:
            case Display::menuDividerImageIndex:
            case Display::positiveImageIndex:
            case Display::negativeImageIndex:
            case Display::defaultImageIndex:
            case Display::helpImageIndex:
              break;
            }
          }
        }

        draw->PopForeground();
        draw->PopFont();
      }

      TextFill::TextFill(Display::FillIndex fill)
      : fill(fill)
      {
        // no code
      }

      void TextFill::Draw(OS::DrawInfo* draw,
                         int /*xOff*/, int /*yOff*/, size_t /*width*/, size_t /*height*/,
                         int x, int y, size_t w, size_t h)
      {
        switch (fill) {
        case Display::backgroundFillIndex:
          draw->PushForeground(Display::backgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::buttonBackgroundFillIndex:
        case Display::defaultButtonBackgroundFillIndex:
        case Display::toolbarButtonBackgroundFillIndex:
        case Display::scrollButtonBackgroundFillIndex:
          draw->PushForeground(dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::positiveButtonBackgroundFillIndex:
          draw->PushForeground(dynamic_cast<Theme*>(OS::display->GetTheme())->positiveButtonBackgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::negativeButtonBackgroundFillIndex:
          draw->PushForeground(dynamic_cast<Theme*>(OS::display->GetTheme())->negativeButtonBackgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::tableBackgroundFillIndex:
        case Display::tableBackground2FillIndex:
          if (draw->selected) {
            draw->PushForeground(Display::fillColor);
          }
          else {
            draw->PushForeground(dynamic_cast<Theme*>(OS::display->GetTheme())->tableBackgroundColor);
          }
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::helpBackgroundFillIndex:
        case Display::menuStripBackgroundFillIndex:
          draw->PushForeground(Display::backgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::hprogressFillIndex:
        case Display::vprogressFillIndex:
          draw->PushForeground(Display::fillColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::menuPulldownBackgroundFillIndex:
          if (draw->selected) {
            draw->PushForeground(Display::fillColor);
          }
          else {
            draw->PushForeground(Display::backgroundColor);
          }
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::menuEntryBackgroundFillIndex:
          if (draw->selected) {
            draw->PushForeground(Display::fillColor);
          }
          else {
            draw->PushForeground(Display::backgroundColor);
          }
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::graphBackgroundFillIndex:
          draw->PushForeground(Display::backgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        case Display::toolbarBackgroundFillIndex:
          draw->PushForeground(Display::backgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          break;
        default:
          assert(false);
          break;
        }
      }

      TextFrame::TextFrame(Display::FrameIndex frame)
      : frame(frame)
      {
        switch (frame) {
        case Display::buttonFrameIndex:
        case Display::positiveButtonFrameIndex:
        case Display::negativeButtonFrameIndex:
        case Display::defaultButtonFrameIndex:
        case Display::toolbarButtonFrameIndex:
        case Display::editFrameIndex:
          leftBorder=1;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
          break;
        case Display::hprogressFrameIndex:
        case Display::vprogressFrameIndex:
        case Display::listboxFrameIndex:
        case Display::tabFrameIndex:
        case Display::menuWindowFrameIndex:
          leftBorder=1;
          rightBorder=leftBorder;
          topBorder=1;
          bottomBorder=topBorder;
          break;
        case Display::tabRiderFrameIndex:
          leftBorder=1;
          rightBorder=leftBorder;
          topBorder=1;
          bottomBorder=0;
          break;
        case Display::hknobBoxFrameIndex:
        case Display::vknobBoxFrameIndex:
        case Display::scrolledFrameIndex:
          leftBorder=0;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
          break;
        case Display::columnFrameIndex:
        case Display::menuPulldownFrameIndex:
          leftBorder=1;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
          break;
        }

        minWidth=leftBorder+rightBorder;
        minHeight=topBorder+bottomBorder;
      }

      void TextFrame::Draw(OS::DrawInfo* draw, int x, int y, size_t w, size_t h)
      {
        if (minWidth==0 && minHeight==0) {
          return;
        }

        Color backgroundColor=OS::display->GetColor(Display::backgroundColor);

        switch (frame) {
        case Display::buttonFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L"[",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L"]",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;
        case Display::positiveButtonFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L"<",dynamic_cast<Theme*>(OS::display->GetTheme())->positiveButtonBackgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L">",dynamic_cast<Theme*>(OS::display->GetTheme())->positiveButtonBackgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;
        case Display::negativeButtonFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L"[",dynamic_cast<Theme*>(OS::display->GetTheme())->negativeButtonBackgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L"]",dynamic_cast<Theme*>(OS::display->GetTheme())->negativeButtonBackgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;

        case Display::defaultButtonFrameIndex:
        case Display::toolbarButtonFrameIndex:
        case Display::editFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L"[",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L"]",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;
        case Display::hprogressFrameIndex:
        case Display::vprogressFrameIndex:
        case Display::listboxFrameIndex:
        case Display::tabFrameIndex:
        case Display::menuWindowFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          draw->DrawFillString(x,y,L"\x250c",backgroundColor);
          draw->DrawFillString(x+w-1,y,L"\x2510",backgroundColor);
          draw->DrawFillString(x,y+h-1,L"\x2514",backgroundColor);
          draw->DrawFillString(x+w-1,y+h-1,L"\x2518",backgroundColor);

          // top
          for (int i=x+1; i<=x+(int)w-2; i++) {
            draw->DrawFillString(i,y,L"\x2500",backgroundColor);
          }

          // left
          for (int i=y+1; i<=y+(int)h-2; i++) {
            draw->DrawFillString(x,i,L"\x2502",backgroundColor);
          }

          // right
          for (int i=y+1; i<=y+(int)h-2; i++) {
            draw->DrawFillString(x+w-1,i,L"\x2502",backgroundColor);
          }

          // bottom
          for (int i=x+1; i<=x+(int)w-2; i++) {
            draw->DrawFillString(i,y+h-1,L"\x2500",backgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;
        case Display::tabRiderFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          draw->DrawFillString(x,y,L"\x250c",backgroundColor);
          draw->DrawFillString(x+w-1,y,L"\x2510",backgroundColor);

          // top
          for (int i=x+1; i<=x+(int)w-2; i++) {
            draw->DrawFillString(i,y,L"\x2500",backgroundColor);
          }

          // left
          for (int i=y+1; i<=y+(int)h-1; i++) {
            draw->DrawFillString(x,i,L"\x2502",backgroundColor);
          }

          // right
          for (int i=y+1; i<=y+(int)h-1; i++) {
            draw->DrawFillString(x+w-1,i,L"\x2502",backgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;

        case Display::hknobBoxFrameIndex:
        case Display::vknobBoxFrameIndex:
          // no frame
          break;
        case Display::columnFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L" ",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L" ",dynamic_cast<Theme*>(OS::display->GetTheme())->buttonBackgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;
        case Display::menuPulldownFrameIndex:
          draw->PushFont(OS::display->GetFont());
          draw->PushForeground(Display::blackColor);

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x,i,L" ",backgroundColor);
          }

          for (size_t i=y; i<y+(size_t)h; ++i) {
            draw->DrawFillString(x+w-1,i,L" ",backgroundColor);
          }

          draw->PopForeground();
          draw->PopFont();
          break;

        default:
          assert(false);
          break;
        }
      }

      Theme::Theme(OS::Display *display)
      : OS::Base::Theme(display)
      {
        const char* colorNames[Display::colorCount];

        if (display->GetColorMode()==Display::colorModeMonochrome) {
          colorNames[Display::backgroundColor]="white";
          colorNames[Display::tableTextColor]="black";
          colorNames[Display::textColor]="black";
          colorNames[Display::textSelectColor]="black";
          colorNames[Display::textDisabledColor]="black";
          colorNames[Display::fillColor]="black";
          colorNames[Display::fillTextColor]="white";
          colorNames[Display::graphScaleColor]="black";
          colorNames[Display::blackColor]="black";
          colorNames[Display::whiteColor]="white";
          colorNames[Display::tabTextColor]="black";
          colorNames[Display::tabTextSelectColor]="black";
        }
        else {
          colorNames[Display::backgroundColor]="white";
          colorNames[Display::tableTextColor]="white";
          colorNames[Display::textColor]="black";
          colorNames[Display::textSelectColor]="black";
          colorNames[Display::textDisabledColor]="black";
          colorNames[Display::fillColor]="cyan";
          colorNames[Display::fillTextColor]="black";
          colorNames[Display::graphScaleColor]="black";
          colorNames[Display::blackColor]="black";
          colorNames[Display::whiteColor]="white";
          colorNames[Display::tabTextColor]="black";
          colorNames[Display::tabTextSelectColor]="black";
        }

        for (size_t i=0; i<Display::colorCount; i++) {
          display->AllocateNamedColor(colorNames[i],color[i]);
        }

        if (display->GetColorMode()==Display::colorModeMonochrome) {
          display->AllocateNamedColor("white",tableBackgroundColor);
          display->AllocateNamedColor("white",buttonBackgroundColor);
          display->AllocateNamedColor("white",positiveButtonBackgroundColor);
          display->AllocateNamedColor("white",negativeButtonBackgroundColor);
        }
        else {
          display->AllocateNamedColor("blue",tableBackgroundColor);
          display->AllocateNamedColor("green",buttonBackgroundColor);
          display->AllocateNamedColor("green",positiveButtonBackgroundColor);
          display->AllocateNamedColor("red",negativeButtonBackgroundColor);
        }

        fill[Display::backgroundFillIndex]=new TextFill(Display::backgroundFillIndex);
        fill[Display::buttonBackgroundFillIndex]=new TextFill(Display::buttonBackgroundFillIndex);
        fill[Display::positiveButtonBackgroundFillIndex]=new TextFill(Display::positiveButtonBackgroundFillIndex);
        fill[Display::negativeButtonBackgroundFillIndex]=new TextFill(Display::negativeButtonBackgroundFillIndex);
        fill[Display::defaultButtonBackgroundFillIndex]=new TextFill(Display::defaultButtonBackgroundFillIndex);
        fill[Display::toolbarButtonBackgroundFillIndex]=new TextFill(Display::toolbarButtonBackgroundFillIndex);
        fill[Display::scrollButtonBackgroundFillIndex]=new TextFill(Display::scrollButtonBackgroundFillIndex);
        fill[Display::comboBackgroundFillIndex]=fill[Display::buttonBackgroundFillIndex];
        fill[Display::tableBackgroundFillIndex]=new TextFill(Display::tableBackgroundFillIndex);
        fill[Display::tableBackground2FillIndex]=new TextFill(Display::tableBackground2FillIndex);
        fill[Display::entryBackgroundFillIndex]=fill[Display::tableBackgroundFillIndex];
        fill[Display::helpBackgroundFillIndex]=new TextFill(Display::helpBackgroundFillIndex);
        fill[Display::hprogressBackgroundFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::hprogressFillIndex]=new TextFill(Display::hprogressFillIndex);
        fill[Display::vprogressBackgroundFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::vprogressFillIndex]=fill[Display::hprogressFillIndex];
        fill[Display::hknobBoxFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::vknobBoxFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::tabBackgroundFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::tabRiderBackgroundFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::columnBackgroundFillIndex]=fill[Display::buttonBackgroundFillIndex];
        fill[Display::columnEndBackgroundFillIndex]=fill[Display::backgroundFillIndex];
        fill[Display::menuStripBackgroundFillIndex]=new TextFill(Display::menuStripBackgroundFillIndex);
        fill[Display::menuPulldownBackgroundFillIndex]=new TextFill(Display::menuPulldownBackgroundFillIndex);
        fill[Display::menuEntryBackgroundFillIndex]=new TextFill(Display::menuEntryBackgroundFillIndex);
        fill[Display::graphBackgroundFillIndex]=new TextFill(Display::graphBackgroundFillIndex);
        fill[Display::toolbarBackgroundFillIndex]=new TextFill(Display::toolbarBackgroundFillIndex);

        frame[Display::buttonFrameIndex]=new TextFrame(Display::buttonFrameIndex);
        frame[Display::positiveButtonFrameIndex]=new TextFrame(Display::positiveButtonFrameIndex);
        frame[Display::negativeButtonFrameIndex]=new TextFrame(Display::negativeButtonFrameIndex);
        frame[Display::defaultButtonFrameIndex]=new TextFrame(Display::defaultButtonFrameIndex);
        frame[Display::toolbarButtonFrameIndex]=new TextFrame(Display::toolbarButtonFrameIndex);
        frame[Display::scrollButtonFrameIndex]=emptyFrame;
        frame[Display::editFrameIndex]=new TextFrame(Display::editFrameIndex);
        frame[Display::hprogressFrameIndex]=new TextFrame(Display::hprogressFrameIndex);;
        frame[Display::vprogressFrameIndex]=new TextFrame(Display::vprogressFrameIndex);;
        frame[Display::hknobBoxFrameIndex]=emptyFrame;
        frame[Display::vknobBoxFrameIndex]=emptyFrame;
        frame[Display::hscrollFrameIndex]=emptyFrame;
        frame[Display::vscrollFrameIndex]=emptyFrame;
        frame[Display::tabFrameIndex]=new TextFrame(Display::tabFrameIndex);
        frame[Display::tabRiderFrameIndex]=new TextFrame(Display::tabRiderFrameIndex);
        frame[Display::listboxFrameIndex]=new TextFrame(Display::listboxFrameIndex);
        frame[Display::scrolledFrameIndex]=emptyFrame;
        frame[Display::columnFrameIndex]=new TextFrame(Display::columnFrameIndex);
        frame[Display::columnLeftFrameIndex]=frame[Display::columnFrameIndex];
        frame[Display::columnMiddleFrameIndex]=frame[Display::columnFrameIndex];
        frame[Display::columnRightFrameIndex]=frame[Display::columnFrameIndex];
        frame[Display::columnEndFrameIndex]=emptyFrame;
        frame[Display::comboFrameIndex]=frame[Display::buttonFrameIndex];
        frame[Display::editComboFrameIndex]=emptyFrame;
        frame[Display::mainWindowFrameIndex]=emptyFrame;
        frame[Display::menuWindowFrameIndex]=new TextFrame(Display::menuWindowFrameIndex);
        frame[Display::popupWindowFrameIndex]=emptyFrame;
        frame[Display::tooltipWindowFrameIndex]=emptyFrame;
        frame[Display::hscaleFrameIndex]=emptyFrame;
        frame[Display::vscaleFrameIndex]=emptyFrame;
        frame[Display::focusFrameIndex]=emptyFrame;
        frame[Display::statusbarFrameIndex]=emptyFrame;
        frame[Display::statuscellFrameIndex]=emptyFrame;
        frame[Display::plateFrameIndex]=emptyFrame;
        frame[Display::valuebarFrameIndex]=emptyFrame;
        frame[Display::groupFrameIndex]=emptyFrame; // TODO
        frame[Display::menuStripFrameIndex]=emptyFrame;
        frame[Display::menuPulldownFrameIndex]=new TextFrame(Display::menuPulldownFrameIndex);
        frame[Display::labelFrameIndex]=emptyFrame;

        image[Display::hknobImageIndex]=new TextImage(Display::hknobImageIndex);
        image[Display::vknobImageIndex]=new TextImage(Display::vknobImageIndex);
        image[Display::arrowLeftImageIndex]=new TextImage(Display::arrowLeftImageIndex);
        image[Display::arrowRightImageIndex]=new TextImage(Display::arrowRightImageIndex);
        image[Display::arrowUpImageIndex]=new TextImage(Display::arrowUpImageIndex);
        image[Display::arrowDownImageIndex]=new TextImage(Display::arrowDownImageIndex);
        image[Display::arrowScrollLeftImageIndex]=new TextImage(Display::arrowScrollLeftImageIndex);
        image[Display::arrowScrollRightImageIndex]=new TextImage(Display::arrowScrollRightImageIndex);
        image[Display::arrowScrollUpImageIndex]=new TextImage(Display::arrowScrollUpImageIndex);
        image[Display::arrowScrollDownImageIndex]=new TextImage(Display::arrowScrollDownImageIndex);
        image[Display::comboImageIndex]=new TextImage(Display::comboImageIndex);
        image[Display::comboEditButtonImageIndex]=new TextImage(Display::comboEditButtonImageIndex);
        image[Display::comboDividerImageIndex]=new TextImage(Display::comboDividerImageIndex);
        image[Display::treeExpanderImageIndex]=new TextImage(Display::treeExpanderImageIndex);
        image[Display::hscaleKnobImageIndex]=new TextImage(Display::hscaleKnobImageIndex);
        image[Display::vscaleKnobImageIndex]=new TextImage(Display::vscaleKnobImageIndex);
        image[Display::leftSliderImageIndex]=new TextImage(Display::leftSliderImageIndex);
        image[Display::rightSliderImageIndex]=new TextImage(Display::rightSliderImageIndex);
        image[Display::topSliderImageIndex]=new TextImage(Display::topSliderImageIndex);
        image[Display::bottomSliderImageIndex]=new TextImage(Display::bottomSliderImageIndex);
        image[Display::menuDividerImageIndex]=new TextImage(Display::menuDividerImageIndex);
        image[Display::menuSubImageIndex]=new TextImage(Display::menuSubImageIndex);
        image[Display::menuCheckImageIndex]=new TextImage(Display::menuCheckImageIndex);
        image[Display::menuRadioImageIndex]=new TextImage(Display::menuRadioImageIndex);
        image[Display::closeImageIndex]=new TextImage(Display::closeImageIndex);
      }

      Theme::~Theme()
      {
        display->FreeColor(tableBackgroundColor);
        display->FreeColor(buttonBackgroundColor);
        display->FreeColor(positiveButtonBackgroundColor);
        display->FreeColor(negativeButtonBackgroundColor);

        for (size_t i=0; i<Display::colorCount; i++) {
          display->FreeColor(color[i]);
        }
      }

      bool Theme::HasFontSettings() const
      {
        return true;
      }

      std::wstring Theme::GetProportionalFontName() const
      {
        return L"";
      }

      size_t Theme::GetProportionalFontSize() const
      {
        return 1;
      }

      std::wstring Theme::GetFixedFontName() const
      {
        return L"";
      }

      size_t Theme::GetFixedFontSize() const
      {
        return 1;
      }

      void Theme::GetObjectSize(ObjectType object,
                                size_t& width,
                                size_t& height) const
      {
        switch (object) {
        case objRadio:
        case objCheck:
        case objLED:
          width=1;
          height=1;
          break;
        }
      }

      bool Theme::NeedsObjectParentDraw(ObjectType object) const
      {
        switch (object) {
        case objRadio:
        case objCheck:
        case objLED:
          return false;
          break;
        }
      }

      void Theme::DrawObject(DrawInfo* draw,
                             ObjectType object,
                             int x, int y, size_t width, size_t height,
                             int cX, int cY, size_t cWidth, size_t cHeight)
      {
        draw->PushFont(OS::display->GetFont());
        draw->PushForeground(Display::textColor);

        for (int i=x; i<x+width; ++i) {
          for (int j=y; j<y+height; ++j) {
            switch (object) {
            case objCheck:
              draw->PushForeground(Display::textColor);
              if (draw->selected) {
                draw->DrawString(i,j,L"X");
              }
              else {
                draw->DrawString(i,j,L"o");
              }
              draw->PopForeground();
              break;
            case objRadio:
              draw->PushForeground(Display::textColor);
              if (draw->selected) {
                draw->DrawString(i,j,L"*");
              }
              else {
                draw->DrawString(i,j,L"o");
              }
              draw->PopForeground();
              break;
            case objLED:
              if (draw->selected) {
                draw->DrawString(i,j,L"*");
              }
              else {
                draw->DrawString(i,j,L"o");
              }
              break;
            }
          }
        }

        draw->PopForeground();
        draw->PopFont();
      }

      size_t Theme::GetSpaceHorizontal(Display::Space space) const
      {
        switch (space) {
        case Display::spaceWindowBorder:
          return 0;
        case Display::spaceInterGroup:
          return 1;
        case Display::spaceGroupIndent:
          return 1;
        case Display::spaceInterObject:
          return 0;
        case Display::spaceObjectBorder:
          return 0;
        case Display::spaceIntraObject:
          return 1;
        case Display::spaceLabelObject:
          return 1;
        case Display::spaceObjectDetail:
          return 1;
        }

        return 1;
      }

      size_t Theme::GetSpaceVertical(Display::Space space) const
      {
        switch (space) {
        case Display::spaceWindowBorder:
          return 0;
        case Display::spaceInterGroup:
          return 0;
        case Display::spaceGroupIndent:
          return 0;
        case Display::spaceInterObject:
          return 0;
        case Display::spaceObjectBorder:
          return 0;
        case Display::spaceIntraObject:
          return 0;
        case Display::spaceLabelObject:
          return 0;
        case Display::spaceObjectDetail:
          return 0;
        }

        return 1;
      }

      size_t Theme::GetDragStartSensitivity() const
      {
        return 1;
      }

      size_t Theme::GetMouseClickHoldSensitivity() const
      {
        return 1;
      }

      size_t Theme::GetMinimalButtonWidth() const
      {
        return 0;
      }

      size_t Theme::GetFirstTabOffset() const
      {
        return 0;
      }

      size_t Theme::GetLastTabOffset() const
      {
        return 0;
      }

      size_t Theme::GetEntryHorizTextOffset() const
      {
        return 0;
      }

      size_t Theme::GetEntryVertTextOffset() const
      {
        return 0;
      }

      OS::Theme::ScrollKnobMode Theme::GetScrollKnobMode() const
      {
        return OS::Theme::scrollKnobModeDouble;
      }

      OS::Theme::ToolbarPosition Theme::GetToolbarPosition() const
      {
        return OS::Theme::toolbarTop;
      }

      bool Theme::ShowButtonImages() const
      {
        return false;
      }

    }
  }
}
