/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Combo.h>

#include <Lum/PopupGroup.h>
#include <Lum/Text.h>

namespace Lum {
  static Combo::Prefs *prefs= new Combo::Prefs();

  Combo::Prefs::Prefs()
  : Control::Prefs(L"Combo")
  {
    // no code
  }

  void Combo::Prefs::Initialize()
  {
    Control::Prefs::Initialize();

    background=OS::display->GetFill(OS::Display::comboBackgroundFillIndex);

    readOnlyFrame=OS::display->GetFrame(OS::Display::comboFrameIndex);
    readOnlyImage=OS::display->GetImage(OS::Display::comboImageIndex);
    divider=OS::display->GetImage(OS::Display::comboDividerImageIndex);

    editableFrame=OS::display->GetFrame(OS::Display::editComboFrameIndex);
    buttonImage=OS::display->GetImage(OS::Display::comboEditButtonImageIndex);

    imageRight=true;
  }

  Combo::Table::Table()
  : success(false),
    commit(new Model::Action())
  {
    AttachModel(GetMouseSelectionAction());
    AttachModel(commit);
  }

  Combo::Table::~Table()
  {
    UnattachModel(commit);
    UnattachModel(GetMouseSelectionAction());
  }

  void Combo::Table::CalcSize()
  {
    dynamic_cast<Dialog*>(GetWindow()->GetMaster())->RegisterCommitShortcut(this,commit);

    TableView::CalcSize();
  }

  void Combo::Table::Resync(Base::Model* model,
                            const Base::ResyncMsg& msg)
  {
    if (model==GetMouseSelectionAction() && GetMouseSelectionAction()->IsFinished()) {
      success=true;
      GetWindow()->Exit();
    }
    else if (model==commit && commit->IsFinished()) {
      success=true;
      GetWindow()->Exit();
    }
    else {
      TableView::Resync(model,msg);
    }
  }

  Combo::Combo(Object* value, bool editable)
  : editable(editable),
    table(NULL),
    popup(NULL),
    prevAction(new Lum::Model::Action()),
    nextAction(new Lum::Model::Action()),
    tableModel(NULL),
    selection(new Model::SingleLineSelection()),
    value(value)
  {
    SetCanFocus(true);
    RequestFocus();
    SetCanDrawMouseActive(true);

    value->SetParent(this);

    if (dynamic_cast<Control*>(value)==NULL) {
      value->SetBackground(NULL);
    }

    SetPrefs(::Lum::prefs);

    AttachModel(prevAction);
    AttachModel(nextAction);
  }

  Combo::~Combo()
  {
    UnattachModel(nextAction);
    UnattachModel(prevAction);

    delete value;
  }

  bool Combo::HasBaseline() const
  {
    return value!=NULL && value->HasBaseline();
  }

  size_t Combo::GetBaseline() const
  {
    if (value!=NULL && value->HasBaseline()) {
      // We also assume that is vertically flexible
      return frame->topBorder+value->GetBaseline()+OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
    }
    else {
      return 0;
    }
  }

  Object* Combo::GetFocusObject()
  {
    if (value->CanFocus()) {
      return value;
    }
    else {
      return this;
    }
  }

  bool Combo::VisitChildren(Visitor &visitor, bool /*onlyVisible*/)
  {
    if (value->CanFocus()) {
      if (!visitor.Visit(value)) {
        return false;
      }
    }

    return true;
  }

  /**
    Call this method to assign the table model that will be used to display
    all possible values in the pop window.
  */
  void Combo::SetTableModel(Model::Table* model)
  {
    Base::ResyncMsg msg;

    tableModel=model;
    if (model!=NULL) {
      Resync(model,msg);
    }
  }

  void Combo::CalcSize()
  {
    /* Delegate Focus visualisation to value class, if possible */
    if (value->RequestedFocus()) {
      UnrequestFocus();
    }

    if (editable) {
      SetFrame(dynamic_cast<Prefs*>(prefs)->editableFrame);
      if (frame->minHeight>0 || frame->minWidth>0) {
        value->SetFrame(new OS::EmptyFrame());
      }
    }
    else {
      SetFrame(dynamic_cast<Prefs*>(prefs)->readOnlyFrame);
    }

    value->SetFlex(true,true);
    value->CalcSize();

    if (editable) {
      minWidth=dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth();

      width=minWidth;

      minWidth+=value->GetOMinWidth();
      minWidth+=value->GetOWidth();

      minHeight=std::max(value->GetOMinHeight(),
                         dynamic_cast<Prefs*>(prefs)->buttonImage->GetHeight());
      height=std::max(value->GetOHeight(),
                      dynamic_cast<Prefs*>(prefs)->buttonImage->GetHeight());
    }
    else {
      minWidth=OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
               dynamic_cast<Prefs*>(prefs)->divider->GetWidth()+
               OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
               dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth();

      width=minWidth;

      minWidth+=value->GetOMinWidth();
      minWidth+=value->GetOWidth();

      minHeight=std::max(value->GetOMinHeight(),dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight());
      height=std::max(value->GetOHeight(),dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight());
    }

    Control::CalcSize();
  }

  /**
    Overload this method if you want some inital initialisation of the table
    object used for the popup window. The baseclass is empty.

    Normally this will be done by a concrete implementation of Combo.
  */
  void Combo::InitTable(::Lum::Table* /*table*/)
  {
    // no code
  }

  Combo::Popup::Popup(Combo *combo)
  : combo(combo)
  {
    AttachModel(GetOpenedAction());
  }

  Combo::Popup::~Popup()
  {
    UnattachModel(GetOpenedAction());
  }

  void Combo::Popup::PreInit()
  {
    PopupGroup *container;

    container=new PopupGroup();
    container->SetFlex(true,true);

    combo->table=new Lum::Table();

    combo->table->SetTableView(new Table());
    combo->table->SetFlex(true,true);
    combo->table->SetMinHeight(Base::Size::stdCharHeight,1);
    combo->table->SetMaxWidth(Base::Size::screenVRel,40);
    combo->table->SetMaxHeight(Base::Size::screenVRel,40);
    combo->table->SetModel(combo->tableModel);
    combo->table->SetSelection(combo->selection);
    if (combo->tableModel.Valid() && combo->tableModel->GetColumns()==1) {
      Model::HeaderRef headerModel=new Model::HeaderImpl();
      headerModel->AddColumn(L"",Lum::Base::Size::modePixel,0);

      combo->table->SetHeaderModel(headerModel);
      combo->table->GetTableView()->SetAutoFitColumns(true);
    }
    combo->table->GetTableView()->SetAutoHSize(true);
    combo->table->GetTableView()->SetAutoVSize(true);

    combo->InitTable(combo->table);

    container->SetMain(combo->table);

    SetTop(container);

    Dialog::PreInit();
  }

  void Combo::Popup::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (combo->tableModel.Valid() && combo->selection->GetLine()>0) {
        combo->table->GetTableView()->MakeVisible(1,combo->selection->GetLine());
      }
    }

    Dialog::Resync(model,msg);
  }

  void Combo::OpenPopup()
  {
    popup=new Popup(this);
    popup->SetType(OS::Window::typePopup);
    popup->SetParent(GetWindow());
    popup->SetReference(this);

    if (popup->Open()) {
      popup->EventLoop();
      popup->Close();

      if (dynamic_cast<Table*>(table->GetTableView())->success && tableModel.Valid() && selection->HasSelection()) {
        CopySelection(selection->GetLine());
      }
    }

    delete popup;
    popup=NULL;
  }

  void Combo::SetTableRow(size_t row)
  {
    selection->SelectCell(1,row);
    CopySelection(row);
  }

  bool Combo::HandleMouseEvent(const OS::MouseEvent& event)
  {
    if (!visible || (GetModel()==NULL) || !GetModel()->IsEnabled()) {
      return false;
    }

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      OpenPopup();
      return true;
    }

    return false;
  }

  bool Combo::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (dynamic_cast<Control*>(value)!=NULL && dynamic_cast<Control*>(value)->HandleKeyEvent(event)) {
      return true;
    }

    if (event.type==OS::KeyEvent::down) {
      switch (event.key) {
      case OS::keyUp:
        prevAction->Trigger();
        return true;
        break;

      case OS::keyDown:
        nextAction->Trigger();
        return true;
        break;

      default:
        break;
      }
    }

    return false;
  }

  void Combo::Layout()
  {
    if (editable) {
      value->Resize(width-dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth(),height);

      if (dynamic_cast<Prefs*>(prefs)->imageRight) {
        value->Move(x,y+(height-value->GetOHeight()) / 2);
      }
      else {
        value->Move(x+dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth(),
                    y+(height-value->GetOHeight()) / 2);
      }
    }
    else {
      value->Resize(width-
                    dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth()-
                    2*OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)-
                    dynamic_cast<Prefs*>(prefs)->divider->GetWidth(),
                    height-2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));

      if (dynamic_cast<Prefs*>(prefs)->imageRight) {
        value->Move(x,
                    y+(height-value->GetOHeight()) / 2);
      }
      else {
        value->Move(x+dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth()+
                    2*OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
                     dynamic_cast<Prefs*>(prefs)->divider->GetWidth(),
                    y+(height-value->GetOHeight()) / 2);
      }
    }

    Control::Layout();
  }

  void Combo::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    if (IsMouseActive()) {
      draw->activated=true;
    }

    draw->focused=HasFocus();

    Control::Draw(x,y,w,h);

    draw->activated=false;
    draw->focused=false;
    draw->disabled=!(GetModel()!=NULL) || GetModel()->IsNull() || !GetModel()->IsEnabled();

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    if (editable) {
      value->Draw(x,y,w,h);

      if (IsMouseActive()) {
        draw->activated=true;
      }

      draw->focused=HasFocus();

      if (dynamic_cast<Prefs*>(prefs)->imageRight) {
        int    bx,by;
        size_t bw,bh;

        bx=this->x+value->GetOWidth();
        by=y;
        bw=dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth();
        bh=height;

        dynamic_cast<Prefs*>(prefs)->buttonImage->Draw(draw,
                                                       bx,by,
                                                       dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth(),
                                                       height);
      }
      else {
        dynamic_cast<Prefs*>(prefs)->buttonImage->Draw(draw,
                                                       this->x,this->y,
                                                       dynamic_cast<Prefs*>(prefs)->buttonImage->GetWidth(),
                                                       height);
      }
    }
    else {
      if (IsMouseActive()) {
        draw->activated=true;
      }

      draw->focused=HasFocus();

      draw->PushClipBegin(x,y,w,h);
      draw->SubClipRegion(value->GetOX(),value->GetOY(),value->GetOWidth(),value->GetOHeight());
      draw->PushClipEnd();
      DrawBackground(x,y,w,h);
      draw->PopClip();

      value->Draw(x,y,w,h);

      if (dynamic_cast<Prefs*>(prefs)->imageRight) {
        dynamic_cast<Prefs*>(prefs)->divider->Draw(draw,
                                                   this->x+width-
                                                   dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth()-
                                                   OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)-
                                                   dynamic_cast<Prefs*>(prefs)->divider->GetWidth(),
                                                   this->y,
                                                   dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth(),
                                                   height);

        dynamic_cast<Prefs*>(prefs)->readOnlyImage->Draw(draw,
                                                         this->x+width-
                                                         dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth(),
                                                         this->y+(height-dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight()) / 2,
                                                         dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth(),
                                                         dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight());
      }
      else {
        dynamic_cast<Prefs*>(prefs)->readOnlyImage->Draw(draw,
                                                         x,
                                                         y+(height-dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight()) / 2,
                                                         dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth(),
                                                         dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetHeight());

        dynamic_cast<Prefs*>(prefs)->divider->Draw(draw,
                                                   x+dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth()+
                                                   OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject),
                                                   this->y,
                                                   dynamic_cast<Prefs*>(prefs)->readOnlyImage->GetWidth(),
                                                   height);

      }
    }

    draw->activated=false;
    draw->focused=false;
    draw->disabled=false;
  }

  void Combo::DrawFocus()
  {
    if (editable && value!=NULL) {
      value->SetShowFocus(true);
    }

    Control::DrawFocus();
  }

  void Combo::HideFocus()
  {
    if (editable && value!=NULL) {
      value->SetShowFocus(false);
    }
    Control::HideFocus();
  }

  void Combo::Hide()
  {
    if (visible) {
      value->Hide();
      Control::Hide();
    }
  }

  void Combo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (tableModel.Valid()) {
      if (model==prevAction && prevAction->IsFinished()) {
        if (selection->HasSelection()) {
          if (selection->GetLine()>1) {
            SetTableRow(selection->GetLine()-1);
          }
        }
        else if (tableModel->GetRows()>0) {
          SetTableRow(1);
        }
      }
      else if (model==nextAction && nextAction->IsFinished()) {
        if (selection->HasSelection()) {
          if (selection->GetLine()<tableModel->GetRows()) {
            SetTableRow(selection->GetLine()+1);
          }
        }
        else if (tableModel->GetRows()>0) {
          SetTableRow(1);
        }
      }
    }

    Control::Resync(model,msg);
  }

  TextCombo::TextCombo()
  : Combo(new Text(),false),
    model(NULL)
  {
    // no code
  }

  TextCombo::~TextCombo()
  {
    if (model.Valid()) {
      UnattachModel(model);
    }
  }

  bool TextCombo::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::String*>(model);

    Combo::SetModel(this->model);

    return this->model.Valid();
  }

  /**
    This method will be called if a new value was selected from the popup
    window. The baseclass will try its best to assign a sensefull value to
    the model assigned to Combo. If the model is numemric it will
    assign the index of the current selcted (starting with 0), if its of type
    text, if will assign the text of the current selected (if the table model
    has more than one coloum if will iterate from left to right until some
    valid text will be returned).

    If you want some other (or more) behaviour, overwrite this method
    and call baseclass if necessary.
  */
  void TextCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      while (res.empty() && x<=tableModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      dynamic_cast<Text*>(value)->SetText(res);

      if (model.Valid()) {
        model->Set(res);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }

  void TextCombo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model) {
      dynamic_cast<Text*>(value)->SetText(this->model->Get());
    }

    Combo::Resync(model,msg);
  }


  IndexCombo::IndexCombo()
  : Combo(new Text(),false),
    model(NULL)
  {
    // no code
  }

  IndexCombo::~IndexCombo()
  {
    if (model.Valid()) {
      UnattachModel(model);
    }
  }

  bool IndexCombo::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::SizeT*>(model);

    Combo::SetModel(this->model);

    return this->model.Valid();
  }

  void IndexCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      while (res.empty() && x<=tableModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      dynamic_cast<Text*>(value)->SetText(res);

      if (model.Valid()) {
        model->Set(row);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }

  void IndexCombo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if ((model==tableModel || model==this->model) &&
        this->model.Valid() && tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      if (!this->model->IsNull()) {
        selection->SelectCell(1,this->model->Get());

        while (res.empty() && x<=tableModel->GetColumns()) {
          res=tableModel->GetString(x,this->model->Get());
          x++;
        }
      }

      dynamic_cast<Text*>(value)->SetText(res);
    }
    else if (model==this->model) {
      Redraw();
    }

    Combo::Resync(model,msg);
  }


  TextEditCombo::TextEditCombo()
  : Combo(new String,true),
    model(NULL)
  {
    string=dynamic_cast<String*>(value);
    string->SetFlex(true,true);
    string->SetCursorUpAction(prevAction);
    string->SetCursorDownAction(nextAction);
  }

  TextEditCombo::~TextEditCombo()
  {
    if (model.Valid()) {
      UnattachModel(model);
    }
  }

  bool TextEditCombo::SetModel(Base::Model* model)
  {
    if (string->SetModel(model)) {
      this->model=dynamic_cast<Model::String*>(model);

      Combo::SetModel(this->model);

      return this->model.Valid();
    }
    else {
      return false;
    }
  }

  String* TextEditCombo::GetString() const
  {
    return string;
  }

  void TextEditCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x;
      std::wstring res;

      x=1;
      while (res.empty() && x<=tableModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      if (model.Valid()) {
        model->Set(res);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }
}

