/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/X11/Display.h>

#include <cassert>

#if defined(HAVE_UNISTD_H)
  #include <unistd.h>
#endif

#include <stdint.h>

#if defined(HAVE_PWD_H)
  #include <pwd.h>
#endif

// for select TODO: Write a configure test for it
#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>

#include <iostream>
#include <limits>

#include <X11/X.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h> // TODO: Add configure check

#if defined(HAVE_LIB_GTK)
  #include <gtk/gtk.h>
  #include <gdk/gdkx.h>
  #include <gdk-pixbuf-xlib/gdk-pixbuf-xlib.h>
#endif

#include <Lum/Base/String.h>

#include <Lum/Images/Image.h>

#include <Lum/OS/Base/Theme.h>

#include <Lum/OS/X11/DrawInfo.h>
#include <Lum/OS/X11/Font.h>
#include <Lum/OS/X11/Image.h>
#include <Lum/OS/X11/Tray.h>
#include <Lum/OS/X11/Window.h>

#if defined(HAVE_LIB_GTK)
  #include <Lum/OS/X11/GtkTheme.h>
#endif

#if defined(HAVE_LIB_HILDON)
  #include <Lum/OS/X11/HildonKeyboardHandler.h>
#endif

#if defined(CAIRO_HAS_XLIB_SURFACE)
  #include <Lum/OS/Cairo/Display.h>
#endif

#include <Lum/Prefs/Base.h>

namespace Lum {
  namespace OS {
    namespace X11 {

      static const char* atomNames[Display::atomCount] = {
        "ATOM",
        "WM_DELETE_WINDOW",
        "WM_STATE",
        "XdndAware",
        "XdndEnter",
        "XdndLeave",
        "XdndPosition",
        "XdndStatus",
        "XdndFinished",
        "XdndDrop",
        "XdndActionCopy",
        "XdndActionMove",
        "XdndActionLink",
        "XdndActionAsk",
        "XdndActionPrivate",
        "XdndActionList",
        "XdndSelection",
        "XdndTypeList",
        "_ILLUMINATION_DROP_DATA",
        "_ILLUMINATION_SELECTION_DATA",
        "COMPOUND_TEXT",
        "CLIPBOARD",
        "_ILLUMINATION_CLIPBOARD_DATA",
        "_NET_WM_NAME",
        "_NET_WM_WINDOW_TYPE",
        "_NET_WM_WINDOW_TYPE_DESKTOP",
        "_NET_WM_WINDOW_TYPE_TOOLBAR",
        "_NET_WM_WINDOW_TYPE_MENU",
        "_NET_WM_WINDOW_TYPE_UTILITY",
        "_NET_WM_WINDOW_TYPE_SPLASH",
        "_NET_WM_WINDOW_TYPE_DIALOG",
        "_NET_WM_WINDOW_TYPE_NORMAL",
        "_NET_WM_PING",
        "_NET_WM_PID",
	"_NET_WM_STATE",
	"_NET_WM_STATE_FULLSCREEN",
        "_NET_SYSTEM_TRAY_OPCODE",
        "_NET_SYSTEM_TRAY_MESSAGE_DATA",
        "_NET_WORKAREA",
        "UTF8_STRING",
        "_XEMBED",
        "_XEMBED_INFO",
        "CARDINAL",
        "WINDOW",
        "TARGETS"
      };

      static const int sleepCursor=150;
      static const int popCursor=94;
      static const int dndCursor=50;
#if defined(HAVE_LIB_DBUS)
      static const char* dbusDomain="org.teulings.illumination";
#endif

      const char* unicodeEncoding="iso10646-1";

      /*
      class SleepDesc : public ::Lum::OS::Display::SleepDesc
      {
        protected:
        Sleep next;
        ::Lum::Base::Object::MsgObject object;
      };

      class ChannelDesc : public ::Lum::OS::Display::ChannelDesc
      {
        protected:
        Channel next;
        SelectionKey key;
        ::Lum::Base::Object::MsgObject object;
        void SendNotify();
      };

      class PenDashDesc
      {
        protected:
        PenDash next;
        char *list;
        int mode;
      };

      class PatternDesc
      {
        protected:
        Pattern next;
        Pixmap pixMap;
        int mode;
      };
*/

      class Prefs : public ::Lum::OS::Display::Prefs
      {
      private:
        Display* display;

      public:
        Prefs(Display* display);

        void Initialize();

        void ReadConfig(::Lum::Config::Node *top);
      };

      /**
        Initializes an instance.
      */
      Prefs::Prefs(Display* display)
      : display(display)
      {
        // no code
      }

      void Prefs::Initialize()
      {
#ifdef HAVE_LIB_HILDON
        theme=L"Gtk";
        propFont="Bitstream Vera Sans";
        propFontSize=16;
#else
        propFont="helvetica";
        propFontSize=12;
#endif
        fixedFont="fixed";
        fixedFontSize=13;

        //p.contextTimer=2500;
      }

      void Prefs::ReadConfig(Config::Node *top)
      {
        Config::Node *node,*sub;

        node=top->GetChild(L"theme");
        if (node!=NULL) {
          theme=node->GetValue();
        }

        sub=top->GetChild(L"Fonts");
        if (sub!=NULL) {
          sub->GetAttribute(L"proportional",propFont);
          sub->GetAttribute(L"proportionalSize",propFontSize);
          sub->GetAttribute(L"fixed",fixedFont);
          sub->GetAttribute(L"fixedSize",fixedFontSize);
        }
      }

      double Display::GetDPI() const
      {
        return dpi;
      }

      void Display::SetAtomProperty(Window* window, Atom property, Atom value)
      {
        XChangeProperty(display,window->window,property,atoms[atomAtom],32,
                        PropModeReplace,(unsigned char*)&value,1);
      }

      void Display::SetCardinalProperty(Window* window, Atom property, int value)
      {
        XChangeProperty(display,window->window,property,atoms[cardinalAtom],32,
                        PropModeReplace,(unsigned char*)&value,1);
      }

      void Display::SetEmbedInfoProperty(::Window window)
      {
        uint32_t data[2];

        data[0]=0;
        data[1]=1;

        ::XChangeProperty(display,
                          window,
                          atoms[xEmbedInfoAtom],
                          atoms[xEmbedInfoAtom],32,
                          PropModeReplace,
                        (unsigned char*)data,2);
      }

      void Display::SendNetSystemTrayRequestDock(::Window window)
      {
        ::Window selWin;

        selWin=::XGetSelectionOwner(display,XInternAtom(display,"_NET_SYSTEM_TRAY_S0",False));

        if (selWin!=0) {
          ::XEvent trayEvt;

          memset(&trayEvt,0,sizeof(trayEvt));
          trayEvt.xclient.type=ClientMessage;
          trayEvt.xclient.window=selWin;
          trayEvt.xclient.message_type=atoms[netSystemTrayOpcodeAtom];
          trayEvt.xclient.format=32;
          trayEvt.xclient.data.l[0]=CurrentTime;
          trayEvt.xclient.data.l[1]=0;
          trayEvt.xclient.data.l[2]=window;
          trayEvt.xclient.data.l[3]=0;
          trayEvt.xclient.data.l[4]=0;
          if (XSendEvent(display,selWin,False,NoEventMask,&trayEvt)==0) {
            std::cerr << "Error while sending SystemtrayRequestDock!" << std::endl;
          }
        }
      }

#if 0

/**
  Returns the window currently under the mouse pointer. It also returns the
  screen relative and window relative mouse coordinates.
*/
              Window Display::GetWindowOnScreen(int rX, int rY, int cX, int cY)
              {
                int help;
                int rW;
                int cW;
                Window dragWin;
                unsigned long mask;
                Atom retType;
                int retFormat;
                int itemsReturn;
                int bytesLeft;
                ::C::string data;

                dragWin=0;
                data=NULL;
                itemsReturn=0;
                if (X11.XQueryPointer(d.display,X11.XRootWindow(d.display,0),rW,cW,rX,rY,cX,cY,mask)==X11.True) {
                  while ((rW!=0) && (dragWin==0) && (cW!=0) && (rW!=cW)) {
                    if ((X11.XGetWindowProperty(d.display,cW,d.atoms[wmStateAtom],0,MAX(LONGINT),X11.False,X11.AnyPropertyType,retType,retFormat,itemsReturn,bytesLeft,data)==X11.Success) && (retType!=X11.None)) {
                      dragWin=cW;
                    }
                    if (rW!=0) {
                      help=cW;
                      if (X11.XTranslateCoordinates(d.display,rW,cW,cX,cY,cX,cY,cW)==X11.True) {
                        rW=help;
                      }
                      else {
                        Err.String("Cannot convert coordinates!");
                        Err.Ln;
                        rW=0;
                      }
                    }
                  }
                }
                return dragWin;
              }

         #endif

      /**
        Get the Illumination window matching the given X11 window.
      */
      Window* Display::GetWindow(::Window window)
      {
        std::list<Window*>::const_iterator iter;

        iter=winList.begin();
        while (iter!=winList.end()) {
          if ((*iter)->window==window) {
            return *iter;
          }

          ++iter;
        }
        return NULL;
      }

      /**
        Return the Illumination tray belonging to the given X11 window.
      */
      Tray* Display::GetTray(::Window window)
      {
        std::list<Tray*>::const_iterator iter;

        iter=trayList.begin();
        while (iter!=trayList.end()) {
          if ((*iter)->window==window) {
            return *iter;
          }

          ++iter;
        }
        return NULL;
      }

      FontRef Display::GetFontInternal(FontType type, size_t size) const
      {
        FontRef font;

        font=driver->CreateFont();
        font->features=Font::attrName|Font::attrHeight|Font::attrSpacing|Font::attrCharSet;

        switch (type) {
        case fontTypeProportional:
          font->name=propFont->name;
          font->spacing=Font::spacingProportional;
          break;
        case fontTypeFixed:
          font->name=fixedFont->name;
          font->spacing=Font::spacingMonospace;
          break;
        }

        font->pixelHeight=size;
        font->charSet=unicodeEncoding;

        font=font->Load();

        if (font.Valid()) {
          return font.Get();
        }
        else {
          return NULL;
        }
      }

      /**
        Call this method if you want the Display to stop generating
        QuickHelp calls to windows. This is necessesarry, if you are
        opening a QuickHelp and don't want to have a second after the
        second timeout.
      */
      void Display::StopContextHelp()
      {
        if (contextHelp) {
          contextHelp=false;
          /*if (d.contextTimer.active) {
            d.RemoveTimer(d.contextTimer);
          }*/
        }
      }

      /**
        Restart the generation of QuickHelp calls to windows stoped
        with Display.StopContextHelp.
      */
      void Display::RestartContextHelp()
      {
        /*if (d.contextTimer.active) {
          d.RemoveTimer(d.contextTimer);
        }
        d.AddTimer(d.contextTimer);*/
        contextHelp=true;
      }

      /**
        Restart the generation of QuickHelp calls to windows stoped
        with Display.StopContextHelp.
      */
      void Display::StartContextHelp()
      {
        if (!contextHelp) {
          //d.AddTimer(d.contextTimer);
          contextHelp=true;
        }
      }

      /**
        Adds the window to the internal list of windows.
      */
      void Display::AddWindow(Window* w)
      {
        winList.push_back(w);
      }

      /**
        Removes the window from the internal list of windows.
      */
      void Display::RemoveWindow(Window* w)
      {
        winList.remove(w);
      }

      /**
        Adds the tray to the internal list of trays.
      */
      void Display::AddTray(Tray* tray)
      {
        trayList.push_back(tray);
      }

      /**
        Removes the tray from the internal list of trays.
      */
      void Display::RemoveTray(Tray* w)
      {
        trayList.remove(w);
      }

#if 0
              ::Lum::OS::FontList Display::GetFontList()
              {
                int count;
                int x;
                int y;
                int size;
                ::C::charPtr2d names;
                ::Lum::OS::FontName name;
                ::Lum::OS::FontName field;
                ::Lum::OS::FontName field2;
                ConvResults res;
                ::Lum::OS::FontList list;
                ::Lum::OS::FontFoundry foundry;
                ::Lum::OS::FontFamily family;
                ::Lum::OS::FontEncoding encoding;
                ::Lum::OS::FontSize fSize;
                FcConfigPtr config;
                FcPatternPtr pattern;
                FcPatternPtr pattern2;
                FcFontSetPtr set;
                FcFontSetPtr set2;
                FcObjectSetPtr oSet;
                FcObjectSetPtr oSet2;
                XftChar8Ptr buffer;
                XftChar8Ptr buffer2;
                int z;
                ::Lum::OS::FontName fName;
                ::Lum::OS::FontName fName;
                XftFontSetPtr set;
                XftFontSetPtr set2;
                int z;
                int t;
                ::C::string tmp;


                list=new FontList;
                list.families=NULL;
                names=X11.XListFonts(d.display,"-*-*-medium-r-*--*-*-*-*-*-*-*-*",MAX(INTEGER),count);
                if ((names!=NULL) && (count>0)) {
                  for (/*TODO*/x=0TOcount-1) {
                    y=0;
                    while (names[x][y]!=0x0) {
                      name[y]=names[x][y];
                      y++;
                    }
                    name[y]=0x0;
                    GetFontElement(name,2,field);
                    y=0;
                    while (field[y]!=0x0) {
                      if ((y==0) || (field[y-1]==" ")) {
                        field[y]=CAP(field[y]);
                      }
                      y++;
                    }
                    family=list.GetOrCreateFamily(field);
                    GetFontElement(name,1,field);
                    if (field=="") {
                      field="X11";
                    }
                    foundry=family.GetOrCreateFoundry(field);
                    GetFontElement(name,13,field);
                    GetFontElement(name,14,field2);
                    str.Append("-",field);
                    str.Append(field2,field);
                    encoding=foundry.GetOrCreateEncoding(field);
                    GetFontElement(name,7,field);
                    if (field!="*") {
                      co.StrToInt(field,size,res);
                      if (size!=0) {
                        if (~encoding.anySize) {
                          fSize=encoding.GetOrCreateSize(size);
                        }
                      }
                      else {
                        encoding.anySize=true;
                        encoding.sizes=NULL;
                      }
                    }
                  }
                }
                X11.XFreeFontNames(names); IF HAVE_LIB_FONTCONFIG=TRUE THEN config=fc.FcInitLoadConfigAndFonts();
                oSet=fc.FcObjectSetCreate();
                fc.FcObjectSetAdd(oSet,fc.FC_FOUNDRY);
                oSet2=fc.FcObjectSetCreate();
                fc.FcObjectSetAdd(oSet2,fc.FC_FAMILY);
                pattern=fc.FcPatternCreate();
                set=fc.FcFontList(config,pattern,oSet);
                for (/*TODO*/x=0TOset.nfont-1) {
                  if (fc.FcPatternGetString(set.fonts[x],fc.FC_FOUNDRY,0,buffer)==0) {
                    z=0;
                    while (buffer[z]!=0x0) {
                      fName[z]=buffer[z];
                      z++;
                    }
                    fName[z]=0x0;
                    pattern2=fc.FcPatternCreate();
                    if (fc.FcPatternAddString(pattern2,fc.FC_FOUNDRY,buffer^)) {
                    }
                    set2=fc.FcFontList(config,pattern2,oSet2);
                    fc.FcPatternDestroy(pattern2);
                  }
                  else {
                    fName="Xft";
                    set2=fc.FcFontList(config,pattern,oSet2);
                  }
                  for (/*TODO*/y=0TOset2.nfont-1) {
                    if (fc.FcPatternGetString(set2.fonts[y],fc.FC_FAMILY,0,buffer2)==0) {
                      z=0;
                      while (buffer2[z]!=0x0) {
                        if ((z==0) || (buffer2[z-1]==" ")) {
                          name[z]=CAP(buffer2[z]);
                        }
                        else {
                          name[z]=buffer2[z];
                        }
                        z++;
                      }
                      name[z]=0x0;
                      family=list.GetOrCreateFamily(name);
                      foundry=family.GetOrCreateFoundry(fName);
                      encoding=foundry.GetOrCreateEncoding(unicodeEncoding);
                    }
                  }
                  fc.FcFontSetDestroy(set2);
                }
                fc.FcFontSetDestroy(set); ELSIF HAVE_LIB_XFT=TRUE THEN set=Xft.XftListFonts(D.display(Display).display,D.display(Display).scrNum,NULL,Xft.XFT_FOUNDRY,NULL);
                if (set!=NULL) {
                  for (/*TODO*/x=0TOset.nfont-1) {
                    if (Xft.XftPatternGetString(set.fonts[x],Xft.XFT_FOUNDRY,0,tmp)==0) {
                      z=0;
                      while (tmp[z]!=0x0) {
                        fName[z]=tmp[z];
                        z++;
                      }
                      fName[z]=0x0;
                      set2=Xft.XftListFonts(D.display(Display).display,D.display(Display).scrNum,Xft.XFT_FOUNDRY,Xft.XftTypeString,s.ADR(fName[0]),NULL,Xft.XFT_FAMILY,Xft.XFT_ENCODING,NULL);
                    }
                    else {
                      fName="Xft";
                      set2=Xft.XftListFonts(D.display(Display).display,D.display(Display).scrNum,NULL,Xft.XFT_FAMILY,Xft.XFT_ENCODING,NULL);
                    }
                    for (/*TODO*/y=0TOset2.nfont-1) {
                      if (Xft.XftPatternGetString(set2.fonts[y],Xft.XFT_FAMILY,0,tmp)==0) {
                        z=0;
                        while (tmp[z]!=0x0) {
                          if ((z==0) || (name[z-1]==" ")) {
                            name[z]=CAP(tmp[z]);
                          }
                          else {
                            name[z]=tmp[z];
                          }
                          z++;
                        }
                        name[z]=0x0;
                        family=list.GetOrCreateFamily(name);
                        foundry=family.GetOrCreateFoundry(fName);
                        t=0;
                        while (Xft.XftPatternGetString(set2.fonts[y],Xft.XFT_ENCODING,t,tmp)==0) {
                          z=0;
                          while (tmp[z]!=0x0) {
                            name[z]=tmp[z];
                            z++;
                          }
                          name[z]=0x0;
                          encoding=foundry.GetOrCreateEncoding(name);
                          t++;
                        }
                      }
                    }
                    Xft.XftFontSetDestroy(set2);
                  }
                  Xft.XftFontSetDestroy(set);
                } END
                return list;
              }

#endif

      bool Display::AllocateColor(double red, double green, double blue,
                                  OS::Color& color)
      {
        XColor xcolor;

        xcolor.red=(unsigned short)(red*std::numeric_limits<unsigned short>::max());
        xcolor.green=(unsigned short)(green*std::numeric_limits<unsigned short>::max());
        xcolor.blue=(unsigned short)(blue*std::numeric_limits<unsigned short>::max());

        if (XAllocColor(display,colorMap,&xcolor)!=0) {
          color=xcolor.pixel;
          return true;
        }
        else {
          return false;
        }
      }

      bool Display::AllocateNamedColor(const std::string& name,
                                       OS::Color& color)
      {
        XColor exact;
        XColor xcolor;

        if (XLookupColor(display,colorMap,name.c_str(),&exact,&xcolor)!=0) {
          if (XAllocColor(display,colorMap,&xcolor)!=0) {
            color=xcolor.pixel;
            return true;
          }
        }

        return false;
      }

      void Display::FreeColor(OS::Color color)
      {
        XFreeColors(display,colorMap,&color,1,0);
      }

#if 0

      Sleep Display::AddSleep(Lum::Base::Object::MsgObject object)
      {
        Sleep sleep;


        sleep=new Sleep;
        sleep.object=object;
        sleep.next=d.sleepList;
        d.sleepList=sleep;
        return sleep;
      }

      void Display::RemoveSleep(OS::Sleep sleep)
      {
        Sleep help;
        Sleep last;
        if (d.sleepList==NULL) {
          return ;
        }
        if (d.sleepList==sleep) {
          d.sleepList=d.sleepList.next;
          return ;
        }
        help=d.sleepList.next;
        last=d.sleepList;
        while ((help!=NULL) && (help!=sleep)) {
          last=help;
          help=help.next;
        }
        if (help!=NULL) {
          last.next=help.next;
        }
      }

      Channel Display::AddChannel(Channel channel, unsigned long ops, OS::Object::MsgObject object)
      {
        Channel entry;


        entry=new Channel;
        entry.channel=channel;
        entry.object=object;
        entry.next=d.channelList;
        d.channelList=entry;
        entry.key=channel.RegisterWithSelector(d.selector,ops,NULL);
        return entry;
      }

      void Display::RemoveChannel(OS::Channel channel)
      {
        Channel help;
        Channel last;
        if (channel(Channel).key!=NULL) {
          channel(Channel).key.Cancel;
          channel(Channel).key=NULL;
        }
        if (d.channelList==NULL) {
          return ;
        }
        if (d.channelList==channel) {
          d.channelList=d.channelList.next;
          return ;
        }
        help=d.channelList.next;
        last=d.channelList;
        while ((help!=NULL) && (help!=channel)) {
          last=help;
          help=help.next;
        }
        if (help!=NULL) {
          last.next=help.next;
        }
      }

#endif

/*      ::Lum::OS::Bitmap Display::CreateBitmapPattern(char pattern[], int width, int height)
      {
        ::Lum::OS::Bitmap bitmap;

        bitmap=D.factory.CreateBitmap();
        bitmap(Bitmap).pixmap=X11.XCreateBitmapFromData(d.display,X11.XRootWindow(d.display,d.scrNum),pattern,width,height);
        if (bitmap(Bitmap).pixmap==0) {
          return NULL;
        }
        bitmap.draw=new DrawInfo();
        bitmap.width=width;
        bitmap.height=height;
        return bitmap;
      }*/

#if defined(HAVE_LIBSM)
      static void callback_die(SmcConn smc_conn, SmPointer client_data)
      {
        ((Display*)OS::display)->CallbackDie(smc_conn,client_data);
      }

      static void callback_save_yourself(SmcConn smc_conn, SmPointer client_data,
                                         int save_style, Bool shutdown,
                                         int interact_style,
                                         Bool fast)
      {
        ((Display*)OS::display)->CallbackSaveYourself(smc_conn,client_data,
                                                      save_style,shutdown,
                                                      interact_style,fast);
      }

      static void callback_shutdown_cancelled(SmcConn smc_conn, SmPointer client_data)
      {
        ((Display*)OS::display)->CallbackShutdownCancelled(smc_conn,client_data);
      }

      static void callback_save_complete(SmcConn smc_conn, SmPointer client_data)
      {
        ((Display*)OS::display)->CallbackSaveComplete(smc_conn,client_data);
      }

      static void ice_io_error_handler(IceConn connection)
      {
        ((Display*)OS::display)->CallbackICEIOError(connection);
      }
#endif

#if defined(HAVE_LIB_DBUS)
      static dbus_bool_t AddWatch(DBusWatch *watch, void *data)
      {
        if (!dbus_watch_get_enabled(watch)) {
          return true;
        }

        Display::DBusWatchEntry handle;
        Display                 *display=static_cast<Display*>(data);

        handle.watch=watch;

        display->dbusWatches.push_back(handle);

        return true;
      }

      static void RemoveWatch(DBusWatch *watch, void *data)
      {
        //std::cout << "RemoveWatch " << dbus_watch_get_unix_fd(watch) << " " << dbus_watch_get_flags(watch) << std::endl;
      }

      static void WatchToggled(DBusWatch *watch, void *data)
      {
        //std::cout << "WatchToggled " << dbus_watch_get_enabled(watch) << std::endl;

        if (dbus_watch_get_enabled(watch)) {
          AddWatch(watch,data);
        }
        else {
          RemoveWatch(watch,data);
        }
      }

      static dbus_bool_t AddTimeout(DBusTimeout *timeout, void *data)
      {
        //std::cout << "AddTimeout " << dbus_timeout_get_interval(timeout) << std::endl;

        if (!dbus_timeout_get_enabled(timeout)) {
          return true;
        }

        Display::DBusTimeoutEntry handle;
        Display                   *display=static_cast<Display*>(data);

        handle.timeout=timeout;

        display->dbusTimeouts.push_back(handle);

        return true;
      }

      static void RemoveTimeout(DBusTimeout *timeout, void *data)
      {
        //std::cout << "RemoveTimeout " << dbus_timeout_get_interval(timeout) << std::endl;
      }

      static void TimeoutToggled(DBusTimeout *timeout, void *data)
      {
        //std::cout << "TimeoutToggled" << std::endl;

        if (dbus_timeout_get_enabled(timeout)) {
          AddTimeout(timeout,data);
        }
        else {
          RemoveTimeout(timeout,data);
        }
      }

      static void WakeupMain(void *data)
      {
        //std::cout << "WakeupMain" << std::endl;
      }

      static DBusHandlerResult DBusMsgHandler(DBusConnection *connection, DBusMessage *msg, void *data)
      {
        const char* interface;
        const char* type;
        bool        isMethod;

        if (dbus_message_get_type(msg)==DBUS_MESSAGE_TYPE_METHOD_CALL) {
          type="method";
          isMethod=true;
        }
        else if (dbus_message_get_type(msg) == DBUS_MESSAGE_TYPE_SIGNAL) {
          type="signal";
          isMethod=false;
        }
        else {
          return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        interface=dbus_message_get_interface(msg);

        //std::out << "Got a " <<  type << " " <<  (isMethod?"to":"from") << " interface '" << interface << "' '" << dbus_message_get_path(msg) << "' '" << dbus_message_get_member(msg) << "'" << std::endl;

        if (dbus_message_get_type(msg)==DBUS_MESSAGE_TYPE_METHOD_CALL) {
          DBusMessage *response;
          std::string appName;
          std::string error;

          appName=Lum::Base::WStringToString(display->GetAppName());
          Lum::Base::ToLower(appName);

          error=std::string(dbusDomain)+"."+appName+".notImplemented";

          response=dbus_message_new_error(msg,error.c_str(),"Message not implemented");
          dbus_connection_send(connection,response,NULL);
          dbus_message_unref(response);
        }

        return DBUS_HANDLER_RESULT_HANDLED;
//          return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
/*
364     for(i=0; i<osso->ifs->len; i++) {
365         _osso_interface_t *intf;
366         DBusHandlerResult r;
367         intf = &g_array_index(osso->ifs, _osso_interface_t, i);
368         if(intf->method == is_method) {
369             dprint("comparing '%s' to '%s'",interface, intf->interface);
370             if(strcmp(interface, intf->interface) == 0) {
371                 dprint("match!, now calling callback at %p", intf->handler);
372                 r = (intf->handler)(osso, msg, intf->data);
373                 if(r == DBUS_HANDLER_RESULT_HANDLED) {
374                     return r;
375                 }
376             }
377         }
378     }
379
380     return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;*/
      }

      DBusConnection *Display::InitiateDBusConnection(DBusBusType busType)
      {
        DBusConnection       *connection;
        DBusError            err;
        //DBusObjectPathVTable vtable;
        std::string          appName;
        std::string          serviceName;
        std::string          objectName;

        dbus_error_init(&err);

        connection=dbus_bus_get(busType,&err);
        if (connection==NULL) {
          std::cerr << "Cannot connect to DBUS session: " << err.message << std::endl;
          dbus_error_free(&err);
          return NULL;
        }

        dbus_connection_set_exit_on_disconnect(connection,false);
        /* ignore */dbus_connection_set_watch_functions(connection,
                                                        AddWatch,
                                                        RemoveWatch,
                                                        WatchToggled,
                                                        (void*)this,
                                                        NULL);

        /* ignore */dbus_connection_set_timeout_functions(connection,
                                                          AddTimeout,
                                                          RemoveTimeout,
                                                          TimeoutToggled,
                                                          (void*)this,
                                                          NULL);

        dbus_connection_set_wakeup_main_function(connection,
                                                 WakeupMain,
                                                 (void*)this,
                                                 NULL);


        appName=Lum::Base::WStringToString(GetAppName());
        for (size_t i=0; i<appName.length(); ++i) {
          appName[i]=tolower(appName[i]);
        }

//        if (busType==DBUS_BUS_SESSION) {
          serviceName=std::string(dbusDomain)+"."+appName;

          if (dbus_bus_request_name(connection,serviceName.c_str(),DBUS_NAME_FLAG_ALLOW_REPLACEMENT,&err)==-1) {
            //std::cerr << "Cannot request name '" << serviceName << "': "<< err.message << std::endl;
            dbus_error_free(&err);
          }
//        }

        /*
        vtable.message_function=DBusMsgHandler;
        vtable.unregister_function=NULL;

        objectName="/or/teulings/illumination/"+appName;

        if(!dbus_connection_register_object_path(connection,objectName.c_str(),&vtable,NULL)) {
          std::cerr << "Cannot register object '" << objectName << "'" << std::endl;
        }
        */

        dbus_connection_add_filter(connection,DBusMsgHandler,NULL,NULL);

        dbus_connection_set_exit_on_disconnect(connection,false);

        return connection;
      }
#endif

#if defined(HAVE_LIB_GSTREAMER)
      static GstBusSyncReply gstMessageHandler(GstBus*, GstMessage *message, gpointer data)
      {
        Display *display=static_cast<Display*>(data);

        if (GST_MESSAGE_TYPE(message)==GST_MESSAGE_ERROR) {
          GError *error;
          char   *text;

          gst_message_parse_error(message,&error,&text);

          std::cerr << "gstreamer ERROR: " << text << std::endl;

         g_free(error);
         g_free(text);
       }
       else if (GST_MESSAGE_TYPE(message)==GST_MESSAGE_WARNING) {
         GError *error;
         char   *text;

         gst_message_parse_warning(message,&error,&text);

         std::cerr << "gstreamer WARNING: " << text << std::endl;

         g_free(error);
         g_free(text);
      }
	
      message=gst_message_copy(message);

      display->HandleGstMessage(message);

      // We don't free the message here, that will be done later in the main event loop
      return GST_BUS_PASS;
    }

      void Display::AddGstListener(GstElement* element, Model::Action* action, bool async)
      {
        Guard<Mutex> guard(gstMutex);

        assert(element!=NULL && action!=NULL);

        GstListener listener;

        GstBus *bus;

        bus=gst_element_get_bus(element);

        if (!gstSyncSet) {
          gstSyncSet=true;
          gst_bus_set_sync_handler(bus,gstMessageHandler,this);
        }

        gst_object_unref(bus);

        listener.async=async;
        listener.element=element;
        listener.action=action;

        gstListener.push_back(listener);
      }

      void Display::RemoveGstListener(Model::Action* action)
      {
        assert(action!=NULL);

        std::list<GstListener>::iterator iter;

        iter=gstListener.begin();
        while (iter!=gstListener.end()) {
          if (iter->action.Get()==action) {
            iter=gstListener.erase(iter);
          }
          else {
            ++iter;
          }
        }
      }

      void Display::HandleGstMessage(GstMessage *message)
      {
        Guard<Mutex> guard(gstMutex);

        // First handle sync notification requests
        for (std::list<GstListener>::iterator iter=gstListener.begin(); iter!=gstListener.end(); ++iter) {
          if (!iter->async && message->src==GST_OBJECT(iter->element)) {
            GstMsg msg;

            msg.message=message;
            iter->action->Trigger(msg);
            gst_message_unref(message);
            return;
          }
        }

        // Now handle async notification requests
        gstMessages.push_back(message);

        SignalEventLoop();
      }
#endif

      Display::Display()
      : currentWin(NULL),
        selectObject(NULL),
        querySelectObject(NULL),
        queryClipboardObject(NULL),
        selClearPend(false),
        contextHelp(true),
        dropWindow(NULL),
        exit(true),
        appWindow(0),
        scrNum(0),
        display(NULL),
        visual(NULL),
        xim(0)
#if defined(HAVE_LIB_DBUS)
        ,dbusSession(NULL),
        dbusSystem(NULL)
#endif
#if defined(HAVE_LIB_GSTREAMER)
        ,gstSyncSet(false)
#endif
      {
        // no code
      }

      static void XIMInstantiateCallback(::Display *display,
                                         XPointer clientData,
                                         XPointer callData)
      {
        std::cout << "XIM found!" << std::endl;

        ((Display*)clientData)->xim=XOpenIM(display,NULL,NULL,NULL);
      }

      bool Display::Open()
      {
#if !defined(HAVE_LIB_GTK)
        const char*    result;
#endif
        OS::FontRef    font;
        Atom           atomType;
        int            atomFormat;
        unsigned long  atomItems;
        unsigned long  atomBytes;
        unsigned char* atomProperties;
        int            res;

        /*if (XInitThreads()==0) {
          std::cerr << "Warning: X11 does not support multiple threads!" << std::endl;
        }*/

#if defined(HAVE_LIB_GTK)
        if (!gtk_init_check(NULL,NULL)) {
          std::cerr << "Cannot initialize GTK!" << std::endl;
          return false;
        }

        g_thread_init(NULL);
        gdk_threads_init();

        display=GDK_DISPLAY_XDISPLAY(gdk_display_get_default());
        scrNum=GDK_SCREEN_XNUMBER(gdk_screen_get_default());
        visual=GDK_VISUAL_XVISUAL(gdk_visual_get_system());

        colorDepth=gdk_visual_get_system()->depth;
        colorMap=GDK_COLORMAP_XCOLORMAP(gdk_colormap_get_system());

        gdk_pixbuf_xlib_init(display,scrNum);
#else
        result=XDisplayName(NULL);
        if (result!=NULL) {
          displayName=result;
        }
        display=XOpenDisplay(displayName.c_str());
        if (display==NULL) {
          std::cerr << "Cannot open X11 display!" << std::endl;
          return false;
        }
        scrNum=XDefaultScreen(display);
        visual=XDefaultVisual(display,scrNum);

        if (visual==NULL) {
          std::cerr << "Cannot get visual!" << std::endl;
          return false;
        }

        colorDepth=XDefaultDepth(display,scrNum);
        colorMap=XDefaultColormap(display,scrNum);
#endif

#if defined(HAVE_LIB_GSTREAMER)
        gst_init(NULL,NULL);
#endif

        if (pipe(signalPipe)!=0) {
          std::cerr << "Cannot create signal pipe!" << std::endl;
        }

        Images::Factory::Set(driver->CreateImageFactory(this));

        scrWidth=XDisplayWidth(display,scrNum);
        scrHeight=XDisplayHeight(display,scrNum);

        type=typeGraphical;

        EvaluateDisplaySize();

        dpi=XDisplayHeight(display,scrNum)/(XDisplayHeightMM(display,scrNum)/25.4);
        /*
        int            eventBase,errorBase;

        // Checking for shape extension
        if (XShapeQueryExtension(display,&eventBase,&errorBase)) {
          // TODO
        }
        */

        if (XSupportsLocale()) {
          if (XSetLocaleModifiers("")==NULL) {
            std::cerr << "Cannot set locale modifiers!" << std::endl;
          }

          xim=XOpenIM(display,NULL,NULL,NULL);

          if (xim==0) {
            XRegisterIMInstantiateCallback(display,
                                           NULL,NULL,NULL,
                                           XIMInstantiateCallback,
                                           (XPointer)display);
          }
        }

        if ((visual->c_class==GrayScale) || (visual->c_class==StaticGray)) {
          if (colorDepth==1) {
            colorMode=colorModeMonochrome;
          }
          else {
            colorMode=colorModeGreyScale;
          }
        }
        else if ((visual->c_class==PseudoColor) ||
                 (visual->c_class==StaticColor) ||
                 (visual->c_class==DirectColor) ||
                 (visual->c_class==TrueColor)) {
          colorMode=colorModeColor;
        }
        else {
          std::cerr << "Unsupported visual class!" << std::endl;
          return false;
        }


        XSetWindowAttributes attr;

        attr.colormap=colorMap;
        attr.bit_gravity=NorthWestGravity;
        attr.event_mask=KeyPressMask | KeyReleaseMask |  ExposureMask |
                        StructureNotifyMask | ButtonPressMask |  ButtonReleaseMask |
                        ButtonMotionMask | PointerMotionMask | FocusChangeMask;

        attr.override_redirect=False;
        attr.save_under=False;
        attr.backing_store=False;
        appWindow=XCreateWindow(display,
                                XRootWindow(display,scrNum),
                                0,0,
                                10,10,
                                CopyFromParent,
                                CopyFromParent,
                                InputOutput,
                                visual,
                                CWBorderPixel
                                | CWBitGravity
                                | CWBackingStore
                                | CWOverrideRedirect
                                | CWSaveUnder
                                | CWEventMask
                                | CWColormap,
                                &attr);

        if (appWindow==0) {
          std::cerr << "Cannot create hidden application window!" << std::endl;
          return false;
        }

        for (size_t i=0; i<atomCount; i++) {
          atoms[i]=XInternAtom(display,atomNames[i],False);
        }

                      /*
        D.smallChess:=d.CreateBitmapPattern(D.disablePattern,D.disableWidth,D.disableHeight);
        D.bigChess:=d.CreateBitmapPattern(D.bigChessPattern,D.bigChessWidth,D.bigChessHeight);

                      */
        prefs=new X11::Prefs(this);
        prefs->Initialize();

        Lum::Prefs::LoadConfig(GetPrefsName());
        Lum::Prefs::ReadDisplayConfig();
        if (!prefs->theme.empty()) {
          Lum::Prefs::LoadTheme(GetThemeName());
          Lum::Prefs::ReadDisplayTheme();
        }

#if defined(HAVE_LIB_GTK)
        if ((driver->GetDriverName()==L"X11" ||
             driver->GetDriverName()==L"CairoX11")) {
          theme=new GtkTheme(this);
        }
        else {
          theme=new OS::Base::DefaultTheme(this);
        }
#else
        theme=new OS::Base::DefaultTheme(this);
#endif

        font=driver->CreateFont();
        font->features=Font::attrName|Font::attrHeight|Font::attrSpacing|Font::attrCharSet;
        font->name=Lum::Base::WStringToString(theme->GetProportionalFontName());
        font->spacing=Font::spacingProportional;
        font->charSet=unicodeEncoding;
        font->pixelHeight=theme->GetProportionalFontSize();

        propFont=font->Load();

        if (!propFont.Valid()) {
          std::cerr << "Cannot load font '" << dynamic_cast<Font*>(font.Get())->fullName << "'/"<< font->name << ", " << font->pixelHeight << "!" << std::endl;
#if !defined(HAVE_LIB_GTK)
          XCloseDisplay(display);
#endif
          return false;
        }

        font=driver->CreateFont();
        font->features=Font::attrName|Font::attrHeight|Font::attrSpacing|Font::attrCharSet;
        font->name=Lum::Base::WStringToString(theme->GetFixedFontName());
        font->spacing=Font::spacingMonospace;
        font->charSet=unicodeEncoding;
        font->pixelHeight=theme->GetFixedFontSize();

        fixedFont=font->Load();

        if (!fixedFont.Valid()) {
          std::cerr << "Cannot load font '" << dynamic_cast<Font*>(font.Get())->fullName << "'/"<< font->name << ", " << font->pixelHeight << "!" << std::endl;
#if !defined(HAVE_LIB_GTK)
          XCloseDisplay(display);
#endif
          return false;
        }

        propFontSize=propFont->pixelHeight;
        fixedFontSize=fixedFont->pixelHeight;

        for (size_t i=0; i<colorCount; i++) {
          color[i]=theme->GetColor((ColorIndex)i);
        }

      /*
        d.sleepCursor:=X11.XCreateFontCursor(d.display,sleepCursor);
        IF d.sleepCursor=0 THEN
          Err.String("Cannot get sleep cursor!"); Err.Ln;
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

        d.popCursor:=X11.XCreateFontCursor(d.display,popCursor);
        IF d.popCursor=0 THEN
          Err.String("Cannot get popup cursor!"); Err.Ln;
          X11.XFreeCursor(d.display,d.sleepCursor);
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

        d.dndCursor:=X11.XCreateFontCursor(d.display,dndCursor);
        IF d.dndCursor=0 THEN
          Err.String("Cannot get drag & drop cursor!"); Err.Ln;
          X11.XFreeCursor(d.display,d.sleepCursor);
          X11.XFreeCursor(d.display,d.popCursor);
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

        d.copyCursor:=CreatePixmapCursor(s.ADR(copyCursorData),s.ADR(copyMaskData),14,15);
        IF d.copyCursor=0 THEN
          Err.String("Cannot create copy cursor"); Err.Ln;
          X11.XFreeCursor(d.display,d.sleepCursor);
          X11.XFreeCursor(d.display,d.popCursor);
          X11.XFreeCursor(d.display,d.dndCursor);
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

        d.moveCursor:=CreatePixmapCursor(s.ADR(moveCursorData),s.ADR(moveMaskData),11,13);
        IF d.moveCursor=0 THEN
          Err.String("Cannot create move cursor"); Err.Ln;
          X11.XFreeCursor(d.display,d.sleepCursor);
          X11.XFreeCursor(d.display,d.popCursor);
          X11.XFreeCursor(d.display,d.dndCursor);
          X11.XFreeCursor(d.display,d.copyCursor);
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

        d.linkCursor:=CreatePixmapCursor(s.ADR(linkCursorData),s.ADR(linkMaskData),15,17);
        IF d.linkCursor=0 THEN
          Err.String("Cannot create move cursor"); Err.Ln;
          X11.XFreeCursor(d.display,d.sleepCursor);
          X11.XFreeCursor(d.display,d.popCursor);
          X11.XFreeCursor(d.display,d.dndCursor);
          X11.XFreeCursor(d.display,d.copyCursor);
          X11.XFreeCursor(d.display,d.moveCursor);
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        END;

      */

        // Find out dimensions of workSpace, this is size of desktop minus globals toolsbars etc...
        atomProperties=NULL;
        res=XGetWindowProperty(display,RootWindow(display,scrNum),
                               atoms[netWorkAreaAtom],2,2,false,atoms[cardinalAtom],
                               &atomType,&atomFormat,&atomItems,&atomBytes,&atomProperties);
        if ((res==Success) && (atomProperties != NULL)) {
          workAreaWidth=reinterpret_cast<uint32_t*>(atomProperties)[0];
          workAreaHeight=reinterpret_cast<uint32_t*>(atomProperties)[1];
        }
        XFree(atomProperties);

        /*
        d.contextTimer.SetSecs(D.prefs.contextTimer DIV 1000,D.prefs.contextTimer MOD 1000);
                      d.contextTimer.SetObject(d);
        */

                      /*
        d.sleepList:=NIL;
        d.channelList:=NIL;

        d.dragging:=FALSE;
        d.dragObject:=NIL;
       */

        multiClickTime=200;

        Lum::Prefs::InitializePrefs(prefs);
        Lum::Prefs::ReadOthersConfig();
        if (!prefs->theme.empty()) {
          Lum::Prefs::ReadOthersTheme();
        }
        Lum::Prefs::FreeData();

#if 0
        if (getenv("SESSION_MANAGER")!=NULL) {
          SmcCallbacks callbacks;
          char*        clientId;
          char         errorString[4096];

          IceSetIOErrorHandler(ice_io_error_handler);

          callbacks.save_yourself.callback=callback_save_yourself;
          callbacks.die.callback=callback_die;
          callbacks.save_complete.callback=callback_save_complete;
          callbacks.shutdown_cancelled.callback=callback_shutdown_cancelled;

          callbacks.save_yourself.client_data=(SmPointer)NULL;
          callbacks.die.client_data=(SmPointer)NULL;
          callbacks.save_complete.client_data=(SmPointer)NULL;
          callbacks.shutdown_cancelled.client_data=(SmPointer)NULL;
          smSession=SmcOpenConnection(NULL,NULL,
                                      SmProtoMajor, SmProtoMinor,
                                      SmcSaveYourselfProcMask | SmcDieProcMask |
                                      SmcSaveCompleteProcMask |
                                      SmcShutdownCancelledProcMask,
                                      &callbacks,NULL,&clientId,
                                      4096,errorString);

          if (smSession) {
            iceSession=SmcGetIceConnection(smSession);
            iceFd=IceConnectionNumber(iceSession);
          }
          else {
            std::cerr << "Error while connecting to session manager: " << errorString << "!" << std::endl;
          }
        }
#endif

#if defined(HAVE_LIB_DBUS)
        dbusSession=InitiateDBusConnection(DBUS_BUS_SESSION);
        if (dbusSession==NULL) {
          std::cerr << "Cannot create session DBUS connection!" << std::endl;
        }
        dbusSystem=InitiateDBusConnection(DBUS_BUS_SYSTEM);
        if (dbusSystem==NULL) {
          std::cerr << "Cannot create system DBUS connection!" << std::endl;
        }
#endif

#if defined(HAVE_LIB_HILDON)
        keyboardHandler=new HildonKeyboardHandler();
#endif
        return true;
      }

      void Display::Close()
      {
        std::list<Window*>::iterator win;

        OS::Base::Display::Close();

#if defined(HAVE_LIB_DBUS)
        if (dbusSystem!=NULL) {
          dbus_connection_unref(dbusSystem);
        }
        if (dbusSession!=NULL) {
          dbus_connection_unref(dbusSession);
        }
#endif

#if 0
        if (smSession!=NULL) {
          SmcCloseConnection(smSession,0,NULL);
        }
#endif

        Lum::Prefs::FreePrefs();

        if (selectObject!=NULL) {
          CancelSelection();
        }
        ClearClipboard();

        win=winList.begin();
        while (win!=winList.end()) {
          std::cerr << "Warning: window '" <<Lum::Base::WStringToString((*win)->GetTitle()) << "' not explicitely closed!" << std::endl;
          (*win)->Close(); // removes itself from list

          win=winList.begin();
        }

        /*
        if (d.sleepCursor!=0) {
          X11.XFreeCursor(d.display,d.sleepCursor);
        }
        */

        XDestroyWindow(display,appWindow);

        delete Images::Factory::factory;

        if (xim!=NULL) {
          if (XCloseIM(xim)!=0) {
          }
        }

        close(signalPipe[0]);
        close(signalPipe[1]);

#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        Cairo::FreeGlobalCairo();
        Cairo::FreeGlobalSurface();
#endif

#if !defined(HAVE_LIB_GTK)
        if (display!=NULL) {
          XCloseDisplay(display);
        }
#endif

#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        //cairo_debug_reset_static_data();
#endif
      }

       /**
        Flushes all pending events. makes only sense for asnsycronous
        windowing systems like X11.
       */
       void Display::Flush()
       {
         XSync(display,False);
       }

       void Display::Beep()
       {
         XBell(display,100);
       }


      bool Display::RegisterSelection(Lum::Base::DnDObject* object,
                                      OS::Window* window)
      {
        if (selectObject!=object) {
          if (selectObject!=NULL) {
            selectObject->Deselect();
            selectObject=NULL;
          }
          XSetSelectionOwner(display,
                             XA_PRIMARY,
                             dynamic_cast<Window*>(window)->window,
                             CurrentTime);
          selectObject=object;
        }

        return true;
      }

      void Display::CancelSelection()
      {
        assert(selectObject!=NULL);
        selClearPend=true;
        XSetSelectionOwner(display,XA_PRIMARY,None,CurrentTime);
        selectObject->Deselect();
        selectObject=NULL;
      }

      Lum::Base::DnDObject* Display::GetSelectionOwner() const
      {
        return selectObject;
      }

      bool Display::QuerySelection(OS::Window* window,
                                   Lum::Base::DnDObject* object)
      {
        XConvertSelection(display,
                          XA_PRIMARY,
                          XA_STRING,
                          atoms[selectionAtom],
                          dynamic_cast<Window*>(window)->window,
                          CurrentTime);

        querySelectObject=object;

        return false;
      }

      bool Display::SetClipboard(const std::wstring& content)
      {
        if (!content.empty()) {
          clipboard=content;
          XSetSelectionOwner(display,atoms[clipboardAtom],appWindow,CurrentTime);
          return true;
        }
        else {
          return false;
        }
      }

      std::wstring Display::GetClipboard() const
      {
        if (!clipboard.empty()) {
          return clipboard;
        }

        XEvent         event;
        Atom           clipboardType=XA_STRING;
        Atom           retType;
        int            retFormat;
        unsigned long  itemsReturn,bytesLeft;
        unsigned char  *data;

        XConvertSelection(display,
                          atoms[clipboardAtom],
                          atoms[targetsAtom],
                          atoms[clipBufferAtom],
                          appWindow,
                          CurrentTime);

        if (!WaitTimedForX11Event(appWindow,SelectionNotify,event) ||
            event.xselection.property==None) {
          // This is not necesarly an error! Not all application might support this!
        }
        else {
          if (XGetWindowProperty(display,
                                 event.xselection.requestor,
                                 event.xselection.property,
                                 0,
                                 65000,
                                 True,
                                 AnyPropertyType,
                                 &retType,&retFormat,
                                 &itemsReturn,&bytesLeft,
                                 &data)!=Success) {
            std::cout << "Canot get list of support selection types!" << std::endl;
          }
          else if (retType!=atoms[atomAtom] || retFormat/8!=sizeof(Atom)) {
            std::cerr << "Unexspected type of clipboard format list: "<< retType << "!" << std::endl;
            ::XFree(data);
            return L"";
          }
          else {
            Atom *formats=(Atom*)data;

            for (size_t x=0; x<itemsReturn; x++) {
              if (formats[x]==atoms[utf8StringAtom]) {
                clipboardType=atoms[utf8StringAtom];
              }
            }

            ::XFree(data);
          }
        }

        XConvertSelection(display,
                          atoms[clipboardAtom],
                          clipboardType,
                          atoms[clipBufferAtom],
                          appWindow,
                          CurrentTime);

        if (!WaitTimedForX11Event(appWindow,SelectionNotify,event)) {
          std::cerr << "Could not get clipboard response!" << std::endl;
          return L"";
        }

        if (event.xselection.property==None) {
          std::cerr << "Clipboard is empty!" << std::endl;
          return L"";
        }

        if (event.xselection.property!=atoms[clipBufferAtom]) {
          std::cerr << "Could not get clipboard!" << std::endl;
          return L"";
        }

        if (XGetWindowProperty(display,
                               event.xselection.requestor,
                               event.xselection.property,
                               0,
                               65000,
                               True,
                               AnyPropertyType,
                               &retType,&retFormat,
                               &itemsReturn,&bytesLeft,
                               &data)!=Success) {
          std::cerr << "Cannot get property data!" << std::endl;
          return L"";
        }

        std::wstring result;

        if (retType==None || itemsReturn==0) {
          result=L"";
        }
        else if (retType==XA_STRING && retFormat==8) {
          result=Lum::Base::StringToWString((char*)data);
        }
        else if (retType==atoms[utf8StringAtom] && retFormat==8) {
          result=Lum::Base::UTF8ToWString((char*)data);
        }
        else {
          std::cerr << "Unsoported clipboard content of type " << retType << " and format " << retFormat << "!"<< std::endl;
          result=L"";
        }

        ::XFree(data);

        return result;
      }

      void Display::ClearClipboard()
      {
        if (!clipboard.empty()) {
          selClearPend=true;
          XSetSelectionOwner(display,atoms[clipboardAtom],None,CurrentTime);
          clipboard=L"";
        }
      }

      void Display::PutBackEvent(OS::Event* event,
                                 OS::Window* destWin)
      {
        XEvent x11Event;

        GetX11Event(event,x11Event);
        x11Event.xany.window=dynamic_cast<Window*>(destWin)->window;
        if (XSendEvent(display,dynamic_cast<Window*>(destWin)->window,
                       True,NoEventMask,&x11Event)==0) {
          std::cerr << "Cannot resend event!" << std::endl;
        }
      }

      void Display::Exit()
      {
        assert(!exit);

        exit=true;
      }

      void Display::ReinitWindows()
      {
        std::list<Window*>::iterator win;

        win=winList.begin();
        while (win!=winList.end()) {
          //(*win)->ReinitWindow();

          ++win;
        }
      }

#if 0
      /**
       Check the list of registered sleeps and send a message for every elapsed
       timeout.
      */
      void Display::CheckSleeps()
      {
        Sleep        sleep;
        OS::SleepMsg msg;

        sleep=d.sleepList;
        while (sleep!=NULL) {

          msg=new SleepMsg;
          msg.sleep=sleep;
          sleep.object.Receive(msg);
          sleep=sleep.next;
        }
      }

#endif

      /**
        Handle the XSelecitionNotify of X11. A SelectionNotify gets send
        as a result of our request for the selection value. The event states
        where one can get the selection value from.
      */
      void Display::HandleXSelectionNotify(const XSelectionEvent& event)
      {
        Atom               retType;
        int                retFormat;
        unsigned long      itemsReturn,bytesLeft;
        unsigned char      *data;
        Lum::Base::DnDData dndData;

        if (event.property==None) {
          return;
        }

        if (XGetWindowProperty(display,
                               event.requestor,
                               event.property,
                               0,
                               65000,
                               True,
                               AnyPropertyType,
                               &retType,&retFormat,
                               &itemsReturn,&bytesLeft,
                               &data)!=Success) {
          std::cerr << "Cannot get property data!" << std::endl;
          return;
        }

        if (retType==None) {
          std::cerr << "Illegal property data type!" << std::endl;
          ::XFree(data);
          return;
        }

        if (retType==XA_STRING && retFormat==8) {
          dndData.SetText(Lum::Base::StringToWString((char*)data));
        }
        else {
          std::cerr << "Unsupported drop datatype!" << std::endl;
        }

        ::XFree(data);

        if (event.property==atoms[selectionAtom]) {
          if (querySelectObject!=NULL) {
            if (!querySelectObject->Drop(Lum::Base::DnDType(::Lum::Base::DnDType::text,Lum::Base::DnDType::plain),
                                         dndData,Lum::Base::DnDObject::dndInsert)) {
              Beep();
            }
            querySelectObject=NULL;
          }
        }
        else if (event.property==atoms[clipBufferAtom]) {
          Lum::Base::DnDObject *queryObject;

          queryObject=queryClipboardObject;
          if (queryObject!=NULL) {
            if (!queryObject->Drop(Lum::Base::DnDType(Lum::Base::DnDType::text,Lum::Base::DnDType::plain),
                                   dndData,Lum::Base::DnDObject::dndInsert)) {
              Beep();
            }
            queryObject=NULL;
          }
        }
        else {
          Beep();
        }
      }

      /**
        handle XSelectionrequest of X11. A SelectionRequest gets send when some application
        wants the selection value and your window has registered the selection. We
        ask the object which has registered the selection for the selection value
        and return a notify message to the requestor.
      */
      void Display::HandleXSelectionRequest(const XSelectionRequestEvent& event)
      {
        Lum::Base::DnDData data;
        XEvent             selNotify;
        //Atom               retType;
        //unsigned long      retFormat,itemsReturn,bytesLeft;
        std::string        text;

        selNotify.xselection.type=SelectionNotify;
        selNotify.xselection.display=event.display;
        selNotify.xselection.requestor=event.requestor;
        selNotify.xselection.selection=event.selection;
        selNotify.xselection.target=event.target;
        selNotify.xselection.property=event.property;
        selNotify.xselection.time=event.time;

        if (event.target==XA_STRING || event.target==atoms[compoundTextAtom]) {
          if (event.selection==XA_PRIMARY) {
            if (selectObject!=NULL) {
              if (selectObject->GetDataOfType(Lum::Base::DnDType(Lum::Base::DnDType::text,Lum::Base::DnDType::plain),data)) {
                text=Lum::Base::WStringToString(data.GetText());
              }
            }
          }
          else if (event.selection==atoms[clipboardAtom]) {
            if (!clipboard.empty()) {
              text=Lum::Base::WStringToString(clipboard);
            }
          }

          if (!text.empty()) {
            XChangeProperty(event.display,event.requestor,event.property,
                            XA_STRING,8,
                            PropModeReplace,(unsigned char*)text.c_str(),
                            text.length());
          }
        }
        else {
          selNotify.xselection.property=None;
        }

        if (XSendEvent(selNotify.xselection.display,
                   selNotify.xselection.requestor,
                   True,
                   NoEventMask,
                   &selNotify)==0) {
          std::cerr << "Error sending selection notify!" << std::endl;
        }
      }

      void Display::GetEvent()
      {
        Tray* currentTray;

        XEvent e;

        XNextEvent(display,&e);

        if (XFilterEvent(&e,0)!=0) {
          return;
        }
      /*
        IF d.contextTimer.active THEN
          IF (e.type=X11.ButtonPress) OR (e.type=X11.ButtonRelease) OR (e.type=X11.MotionNotify) OR
            (e.type=X11.KeyPress) OR (e.type=X11.KeyRelease) THEN
            d.RestartContextHelp;
          END;
        END;
                              */

        switch (e.type) {
        case SelectionClear:
          if (!selClearPend) {
            if (e.xselectionclear.selection==XA_PRIMARY) {
              if (selectObject!=NULL) {
                selectObject->Deselect();
                selectObject=NULL;
              }
            }
            else if (e.xselectionclear.selection==atoms[clipboardAtom]) {
              clipboard=L"";
            }
          }
          else {
            selClearPend=false;
          }
          return;
        case SelectionNotify:
          HandleXSelectionNotify(e.xselection);
          return;
        case SelectionRequest:
          HandleXSelectionRequest(e.xselectionrequest);
          return;
        default:
          break;
        }

        currentWin=GetWindow(e.xany.window);
        currentTray=GetTray(e.xany.window);

        if (currentWin!=NULL) {
          OS::Event *event;

          event=X11::GetEvent(e,currentWin->xic);

          if (event!=NULL) {
            do {
              event->reUse=false;
              currentWin->HandleEvent(event);
            }
            while (event->reUse);
          }
        }
        else if (currentTray!=NULL) {
          OS::Event *event;

          event=X11::GetEvent(e,currentTray->xic);

          if (event!=NULL) {
            currentTray->HandleEvent(event);
          }
        }
      }

      /**
        Waits for certain events to happen:
        *A X11 event occurs
        *Timer run out

        Returns TRUE, if the wait exceeds the given timeout, else FALSE, if Wait
        returned because an X11 event is available.

        File descriptors getting available will be handled internaly. In this
        case a notification will be send and the wait will be restarted.
      */
      bool Display::Wait()
      {

        while (true) {
          fd_set                readSet,writeSet;
          int                   ret;
          Lum::Base::SystemTime tmp;
          int                   max;
          int                   x11Fd;

          x11Fd=ConnectionNumber(display);

          FD_ZERO(&readSet);
          FD_ZERO(&writeSet);

          FD_SET(x11Fd,&readSet);
          max=x11Fd;

          FD_SET(signalPipe[0],&readSet);
          max=std::max(max,signalPipe[0]);
#if 0
          FD_SET(iceFd,&readSet);
          max=std::max(max,iceFd);
#endif

#if defined(HAVE_LIB_DBUS)
          for (std::list<DBusWatchEntry>::const_iterator iter=dbusWatches.begin(); iter!=dbusWatches.end(); ++iter) {
            if (dbus_watch_get_flags(iter->watch) & DBUS_WATCH_READABLE) {
#ifdef HAVE_LIB_HILDON
              FD_SET(dbus_watch_get_fd(iter->watch),&readSet);
              max=std::max(max,dbus_watch_get_fd(iter->watch));
#else
              FD_SET(dbus_watch_get_unix_fd(iter->watch),&readSet);
              max=std::max(max,dbus_watch_get_unix_fd(iter->watch));
#endif
            }

            if (dbus_watch_get_flags(iter->watch) & DBUS_WATCH_WRITABLE) {
#ifdef HAVE_LIB_HILDON
              FD_SET(dbus_watch_get_fd(iter->watch),&writeSet);
              max=std::max(max,dbus_watch_get_fd(iter->watch));
#else
              FD_SET(dbus_watch_get_unix_fd(iter->watch),&writeSet);
              max=std::max(max,dbus_watch_get_unix_fd(iter->watch));
#endif
            }
          }
#endif
          if (GetNextTimer(tmp)) {
            timeval timeout;

            tmp.GetTimeval(timeout);

            ret=select(max+1,&readSet,&writeSet,NULL,&timeout);
          }
          else {
            ret=select(max+1,&readSet,&writeSet,NULL,NULL);
          }

          if (ret==-1) {
            std::cerr << "Error while selecting...!" << std::endl;
          }
          else if (ret>=0) {
            if (FD_ISSET(signalPipe[0],&readSet)) {
              char buffer[2];

              if (read(signalPipe[0],(void*)buffer,(size_t)1)==1) {
#if defined(HAVE_LIB_GSTREAMER)
                {
                  Guard<Mutex> guard(gstMutex);

                  while (gstMessages.begin()!=gstMessages.end()) {
                    for (std::list<GstListener>::iterator iter=gstListener.begin(); iter!=gstListener.end(); ++iter) {
                      if (iter->async && (*gstMessages.begin())->src==GST_OBJECT(iter->element)) {
                        GstMsg msg;

                        msg.message=*gstMessages.begin();
                        iter->action->Trigger(msg);
                      }
                    }

                    gst_message_unref(*gstMessages.begin());
                    gstMessages.erase(gstMessages.begin());
                  }
                }
#endif
              }
              else {
                std::cerr << "Error getting signal!" << std::endl;
              }
            }
#if 0
            if (FD_ISSET(iceFd,&readSet)) {
              if (IceProcessMessages(iceSession,NULL,NULL)==IceProcessMessagesIOError) {
                std::cerr << "Error during processing of Ice message!" << std::endl;
              }
            }
#endif

#if defined(HAVE_LIB_DBUS)
            for (std::list<DBusWatchEntry>::const_iterator iter=dbusWatches.begin(); iter!=dbusWatches.end(); ++iter) {
              if (dbus_watch_get_flags(iter->watch) & DBUS_WATCH_READABLE) {
#ifdef HAVE_LIB_HILDON
                if (FD_ISSET(dbus_watch_get_fd(iter->watch),&readSet)) {
#else
                if (FD_ISSET(dbus_watch_get_unix_fd(iter->watch),&readSet)) {
#endif
                  dbus_watch_handle(iter->watch,DBUS_WATCH_READABLE);
                }
              }
              if (dbus_watch_get_flags(iter->watch) & DBUS_WATCH_WRITABLE) {
#ifdef HAVE_LIB_HILDON
                if (FD_ISSET(dbus_watch_get_fd(iter->watch),&writeSet)) {
#else
                if (FD_ISSET(dbus_watch_get_unix_fd(iter->watch),&writeSet)) {
#endif
                  dbus_watch_handle(iter->watch,DBUS_WATCH_WRITABLE);
                }
              }
            }

            if (dbusSystem!=NULL) {
              while (dbus_connection_get_dispatch_status(dbusSystem)==DBUS_DISPATCH_DATA_REMAINS) {
                dbus_connection_dispatch(dbusSystem);
              }
            }

            if (dbusSession!=NULL) {
              while (dbus_connection_get_dispatch_status(dbusSession)==DBUS_DISPATCH_DATA_REMAINS) {
              dbus_connection_dispatch(dbusSession);
              }
            }
#endif
            return true;
          }
        }
      }

      bool Display::WaitTimedForX11Event(::Window window, int eventType, XEvent& event) const
      {
        Lum::Base::SystemTime timer,max;

        XFlush(display);

        timer.SetTime(3,0);

        max.Add(timer);

        while (true) {
          fd_set                set;
          Lum::Base::SystemTime now,tmp;
          timeval               timeout;
          int                   ret,x11Fd;

          x11Fd=ConnectionNumber(display);

          FD_ZERO(&set);
          FD_SET(x11Fd,&set);

          if (now>=max) {
            return false;
          }
          tmp=max;
          tmp.Sub(now);

          tmp.GetTimeval(timeout);

          ret=select(x11Fd+1,&set,NULL,NULL,&timeout);

          if (ret>0) {
            if (XCheckTypedWindowEvent(display,appWindow,eventType,&event)==True) {
              return true;
            }
          }
        }

      }

      void Display::SignalEventLoop()
      {
        write(signalPipe[1],"+",1);
      }

      void Display::EventLoop()
      {
        assert(exit);

        exit=false;

        HandleAsyncActions();
        HandleEventLoopActions();
        CheckTimers();

        while (true) {
          if (XEventsQueued(display,QueuedAlready)==0 && XEventsQueued(display,QueuedAfterFlush)==0) {
            Wait();
          }

          if (XEventsQueued(display,QueuedAfterReading)>0) {
            GetEvent();
          }

          HandleAsyncActions();
          HandleEventLoopActions();
          CheckTimers();

          if (exit) {
            break;
          }
        }
      }

      bool Display::GetMousePos(int& x, int& y) const
      {
        ::Window     root;
        ::Window     child;
        unsigned int bmask;
        int          wx,wy;

        return XQueryPointer(display,
                             XRootWindow(display,scrNum),&root,&child,
                             &x,&y,&wx,&wy,&bmask)!=False;
      }


#if defined(HAVE_LIBSM)
      void Display::CallbackDie(SmcConn smc_conn, SmPointer client_data)
      {
        std::cout << "CallbackDie" << std::endl;
      }

      void Display::CallbackSaveYourself(SmcConn smc_conn, SmPointer client_data,
                                         int save_style, Bool shutdown,
                                         int interact_style,
                                         Bool fast)
      {
        struct {
          SmPropValue program[1];
          SmPropValue user[1];
          SmPropValue hint[1];
          SmPropValue restart[1];
        } vals;

        SmProp prop[] = {
          { const_cast<char*>(SmProgram),          const_cast<char*>(SmLISTofARRAY8), 1, vals.program},
          { const_cast<char*>(SmUserID),           const_cast<char*>(SmLISTofARRAY8), 1, vals.user   },
          { const_cast<char*>(SmRestartStyleHint), const_cast<char*>(SmCARD8),        1, vals.hint   },
          { const_cast<char*>(SmCloneCommand),     const_cast<char*>(SmLISTofARRAY8), 1, vals.restart},
          { const_cast<char*>(SmRestartCommand),   const_cast<char*>(SmLISTofARRAY8), 1, vals.restart}
        };

        SmProp *props[] = {
          &prop[0],
          &prop[1],
          &prop[2],
          &prop[3],
          &prop[4],
        };

        std::cout << "CallbackSaveYourself" << std::endl;

        vals.program->value=(void*)programName.c_str();
        vals.program->length=programName.length();

#if HAVE_PWD_H
        struct passwd *pw = getpwuid(getuid());

        vals.user->value=(void*) (pw ? pw->pw_name : "");
        vals.user->length=strlen((const char*)vals.user->value);
#endif

        char restartStyle=SmRestartIfRunning;

        vals.hint->value=&restartStyle;
        vals.hint->length=1;

        vals.restart->value=(void*)programName.c_str();
        vals.restart->length=programName.length();

        SmcSetProperties(smc_conn,sizeof(props)/sizeof(SmProp*),props);

        SmcSaveYourselfDone(smc_conn,True);
      }

      void Display::CallbackShutdownCancelled(SmcConn smc_conn, SmPointer client_data)
      {
        std::cout << "ShutdownCanceled" << std::endl;
      }

      void Display::CallbackSaveComplete(SmcConn smc_conn, SmPointer client_data)
      {
        std::cout << "SaveComplete" << std::endl;
      }

      void Display::CallbackICEIOError(IceConn connection)
      {
      }
#endif
    }
  }
}
