/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/FuelGauge.h>

#include <math.h>

#include <Lum/Base/String.h>

namespace Lum {

  static FuelGauge::Prefs *prefs=new ::Lum::FuelGauge::Prefs();

  FuelGauge::Prefs::Prefs()
  : Control::Prefs(L"FuelGauge")
  {
    // no code
  }

  void FuelGauge::Prefs::Initialize()
  {
    Control::Prefs::Initialize();

    hFrame=OS::display->GetFrame(OS::Display::hprogressFrameIndex);
    hBackground=OS::display->GetFill(OS::Display::hprogressBackgroundFillIndex);
    hFill=OS::display->GetFill(OS::Display::hprogressFillIndex);

    vFrame=OS::display->GetFrame(OS::Display::vprogressFrameIndex);
    vBackground=OS::display->GetFill(OS::Display::vprogressBackgroundFillIndex);
    vFill=OS::display->GetFill(OS::Display::vprogressFillIndex);
  }

  FuelGauge::FuelGauge()
  : font(OS::display->GetFont())
  {
    SetPrefs(::Lum::prefs);
  }

  bool FuelGauge::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Double*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void FuelGauge::CalcSize()
  {
    SetBackground(dynamic_cast<Prefs*>(prefs)->hBackground);
    SetFrame(dynamic_cast<Prefs*>(prefs)->hFrame);

    minWidth=2*OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
             std::max(font->height,(size_t)font->StringWidth(L"m",OS::Font::normal));
    minHeight=2*OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    width=minWidth;
    height=minHeight;

    Control::CalcSize();
  }

  void FuelGauge::Layout()
  {
    if (width>=height) {
      SetBackground(dynamic_cast<Prefs*>(prefs)->hBackground);
      SetFrame(dynamic_cast<Prefs*>(prefs)->hFrame);
    }
    else {
      SetBackground(dynamic_cast<Prefs*>(prefs)->vBackground);
      SetFrame(dynamic_cast<Prefs*>(prefs)->vFrame);
    }

    Control::Layout();
  }

  void FuelGauge::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    DrawBackground(x,y,w,h);

    if (model.Valid() && !model->IsNull()) {
      double percent=(model->Get()-model->GetMin())/(model->GetMax()-model->GetMin());

      if (width>=height) {
        Lum::OS::FontExtent extent;
        std::wstring        label(Base::NumberToWString((size_t)(percent*100))+L"%");

        dynamic_cast<Prefs*>(prefs)->hFill->Draw(draw,
                                                 this->x,this->y,
                                                 width,height,
                                                 this->x,this->y,
                                                 (size_t)(width*percent),height);


        font->StringExtent(label,extent);

        draw->PushForeground(OS::Display::textColor);
        draw->PushFont(font,0);
        draw->DrawString(this->x+(width-extent.width+extent.left+extent.right)/2,
                         this->y+(height-font->height)/2+font->ascent,
                         label);
        draw->PopFont();
        draw->PopForeground();

      }
      else {
        dynamic_cast<Prefs*>(prefs)->vFill->Draw(draw,
                                                 this->x,this->y,
                                                 width,height,
                                                 this->x,this->y+height-(size_t)(height*percent),
                                                 width,(size_t)(height*percent));
      }
    }
  }

  void FuelGauge::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model) {
      Redraw();
    }
    Control::Resync(model,msg);
  }
}
