#ifndef LUM_MODEL_TABLE_H
#define LUM_MODEL_TABLE_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <list>
#include <vector>

#include <Lum/Base/Object.h>

#include <Lum/Object.h>

namespace Lum {
  namespace Model {

    /**
      Abstract baseclass for all table models. Its purpose is,
      to define an common interface for all tablemodels. The
      table object itself will only talk with the model through
      this interface.
    */
    class LUMAPI Table : public Base::Model
    {
    public:
      class LUMAPI RefreshCell : public Base::ResyncMsg
      {
      public:
        int column;
        int row;
      };

      class LUMAPI RefreshRow : public Base::ResyncMsg
      {
      public:
        int row;
      };

      class LUMAPI InsertRow : public Base::ResyncMsg
      {
      public:
        int    row;
        size_t count;
      };

      class LUMAPI DeleteRow : public Base::ResyncMsg
      {
      public:
        int    row;
        size_t count;
      };

    public:
      Table();
      ~Table();

      void RedrawRow(size_t row);
      void RedrawCell(size_t column, size_t row);
      void NotifyInsert(size_t row, size_t count);
      void NotifyDelete(size_t row, size_t count);

      virtual size_t GetRows() const = 0;
      virtual size_t GetColumns() const = 0;

      virtual std::wstring GetString(size_t column, size_t row) const = 0;
      virtual Lum::Object* GetObject(size_t column, size_t row) const = 0;
    };

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Table>;
#endif

    typedef Base::Reference<Table> TableRef;

    class LUMAPI ASTable : public Table
    {
    private:
      std::vector<std::vector<std::wstring> > data;

    public:
      ASTable();

      size_t GetRows() const;
      size_t GetColumns() const;

      std::wstring GetString(size_t column, size_t row) const;
      Lum::Object* GetObject(size_t column, size_t row) const;

      void SetSize(size_t column, size_t row);

      void SetString(size_t column, size_t row, const std::wstring& label);
    };

    typedef Base::Reference<ASTable> ASTableRef;

    class LUMAPI ListTable : public Table
    {
    public:
      class LUMAPI Entry
      {
      protected:
        ListTable *table;

      public:
        Entry(ListTable* table);
        virtual ~Entry();

        virtual std::wstring GetString(size_t column) const;
        virtual Lum::Object* GetObject(size_t column) const;
        virtual bool IsGreater(const Entry* other, size_t column) const;
      };

      class LUMAPI StdEntry : public Entry
      {
      private:
        class Cell
        {
        public:
          std::wstring string;
          Lum::Object  *object;

          Cell();
        };

      private:
        std::vector<Cell> row;

      public:
        StdEntry(ListTable* table);
        virtual ~StdEntry();

        std::wstring GetString(size_t column) const;
        Lum::Object* GetObject(size_t column) const;
        void SetString(size_t column, const std::wstring& string);
        void SetObject(size_t column, Lum::Object* object);
      };

    private:
      mutable std::list<Entry*>           list;
      mutable std::list<Entry*>::iterator current;
      mutable size_t                      currentPos;
      size_t                              columns;

    public:
      ListTable(size_t columns);
      virtual ~ListTable();

      size_t GetRows() const;
      size_t GetColumns() const;

      std::wstring GetString(size_t column, size_t row) const;
      Lum::Object* GetObject(size_t column, size_t row) const;

      Entry* GetEntry(size_t row) const;
      size_t GetRow(Entry* entry) const;
      void Sort(size_t column, bool down=true);
      // Insert
      // SwapWithNext
      void Delete(size_t row);
      void Prepend(Entry* entry);
      void Append(Entry* entry);
      void AppendString(const std::wstring& string);
      void Clear();
    };

    typedef Base::Reference<ListTable> ListTableRef;

    template<class N>
    class LUMAPI StdListTable : public Table
    {
    public:
      typedef typename std::list<N>           List;
      typedef typename std::list<N>::iterator Iterator;

    public:
      class LUMAPI DataProvider
      {
      public:
        virtual std::wstring GetString(const Iterator& iter, size_t column) const
        {
          return L"";
        }

        virtual Lum::Object* GetObject(const Iterator& iter, size_t column) const
        {
          return NULL;
        }
      };

    private:
      /*
      class ListCompare
      {
      private:
        size_t column;
        bool   down;

      public:
        ListCompare(size_t column, bool down)
        : column(column),down(down)
        {
          // no code
        }

        bool operator()(const ListTable::Entry* a, const ListTable::Entry* b)
        {
          if (down) {
            return !a->IsGreater(b,column);
          }
          else {
            return a->IsGreater(b,column);
          }
        }
      };*/

    private:
      mutable List     &list;
      mutable Iterator current;
      mutable size_t   currentPos;
      size_t           columns;
      DataProvider     *dataProvider;

    private:
      void GotoEntry(size_t row) const
      {
        assert(row>=1 && row<=list.size());

        if (current==list.end()) {
          current=list.begin();
          currentPos=1;
        }

        if (currentPos<row) {
          while (currentPos<row) {
            ++current;
            ++currentPos;
          }
        }
        else if (currentPos>row) {
          while (currentPos>row) {
            --current;
            --currentPos;
          }
        }
      }

    public:
      StdListTable(List& list, DataProvider* dataProvider, size_t columns)
       : list(list),
         current(list.end()),
         currentPos(0),
         columns(columns),
         dataProvider(dataProvider)
      {
        // no code
      }

      ~StdListTable()
      {
        delete dataProvider;
      }

      size_t GetRows() const
      {
        return list.size();
      }

      size_t GetColumns() const
      {
        return columns;
      }

      std::wstring GetString(size_t column, size_t row) const
      {
        GotoEntry(row);
        return dataProvider->GetString(current,column);
      }

      Lum::Object* GetObject(size_t column, size_t row) const
      {
        GotoEntry(row);
        return dataProvider->GetObject(current,column);
      }

      //size_t GetRow(Entry* entry) const;
      /*
      void Sort(size_t column, bool down=true)
      {
        list.sort(ListCompare(column,down));

        current=list.end();
        currentPos=0;

        Notify();
      }*/

      // Insert
      // SwapWithNext
      N Delete(size_t row)
      {
        GotoEntry(row);

        assert(current!=list.end());

        list.erase(current);

        current=list.end();
        currentPos=0;

        NotifyDelete(row,1);
      }

      void Prepend(const N& entry)
      {
        list.push_front(entry);
        currentPos++;
        NotifyInsert(1,1);
      }

      void Append(const N& entry)
      {
        list.push_back(entry);

        NotifyInsert(list.size(),1);
      }

      void Clear()
      {
        current=list.end();
        currentPos=0;

        list.clear();

        Notify();
      }
    };

    /*
    template<class N>
    typedef Base::Reference<StdListTable<N> > StdListTableRef<N>;
      */
  }
}

#endif
