#include <Lum/Dlg/File.h>

#include <Lum/Private/Config.h>

#if defined(HAVE_UNISTD_H)
  #include <unistd.h>
#endif

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/Util.h>

#include <Lum/Model/Selection.h>
#include <Lum/Model/Table.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Combo.h>
#include <Lum/DirSelect.h>
#include <Lum/Panel.h>
#include <Lum/PopupGroup.h>
#include <Lum/Space.h>
#include <Lum/WindowGroup.h>

namespace Lum {
  namespace Dlg {
    class RootsEntry : public Lum::Model::ListTable::Entry
    {
    public:
      std::wstring name;
      std::wstring path;

    public:
      RootsEntry(Lum::Model::ListTable* table,
                 const std::wstring& name,
                 const std::wstring& path)
      : Lum::Model::ListTable::Entry(table),name(name),path(path)
      {
        // no code
      }

      std::wstring GetString(size_t /*column*/) const
      {
        return name;
      }
    };

    class Locations : public Dialog
    {
    public:
      std::wstring                  result;
      Table                         *table;
      Model::SingleLineSelectionRef selection;
      Model::ListTableRef           roots;

    public:
      Locations()
      : table(new Table()),
        selection(new Model::SingleLineSelection),
        roots(new Model::ListTable(1))
      {
        AttachModel(table->GetTableView()->GetMouseSelectionAction());
      }

      ~Locations()
      {
        UnattachModel(table->GetTableView()->GetMouseSelectionAction());
      }

      void PreInit()
      {
        PopupGroup *container;

        std::vector<Base::Path::PathDescription> list;

        Base::Path::GetStdPathes(list);

        Model::HeaderRef headerModel=new Model::HeaderImpl();
        headerModel->AddColumn(L"",Lum::Base::Size::modePixel,0);

        for (size_t x=0; x<list.size(); x++) {
          roots->Append(new RootsEntry(roots,list[x].description,list[x].path));
        }

        container=new PopupGroup();
        //container->SetFlex(true,true);

        table->SetFlex(true,true);
        table->SetMinHeight(Base::Size::stdCharHeight,1);
        table->SetMaxWidth(Base::Size::screenVRel,40);
        table->SetMaxHeight(Base::Size::screenVRel,40);
        table->RequestFocus();
        table->SetHeaderModel(headerModel);
        table->SetSelection(selection);
        table->SetModel(roots);
        table->GetTableView()->SetAutoFitColumns(true);
        table->GetTableView()->SetAutoHSize(true);
        table->GetTableView()->SetAutoVSize(true);

        container->SetMain(table);

        SetTop(container);

        Dialog::PreInit();
      }

      void Resync(Base::Model* model, const Base::ResyncMsg& msg)
      {
        if (model==table->GetTableView()->GetMouseSelectionAction() && table->GetTableView()->GetMouseSelectionAction()->IsFinished()) {
          if (selection->HasSelection()) {
            result=dynamic_cast<RootsEntry*>(roots->GetEntry(selection->GetLine()))->path;
          }
          Exit();
        }

        Dialog::Resync(model,msg);
      }
    };

    File::Options::Options()
    : existingOnly(false),mode(modeFile),defaultFilter(0)
    {
      // no code
    }

    File::Options::~Options()
    {
      for (size_t x=0; x<filters.size(); ++x) {
        delete filters[x];
      }
    }

    void File::Options::SetMode(Mode mode)
    {
      this->mode=mode;
    }

    void File::Options::SetExistingOnly(bool exists)
    {
      existingOnly=exists;
    }

    void File::Options::AddFilter(Model::Dir::Filter* filter)
    {
      filters.push_back(filter);
    }

    void File::Options::SetDefaultFilter(size_t pos)
    {
      defaultFilter=pos;
    }

    void File::Options::SetDefaultFilter(const std::wstring& name)
    {
      for (defaultFilter=0; defaultFilter<filters.size(); ++defaultFilter) {
        if (filters[defaultFilter]->GetName()==name) {
          return;
        }
      }

      defaultFilter=0;
    }

    class FileListFormatProvider : public TableView::FormatProvider
    {
    public:
      TableView::Alignment GetAlignment(size_t column, size_t /*row*/) const
      {
        switch (column) {
        case 1:
          return TableView::left;
        case 2:
          return TableView::right;
        case 3:
          return TableView::right;
        default:
          return TableView::left;
        }
      }
    };

    File::File()
    : dirList(new Model::Dir()),
      path(new Model::Path()),
      full(new Model::String()),
      filter(new Model::SizeT()),
      okAction(new Model::Action()),
      reloadAction(new Model::Action()),
      parentAction(new Model::Action()),
      fullAction(new Model::Action()),
      selectionAction(new Model::Action()),
      doubleClickAction(new Model::Action()),
      locationsAction(new Model::Action()),
      table(NULL),
      result(false)
    {
      Base::Path tmp;

      tmp.SetNativeDir(Base::Path::GetCWD());

      path->Off();

      path->Set(tmp);
      full->Set(Base::Path::GetCWD());

      AttachModel(GetOpenedAction());
      AttachModel(GetClosedAction());
      AttachModel(path);
      AttachModel(full);
      AttachModel(filter);
      AttachModel(okAction);
      AttachModel(reloadAction);
      AttachModel(parentAction);
      AttachModel(fullAction);
      AttachModel(selectionAction);
      AttachModel(doubleClickAction);
      AttachModel(locationsAction);
    }

    File::~File()
    {
      UnattachModel(locationsAction);
      UnattachModel(doubleClickAction);
      UnattachModel(selectionAction);
      UnattachModel(fullAction);
      UnattachModel(parentAction);
      UnattachModel(reloadAction);
      UnattachModel(okAction);
      UnattachModel(filter);
      UnattachModel(full);
      UnattachModel(path);
      UnattachModel(GetClosedAction());
      AttachModel(GetOpenedAction());
    }

    void File::SetOptions(Options* options)
    {
      this->options=options;

      if (options->filters.size()>0) {
        filter->Set(options->defaultFilter+1);
      }
      else {
        filter->SetNull();
      }
    }

    bool File::GetResult() const
    {
      return result;
    }

    void File::PreInit()
    {
      Button           *button;
      ButtonRow        *row;
      DirSelect        *dirSel;
      Panel            *hPanel,*vPanel;
      Space            *space;
      WindowGroup      *wGroup;
      Model::HeaderRef header;

      vPanel=new VPanel();
      vPanel->SetFlex(true,true);

      dirSel=new DirSelect();
      dirSel->SetFlex(true,false);
      dirSel->SetModel(path);
      vPanel->Add(dirSel);

      vPanel->Add(new VSpace());

      header=new Model::HeaderImpl();
      header->AddColumn(_l(L"MODEL_DIR_FILENAME",L"Filename"),Base::Size::stdCharWidth,15,true);
      header->AddColumn(_l(L"MODEL_DIR_SIZE",L"Size"),Base::Size::stdCharWidth,9);
      header->AddColumn(_l(L"MODEL_DIR_DATE",L"Date"),Base::Size::stdCharWidth,10);

      table=new Table();
      table->SetFlex(true,true);
      table->SetMinWidth(Base::Size::stdCharWidth,40);
      table->RequestFocus();
      table->SetShowHeader(true);
      table->GetTableView()->SetAutoFitColumns(true);
      table->GetTableView()->SetAutoHSize(true);
      table->SetSelection(new Model::SingleLineSelection);
      table->GetTableView()->SetSelectionAction(selectionAction);
      table->SetDoubleClickAction(doubleClickAction);
      table->SetFormatProvider(new FileListFormatProvider());
      table->SetModel(dirList);
      table->SetHeaderModel(header);

      vPanel->Add(table);

      vPanel->Add(new VSpace());

      hPanel=new HPanel();
      hPanel->SetFlex(true,false);

      locations=new Lum::Button();
      locations->SetFlex(false,true);
      locations->RequestFocus();
      locations->SetModel(locationsAction);
      locations->SetText(_l(L"DLG_FILE_LOCATIONS",L"_Locations"));
      hPanel->Add(locations);

      space=new HSpace();
      space->SetFlex(true,false);
      hPanel->Add(space);

      if (options->filters.size()>0) {
        Combo            *combo;
        Model::ListTable *table;

        table=new Model::ListTable(1);
        for (size_t x=0; x<options->filters.size(); ++x) {
          table->AppendString(options->filters[x]->GetName());
        }

        combo=new IndexCombo();
        combo->SetWidth(Base::Size::stdCharWidth,15);
        combo->RequestFocus();
        combo->SetModel(filter);
        combo->SetTableModel(table);
        hPanel->Add(combo);
      }

      vPanel->Add(hPanel);
      vPanel->Add(new VSpace());

      hPanel=new HPanel();
      hPanel->SetFlex(true,false);

      fullStr=new String();
      fullStr->SetFlex(true,false);
      fullStr->RequestFocus();
      fullStr->SetModel(full);
      fullStr->SetReturnAction(fullAction);
      hPanel->Add(fullStr);

      button=new Button();
      button->SetFlex(false,true);
      button->RequestFocus();
      button->SetModel(reloadAction);
      button->SetText(_l(L"DLG_FILE_RELOAD",L"_Reload"));
      hPanel->Add(button);

      vPanel->Add(hPanel);
      vPanel->Add(new VSpace());

      row=new ButtonRow();
      row->SetFlex(true,false);

      button=new Button;
      button->SetFlex(true,true);
      button->RequestFocus();
      button->SetModel(okAction);
      button->SetText(_ld(dlgButtonOk));
      button->SetType(Button::typeCommit);
      row->Add(button);

      button=new Button;
      button->SetFlex(true,true);
      button->RequestFocus();
      button->SetModel(GetClosedAction());
      button->SetText(_ld(dlgButtonCancel));
      button->SetType(Button::typeCancel);
      row->Add(button);
      vPanel->Add(row);

      wGroup=new WindowGroup();
      wGroup->SetFlex(true,true);
      wGroup->SetMinHeight(Base::Size::screenVRel,40);

      if (OS::display->GetSize()<OS::Display::sizeNormal) {
        wGroup->SetMinWidth(Base::Size::screenVRel,80);
      }


      wGroup->SetMain(vPanel);

      SetTop(wGroup);

      RegisterCommitShortcut(table->GetTableView(),okAction);
      RegisterShortcut(table->GetTableView(),0,L"BackSpace",parentAction);

      Dialog::PreInit();
    }

    void File::SetPath(const std::wstring& path)
    {
      if (options->existingOnly && !Base::Path::Exists(path)) {
        return;
      }

      Base::Path newPath;

      if (Base::Path::IsDir(path)) {
        std::wstring dir;

        if (Base::Path::GetAbsoluteFilename(path,dir)) {
          newPath.SetNativeDir(dir);
        }
      }
      else {
        std::wstring file;

        if (Base::Path::GetAbsoluteFilename(path,file)) {
          newPath.SetNativePath(file);
        }
      }

      this->path->Set(newPath);
    }

    void File::Load()
    {
      dirList->Off();
      table->GetTableView()->GetSelection()->Clear();
      dirList->SetDirectory(path->Get().GetDir(),
                            false,
                            options->mode==Options::modeDirectory);

      table->GetTableView()->GetVAdjustment()->SetTop(1);
      dirList->On();

      if (dirList->GetRows()>0) {
        dynamic_cast<Model::SingleLineSelection*>(table->GetTableView()->GetSelection())->SelectLine(1);
      }
    }

    void File::GotoParent()
    {
      if (path->Get().IsRoot()) {
        return;
      }

      Base::Path newPath(path->Get());

      newPath.GoUp();

      path->Set(newPath);
    }

    void File::GotoChildDirectory(const std::wstring& dir)
    {
      Base::Path newPath(path->Get());

      newPath.AppendDir(dir);

      path->Set(newPath);
    }

    bool File::GetSelection(Model::Dir::Entry& entry) const
    {
      size_t line;

      if (table->GetTableView()->GetSelection()->HasSelection()) {
        line=dynamic_cast<Model::SingleLineSelection*>(table->GetTableView()->GetSelection())->GetLine();

        dirList->GetEntry(line,entry);
        return true;
      }
      else {
        return false;
      }
    }

    /**
      This callback is called, if the selection is the file/dir box changes.
    */
    void File::OnSelectionChange()
    {
      Model::Dir::Entry entry;

      if (!GetSelection(entry)) {
        return;
      }

      Base::Path newPath(path->Get());

      if (options->mode==Options::modeDirectory) {
        if (entry.type==Model::Dir::dirType) {
          newPath.AppendDir(entry.name);
          full->Set(newPath.GetDir(true));
        }
      }
      else {
        if (entry.type==Model::Dir::dirType) {
          full->Set(newPath.GetDir(true));
        }
        else {
          newPath.SetBaseName(entry.name);
          full->Set(newPath.GetPath());
        }
      }
    }

    /**
      This callback is called, if "return" gets triggered while editing
      (focusing) the edit field below the file/dir box.

      It uses this->full->Get() as source for actions.
    */
    void File::HandleChange()
    {
      std::wstring full;

      full=this->full->Get();

      if (full.empty()) {
        // Someone has completely deleted string entry field
        this->full->Set(path->Get().GetPath());
        return;
      }

      if (!Base::Path::Exists(full) && options->existingOnly) {
        this->full->Set(path->Get().GetPath());
        return;
      }

      Base::Path newPath(full);

      if (options->mode==Options::modeDirectory) {
        if (newPath.IsDir() || Base::Path::IsDir(full)) {
          result=true;
          Exit();
        }
        else {
          newPath.SetBaseName(L"");
          path->Set(newPath);
        }
      }
      else {
        if (newPath.IsDir() || Base::Path::IsDir(full)) {
          newPath.SetNativeDir(full);
          path->Set(newPath);
        }
        else {
          result=true;
          Exit();
        }
      }
    }

    /**
      This callback is called, if the OK button is clicked or selected by
      pressing the key shortcut "return".

      It uses this->path->Get() together with the current selection
      as source for actions.
    */
    void File::OnOk()
    {
      Model::Dir::Entry entry;

      if (!GetSelection(entry)) {
        return;
      }

      Base::Path newPath(full->Get());

      if (options->mode==Options::modeDirectory) {
        if (entry.type==Model::Dir::dirType) {
          if (newPath.Exists() && newPath.IsDir()) {
            result=true;
            Exit();
          }
        }
      }
      else {
        if (entry.type==Model::Dir::dirType) {
          newPath.AppendDir(entry.name);
          path->Set(newPath);
        }
        else {
          if ((options->existingOnly && newPath.Exists() && !newPath.IsDir()) ||
              !options->existingOnly) {
            result=true;
            Exit();
          }
        }
      }
    }

    void File::OnDoubleClick()
    {
      Model::Dir::Entry entry;

      if (GetSelection(entry)) {
        if (entry.type==Model::Dir::dirType) {
          GotoChildDirectory(entry.name);
        }
        else {
          OnOk();
        }
      }
    }

    void File::Resync(Base::Model* model, const Base::ResyncMsg& msg)
    {
      if (model==path) {
        full->Set(path->Get().GetPath());
        Load();
      }
      else if (model==okAction  && okAction->IsFinished()) {
        OnOk();
      }
      else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
        result=false;
        Exit();
      }
      else if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
        path->On();
        SetFocus(table->GetTableView());
      }
      else if (model==reloadAction && reloadAction->IsFinished()) {
        Load();
      }
      else if (model==parentAction && parentAction->IsFinished()) {
        GotoParent();
      }
      else if (model==fullAction && fullAction->IsFinished()) {
        HandleChange();
      }
      else if (model==selectionAction && selectionAction->IsFinished()) {
        OnSelectionChange();
      }
      else if (model==doubleClickAction && doubleClickAction->IsFinished()) {
        OnDoubleClick();
      }
      else if (model==locationsAction && locationsAction->IsFinished()) {
        Locations *popup;

        popup=new Locations();
        popup->SetType(OS::Window::typePopup);
        popup->SetParent(GetWindow());
        popup->SetReference(locations);

        if (popup->Open()) {
          popup->EventLoop();
          popup->Close();
        }

        if (!popup->result.empty()) {
          Base::Path tmp;

          tmp.SetNativeDir(popup->result);
          path->Set(tmp);
        }

        delete popup;
       }
      else if (model==filter && dirList.Valid() && options.Valid()) {
        dirList->SetFilter(options->filters[filter->Get()-1]);
        if (table!=NULL) {
          Load();
        }
      }

      Dialog::Resync(model,msg);
    }

    std::wstring File::GetPath() const
    {
      return full->Get();
    }

    bool File::GetFile(OS::Window* parent,
                       const std::wstring& title,
                       std::wstring& file,
                       Options* options)
    {
      File *dialog;
      bool result=false;

      if (options==NULL) {
        options=new Options();
      }
      options->SetMode(Options::modeFile);
      options->SetExistingOnly(true);

      dialog=new File();
      dialog->SetParent(parent);
      dialog->SetTitle(title);
      dialog->SetOptions(options);
      dialog->SetPath(file);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();

        if (dialog->GetResult()) {
          file=dialog->GetPath();
          result=true;
        }
      }

      delete dialog;

      return result;
    }

    bool File::SaveFile(OS::Window* parent,
                        const std::wstring& title,
                        std::wstring& file,
                        Options* options)
    {
      File *dialog;
      bool result=false;

      if (options==NULL) {
        options=new Options();
      }
      options->SetMode(Options::modeFile);
      options->SetExistingOnly(false);

      dialog=new File();
      dialog->SetParent(parent);
      dialog->SetTitle(title);
      dialog->SetOptions(options);
      dialog->SetPath(file);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();

        if (dialog->GetResult()) {
          file=dialog->GetPath();
          result=true;
        }
      }

      delete dialog;

      return result;
    }

    bool File::GetDir(OS::Window* parent,
                      const std::wstring& title,
                      std::wstring& dir,
                      Options* options)
    {
      File *dialog;
      bool result=false;

      if (options==NULL) {
        options=new Options();
      }
      options->SetMode(Options::modeDirectory);
      options->SetExistingOnly(true);

      dialog=new File();
      dialog->SetParent(parent);
      dialog->SetTitle(title);
      dialog->SetOptions(options);
      dialog->SetPath(dir);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();

        if (dialog->GetResult()) {
          dir=dialog->GetPath();
          result=true;
        }
      }

      delete dialog;

      return result;
    }
  }
}
