#include <Lum/Button.h>

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/DateTime.h>

#include <Lum/Dialog.h>
#include <Lum/Text.h>

namespace Lum {

  static Button::Prefs *prefs=new Button::Prefs();

  static size_t pulseSec  = 0;      // Make this a global preference value
  static size_t pulseMSec = 100000;

  Button::Prefs::Prefs()
  : Object::Prefs(L"Button")
  {
    // no code
  }

  void Button::Prefs::Initialize()
  {
    Object::Prefs::Initialize();

    frame=OS::display->GetFrame(OS::Display::buttonFrameIndex);
    background=OS::display->GetFill(OS::Display::buttonBackgroundFillIndex);

    returnFrame=OS::display->GetFrame(OS::Display::positiveButtonFrameIndex);
    bgReturn=OS::display->GetFill(OS::Display::positiveButtonBackgroundFillIndex);

    escapeFrame=OS::display->GetFrame(OS::Display::negativeButtonFrameIndex);
    bgEscape=OS::display->GetFill(OS::Display::negativeButtonBackgroundFillIndex);

    defaultFrame=OS::display->GetFrame(OS::Display::defaultButtonFrameIndex);
    bgDefault=OS::display->GetFill(OS::Display::defaultButtonBackgroundFillIndex);

    toolbarFrame=OS::display->GetFrame(OS::Display::toolbarButtonFrameIndex);
    bgToolbar=OS::display->GetFill(OS::Display::toolbarButtonBackgroundFillIndex);

    scrollFrame=OS::display->GetFrame(OS::Display::scrollButtonFrameIndex);
    bgScroll=OS::display->GetFill(OS::Display::scrollButtonBackgroundFillIndex);

    iFrame=frame;

    sHSpace.SetSize(Base::Size::softUnitP,25);
    sVSpace.SetSize(Base::Size::softUnitP,25);
  }

  Button::Button()
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetPrefs(::Lum::prefs);

    SetCanFocus(true);
    SetCanDrawMouseActive(true);
  }

  Button::Button(Object* object)
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetPrefs(::Lum::prefs);

    SetCanFocus(true);
    SetCanDrawMouseActive(true);

    SetLabel(object);
  }

  Button::Button(const std::wstring& text)
  : object(NULL),font(OS::display->GetFont()),type(typeNormal),
    pulse(false),shortCut('\0'),scAssigned(false)
  {
    SetPrefs(::Lum::prefs);

    SetCanFocus(true);
    SetCanDrawMouseActive(true);

    SetText(text);
  }

  Button::~Button()
  {
    delete object;
  }

  bool Button::HasBaseline() const
  {
    return object!=NULL && object->HasBaseline();
  }

  size_t Button::GetBaseline() const
  {
    assert(inited);

    if (object!=NULL && object->HasBaseline()) {
      // We also assume that is vertically flexible

      return frame->topBorder+object->GetBaseline()+OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
    }
    else {
      return 0;
    }
  }

  /**
    Set a new font to be used by the button gadget.
  */
  void Button::SetFont(OS::Font *font)
  {
    this->font=font;
  }

  /**
    Use this method if you do not want text displayed in the button but
    want to use some other Object as label
  */
  void Button::SetLabel(Object* label)
  {
    delete this->object;

    this->object=label;
    this->object->SetParent(this);
  }

  /**
    Assign an additional image. Some buttons type will automatically assign an image.
  */
  void Button::SetImage(OS::ImageRef& image)
  {
    this->image=image;
  }

  /**
    Call this method if you want the given text to be displayed in
    the button.

    This creates an instance of Text using the given text
    and sets it as button label.

    use '_' to mark the next character as key shortcut.
  */
  void Button::SetText(const std::wstring& string)
  {
    Text   *text;
    size_t pos;

    text=new Text();
    text->SetParent(this);
    text->SetFlex(true,true);
    text->SetAlignment(Text::centered);

    pos=string.find('_');
    if (pos!=std::wstring::npos && pos+1<string.length()) {
      shortCut=string[pos+1];
      text->AddText(string.substr(0,pos));
      text->AddText(string.substr(pos+1,1),OS::Font::underlined);
      text->AddText(string.substr(pos+2));
    }
    else {
      text->AddText(string);
    }

    delete object;

    object=text;
  }

  /**
    We can define special types of buttons.

    \see Type.
  */
  void Button::SetType(Type type)
  {
    this->type=type;
  }

  /**
    Is pulsemode is true, the button sends permanent events on
    mouse button down and none on the final button up.

    This is usefull for buttons in a scroller or similar.
  */
  void Button::SetPulse(bool pulse)
  {
    this->pulse=pulse;
  }

  bool Button::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Action*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void Button::CalcSize()
  {
    /* Let the frame calculate its size */
    switch (type) {
      case typeNormal:
      SetFrame(prefs->frame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->background);
      break;
    case typeDefault:
      SetFrame(dynamic_cast<Prefs*>(prefs)->defaultFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->bgDefault);
      image=OS::display->GetImage(OS::Display::defaultImageIndex);
      break;
    case typeCommit:
      SetFrame(dynamic_cast<Prefs*>(prefs)->returnFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->bgReturn);
      image=OS::display->GetImage(OS::Display::positiveImageIndex);
      break;
    case typeCancel:
      SetFrame(dynamic_cast<Prefs*>(prefs)->escapeFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->bgEscape);
      image=OS::display->GetImage(OS::Display::negativeImageIndex);
      break;
    case typeImage:
      SetFrame(dynamic_cast<Prefs*>(prefs)->iFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->background);
      break;
    case typeToolBar:
      SetFrame(dynamic_cast<Prefs*>(prefs)->toolbarFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->bgToolbar);
      break;
    case typeScroll:
      SetFrame(dynamic_cast<Prefs*>(prefs)->scrollFrame);
      SetBackground(dynamic_cast<Prefs*>(prefs)->bgScroll);
      break;
    }

    width=0;
    height=0;
    minWidth=0;
    minHeight=0;

    if (object!=NULL) {
      /*
        Now we let the image calculate its bounds and simply add its size
        to the size of the button.
      */
      object->CalcSize();
      width=object->GetOWidth();
      height=object->GetOHeight();
      minWidth=object->GetOMinWidth();
      minHeight=object->GetOMinHeight();
    }

    if (image.Valid()) {
      minWidth+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      width+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      minHeight=std::max(minHeight,image->GetHeight());
      height=std::max(height,image->GetHeight());
    }

    switch (type) {
    case typeImage:
      minWidth+=dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize()*2;
      width+=dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize()*2;
      minHeight+=dynamic_cast<Prefs*>(prefs)->sVSpace.GetSize()*2;
      height+=dynamic_cast<Prefs*>(prefs)->sVSpace.GetSize()*2;
      break;
    case typeScroll:
      // No space around object
      break;
    default:
      minWidth+=2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
      width+=2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
      minHeight+=2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
      height+=2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
      break;
    }

    maxWidth=30000;
    maxHeight=30000;

    /* We *must* call CalcSize of our superclass! */
    Control::CalcSize();
  }

  bool Button::HandleMouseEvent(const OS::MouseEvent& event)
  {
    /* It makes no sense to get the focus if we are currently not visible */
    if (!visible || !model.Valid() || !model->IsEnabled()) {
      return false;
    }

    /*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
    */

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      if (!model->IsStarted()) {
        /* We change our state to pressed and redisplay ourself */
        model->Start();

        if (pulse) {
          model->Trigger();
          OS::display->AddTimer(pulseSec,pulseMSec,model);
        }

        /*
          Since we want the focus for waiting for buttonup we return
          a pointer to ourself.
        */
        return true;
      }
    }
    else if (event.IsGrabEnd()) {
      if (model->IsStarted()) {
        /*
          If the users released the left mousebutton over our bounds we really
          got selected.
        */
        if (PointIsIn(event)) {
          if (pulse) {
            model->Cancel();
          }
          else {
            model->Finish();
          }
        }
        else {
          model->Cancel();
        }
      }

      // Clean up and remove possibly remaining timer event.
      if (pulse) {
        OS::display->RemoveTimer(model);
      }
    }
    else if (event.type==OS::MouseEvent::move && event.IsGrabed()) {
      if (PointIsIn(event)) {
        if (!model->IsStarted()) {
          model->Start();
        }
      }
      else {
        if (model->IsStarted()) {
          model->Cancel();
        }
      }

      return true;
    }

    return false;
  }

  bool Button::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (event.type==OS::KeyEvent::down) {
      if (event.key==OS::keySpace) {
        model->Start();
        Base::SystemTime::Sleep(1);
        model->Finish();

        return true;
      }
    }

    return false;
  }

  /*
     We tell the image to resize themself to
     our current bounds. Our bounds could have changed
     because Resize may have been called by some layout-objects
     between Button.CalcSize and Button.Draw.
   */
  void Button::Layout()
  {
    int    x,y;
    size_t width,height;

    x=this->x;
    y=this->y;
    width=this->width;
    height=this->height;

    switch (type) {
    case typeImage:
      x+=dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize();
      y+=dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize();
      width-=2*dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize();
      height-=2*dynamic_cast<Prefs*>(prefs)->sHSpace.GetSize();
      break;
    case typeScroll:
      break;
    default:
      x+=OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
      y+=OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
      width-=2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
      height-=2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
      break;
    }

    if (image.Valid()) {
      imageX=x;
      imageY=y+(height-image->GetHeight())/2;
      x+=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
      width-=image->GetWidth()+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    }

    if (object!=NULL) {
      object->Resize(width,height);
    }

    object->Move(x+(width-object->GetOWidth()) / 2,
                 y+(height-object->GetOHeight()) / 2);

    Control::Layout();
  }

  void Button::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    /*
      Set the correct draw mode before calling the baseclass,
      since the baseclass draw the object frame.
    */
    if (model.Valid() && model->IsEnabled()) {
      if (IsMouseActive() && !model->IsStarted()) {
        draw->activated=true;
      }
      if (model->IsStarted()) {
        draw->selected=true;
      }
    }
    else {
      draw->disabled=true;
    }

    draw->focused=HasFocus();

    Control::Draw(x,y,w,h); /* We must call Draw of our superclass */

    if (!Intersect(x,y,w,h)) {
      draw->activated=false;
      draw->selected=false;
      draw->disabled=false;
      draw->focused=false;
      return;
    }

    /*
      We fill the entier region with the background color, extracting
      the region that will be covered by the image object.
    */

    draw->PushClipBegin(this->x,this->y,width,height);
    if (object!=NULL) {
      draw->SubClipRegion(object->GetOX(),object->GetOY(),
                          object->GetOWidth(),object->GetOHeight());
    }
    if (image.Valid() && !image->GetAlpha()) {
      draw->SubClipRegion(imageX,imageY,
                          image->GetWidth(),image->GetHeight());
    }
    draw->PushClipEnd();
    DrawBackground(x,y,w,h);
    draw->PopClip();

    // DRaw object
    if (object!=NULL) {
      object->Draw(x,y,w,h);
    }

    // Draw optional image
    if (image.Valid()) {
      image->Draw(draw,imageX,imageY,image->GetWidth(),image->GetHeight());
    }

    draw->activated=false;
    draw->selected=false;
    draw->disabled=false;
    draw->focused=false;

    if (!scAssigned) {
      Dialog* window;

      window=dynamic_cast<Dialog*>(GetWindow()->GetMaster());
      if (window!=NULL) {
        if (shortCut!='\0') {
          window->RegisterShortcut(this,0,std::wstring(1,shortCut),model);
          window->RegisterShortcut(this,OS::qualifierAlt,std::wstring(1,shortCut),model);
        }

        switch (type) {
        case typeCommit:
          window->RegisterCommitShortcut(this,model);
          break;
        case typeCancel:
          window->RegisterCancelShortcut(this,model);
          break;
        case typeDefault:
          window->RegisterDefaultShortcut(this,model);
          break;
        default:
          break;
        }
      }
      scAssigned=true;
    }
 }

  void Button::Hide()
  {
    if (visible) {
      if (object!=NULL) {
        /* Hide the image */
        object->Hide();
      }
      /* hide the frame */
      Control::Hide();
    }
  }

  void Button::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model && visible) {
      if (this->model->IsInactive()) {
        Redraw();
      }
      else if (this->model->IsStarted()) {
        Redraw();
      }
      else if (this->model->IsFinished()) {
        if (pulse) {
          OS::display->RemoveTimer(this->model);
          OS::display->AddTimer(pulseSec,pulseMSec,this->model);
        }
      }
    }

    Control::Resync(model,msg);
  }
}
