/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/Size.h>

#include <assert.h>

#include <vector>

#include <Lum/OS/Display.h>

namespace Lum {
  namespace Base {

    static size_t              valueUnit=(size_t)-1;  //! Base value in pixel, calculated from the screen width, is always >0
    static size_t              valueSUnit=(size_t)-1; //! Base value in pixel, calculated from the screen width, may be 0

    /**
      Initialy calculates all relevant base values, which are then later used to calculate
      values for all given modes and values.
    */
    void CalculateUnits()
    {
      valueUnit=::Lum::OS::display->GetScreenWidth() / 300;
      valueSUnit=valueUnit;
      if (valueUnit==0) {
        valueUnit=1;
      }
    }

    /**
      Returns the calculated size in pixel for the given \p mode, \p value and \p font.
    */
    int GetSize(Size::Mode mode,
                size_t value,
                const ::Lum::OS::FontRef& font)
    {
      size_t             help;
      ::Lum::OS::FontRef f(font);

      if (!f.Valid()) {
        f=::Lum::OS::display->GetFont();
      }

      if (valueUnit==(size_t)-1) {
        CalculateUnits();
      }

      switch (mode) {
      case Size::unknown:
        assert(false);
        break;
      case Size::modePixel:
        return value;
      case Size::screenHRel:
        return (value*::Lum::OS::display->GetScreenWidth()) / 100;
      case Size::screenVRel:
        return (value*::Lum::OS::display->GetScreenWidth()) / 100;
      case Size::workHRel:
        return (value*::Lum::OS::display->GetWorkAreaWidth()) / 100;
      case Size::workVRel:
        return (value*::Lum::OS::display->GetWorkAreaHeight()) / 100;
      case Size::unit:
        return (value*valueUnit);
      case Size::unitP:
        help=(value*valueUnit) / 100;
        if (help==0) {
          return 1;
        }
        else {
          return help;
        }
      case Size::softUnit:
        return (value*valueSUnit);
      case Size::softUnitP:
        return (value*valueSUnit) / 100;
      case Size::stdCharWidth:
        return value*(f->StringWidth(L"M",0)+f->StringWidth(L"x",0)+f->StringWidth(L"E",0)) / 3;
      case Size::stdCharHeight:
        return value*f->height;
      }

      return 0;
    }

    Size::Size()
    : pixel(0),mode(unknown),size(0)
    {
      // no code
    }

    /**
      Set the abstract size using one of the predefined modes, a value based on definition of the
      mode and - for some modes - the index of one of the internal predefined fonts.
    */
    void Size::SetSize(Size::Mode mode,
                       size_t size,
                       const ::Lum::OS::FontRef& font)
    {
      this->mode=mode;
      this->size=size;
      this->font=font;
      this->pixel=(size_t)-1;
    }

    bool Size::IsSizeSet() const
    {
      return mode!=unknown;
    }

    /**
      Returns the calculated size in pixel for the earlier assigned mode, value and font.
    */
    size_t Size::GetSize() const
    {
      if (pixel==(size_t)-1) {
        pixel=::Lum::Base::GetSize(mode,size,font);
      }

      return pixel;
    }

    bool LoadSize(const std::wstring& name,
                  ::Lum::Config::Node *top,
                  Size& size)
    {
      ::Lum::Config::Node *modeNode;
      std::wstring        modeName;
      bool                found;
      Size::Mode          mode=Size::modePixel;

      top=top->GetSubNodeWithAttrValue(L"Size",L"name",name);
      if (top==NULL) {
        return false;
      }

      modeNode=top->GetChild(L"mode");
      if (modeNode==NULL) {
        return false;
      }

      modeName=modeNode->GetValue();

      found=true;
      if (modeName==L"pixel") {
        mode=Size::modePixel;
      }
      else if (modeName==L"screenHRel") {
        mode=Size::screenHRel;
      }
      else if (modeName==L"screenVRel") {
        mode=Size::screenVRel;
      }
      else if (modeName==L"workHRel") {
        mode=Size::workHRel;
      }
      else if (modeName==L"workVRel") {
        mode=Size::workVRel;
      }
      else if (modeName==L"unit") {
        mode=Size::unit;
      }
      else if (modeName==L"unitP") {
        mode=Size::unitP;
      }
      else if (modeName==L"softUnit") {
        mode=Size::softUnit;
      }
      else if (modeName==L"softUnitP") {
        mode=Size::softUnitP;
      }
      else if (modeName==L"stdCharWidth") {
        mode=Size::stdCharWidth;
      }
      else if (modeName==L"stdCharHeight") {
        mode=Size::stdCharHeight;
      }
      else {
        found=false;
      }

      if (found) {
        size_t value;

        if (top->GetAttribute(L"size",value)) {
          size.SetSize(mode,value);
          return true;
        }
      }

      return false;
    }
  }
}
