#include <Lum/OS/X11/Image.h>

#include <Lum/OS/X11/DrawInfo.h>

#include <iostream>

namespace Lum {
  namespace OS {
    namespace X11 {

      static ::Display    *display=NULL;
      static int          scrNum=0;
      static ::Visual     *visual=NULL;
      static int          depth=0;
      static ::Colormap   colorMap=0;
#if defined(HAVE_LIB_XFT2)
      static bool         render=false;
      ::XRenderPictFormat *format=NULL;
      ::XRenderPictFormat *alphaFormat=NULL;
#endif

      Image::Image()
      : pixmap(0),shape(0)
#if defined(HAVE_LIB_XFT2)
        ,picture(0)
#endif
      {
        // no code
      }

      Image::~Image()
      {
        FreeOSStructures();
      }

      void Image::CreateOSStructures()
      {
        char                       value;
        size_t                     bytes;
        ::XGCValues                values;
        ::GC                       gc;
        ::XImage                   *image;
        char                       *data;
#if defined(HAVE_LIB_XFT2)
        ::XRenderPictureAttributes attr;
#endif

        if (source==NULL) {
          return;
        }

#if defined(HAVE_LIB_XFT2)
        if (render) {
          ::Pixmap pixmap;

          data=new char[4*width*height];

          if (XImageByteOrder(display)==LSBFirst) {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
              /* Premultiplied alpha values */
              data[off]=(current[s].b*current[s].a) / 256;
              off++;
              data[off]=(current[s].g*current[s].a) / 256;
              off++;
              data[off]=(current[s].r*current[s].a) / 256;
              off++;
              data[off]=current[s].a;
              off++;
            }
          }
          else {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
              /* Premultiplied alpha values */
              data[off]=current[s].a;
              off++;
              data[off]=(current[s].r*current[s].a) / 256;
              off++;
              data[off]=(current[s].g*current[s].a) / 256;
              off++;
              data[off]=(current[s].b*current[s].a) / 256;
              off++;
            }
          }

          image=XCreateImage(display,
                             visual,
                             32,
                             ZPixmap,
                             0,
                             data,
                             width,
                             height,
                             32,
                             width*4);

          pixmap=XCreatePixmap(display,
                               XDefaultRootWindow(display),
                               width,
                               height,
                               32);

          gc=XCreateGC(display,pixmap,0,&values);

          XPutImage(display,
                    pixmap,
                    gc,
                    image,
                    0,
                    0,
                    0,
                    0,
                    width,
                    height);

          picture=XRenderCreatePicture(display,
                                       pixmap,
                                       alphaFormat,
                                       0,&attr);

          if (picture==0) {
            std::cerr << "Cannot create picture" << std::endl;
          }

          image->data=NULL;
          XDestroyImage(image);
          XFreeGC(display,gc);
          XFreePixmap(display,pixmap);
          pixmap=0;
          delete [] data;
          data=NULL;

          return;
        }
#endif

        if (depth==24 || depth==32) {
          bytes=4;
        }
        else if (depth==16) {
          bytes=2;
        }
        else if (depth==8) {
          bytes=1;
        }
        else {
          return;
        }

        data=new char[4*width*height];

        image=XCreateImage(display,
                           visual,
                           depth,
                           ZPixmap,
                           0,
                           data,
                           width,
                           height,
                           bytes*8,
                           0);
        if (image==NULL) {
          std::cerr << "Cannot allocate image!" << std::endl;
          return;
        }

        if (image->bits_per_pixel==32) {
          if (image->bitmap_bit_order==LSBFirst && image->byte_order==LSBFirst) {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
             data[off]=(current[s].b*current[s].a) / 256;
              ++off;
              data[off]=(current[s].g*current[s].a) / 256;
              ++off;
              data[off]=(current[s].r*current[s].a) / 256;
              ++off;
              data[off]=current[s].a;
              ++off;
            }
          }
          else if (image->bitmap_bit_order==MSBFirst && image->byte_order==MSBFirst) {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
              data[off]=current[s].a;
              ++off;
              data[off]=current[s].r;
              ++off;
              data[off]=current[s].g;
              ++off;
              data[off]=current[s].b;
              ++off;
            }
          }
          else if (image->bitmap_bit_order==LSBFirst && image->byte_order==MSBFirst) {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
              data[off]=current[s].a;
              ++off;
              data[off]=current[s].r;
              ++off;
              data[off]=current[s].g;
              ++off;
              data[off]=current[s].b;
              ++off;
            }
          }
          else {
            return;
          }
        }
        else if (image->bits_per_pixel==16) {
          if (image->bitmap_bit_order==LSBFirst && image->byte_order==LSBFirst) {
            size_t off=0;
            for (size_t s=0; s<width*height; s++) {
              data[off]=((current[s].g / 4) % 8)*32 + (current[s].b / 8);
              ++off;
              data[off]=(current[s].r / 8)*8+
                               (current[s].g / 4) / 8;
              ++off;
            }
          }
          else if (image->bitmap_bit_order==LSBFirst && image->byte_order==MSBFirst) {
            return;
          }
          else {
            return;
          }
        }
        else if (image->bits_per_pixel==8) {
          for (size_t s=0; s<width*height; s++) {
            /* We assume  2/2/2, which likely will not work (normally 3/3/2 or similar) */
            data[s]=(current[s].r / 64)*16+
                    (current[s].g / 64)*4+
                    (current[s].b / 64);
          }
        }
        else {
          return;
        }

        pixmap=XCreatePixmap(display,
                             XDefaultRootWindow(display),
                             width,
                             height,
                             depth);

        gc=XCreateGC(display,pixmap,0,&values);

        XPutImage(display,
                  pixmap,
                  gc,
                  image,
                  0,
                  0,
                  0,
                  0,
                  width,
                  height);

        image->data=NULL;
        XDestroyImage(image);
        XFreeGC(display,gc);
        delete [] data;
        data=NULL;

        if (alpha) {
          size_t pad,w,x,off;

          /* Create alpha one-bit mask */
          pad=8;

          w=(width+pad-1) / pad;
          data = new char[(w*pad) / 8 * height];

          value=0;
          off=0;
          x=1;
          for (size_t s=0; s<width*height; s++) {
            if (current[s].a==255) {
              switch (x) {
              case 8:
                value+=128;
                break;
              case 7:
                value+=64;
                break;
              case 6:
                value+=32;
                break;
              case 5:
                value+=16;
                break;
              case 4:
                value+=8;
                break;
              case 3:
                value+=4;
                break;
              case 2:
                value+=2;
                break;
              case 1:
                value+=1;
                break;
              }
            }
            x++;

            if (x==9) {
              data[off]=value;
              off++;
              x=1;
              value=0;
            }
            else if ((s % width)==(width-1)) {
              if (x!=1) {
                data[off]=value;
                x=1;
                value=0;
                off++;
              }
            }
          }

          image=XCreateImage(display,
                             visual,
                             1,
                             ZPixmap,
                             0,
                             data,
                             width,
                             height,
                             pad,
                             w);

          if (image==NULL) {
            std::cerr << "Cannot create shape!" << std::endl;
          }

          shape=XCreatePixmap(display,
                              XDefaultRootWindow(display),
                              width,
                              height,
                              1);

          gc=XCreateGC(display,shape,0,&values);

          XPutImage(display,
                    shape,
                    gc,
                    image,
                    0,
                    0,
                    0,
                    0,
                    width,
                    height);

          image->data=NULL;
          XDestroyImage(image);
          XFreeGC(display,gc);
          delete [] data;
        }
      }

      void Image::FreeOSStructures()
      {
        if (pixmap!=0) {
          XFreePixmap(display,pixmap);
          pixmap=0;
        }

        if (shape!=0) {
          XFreePixmap(display,shape);
          shape=0;
        }

#if defined(HAVE_LIB_XFT2)
        if (picture!=0) {
          XRenderFreePicture(display,picture);
          picture=0;
        }
#endif
      }

      void Image::Draw(::Lum::OS::DrawInfo* draw, int x, int y)
      {
        DrawSub(draw,0,0,width,height,x,y);
      }

      void Image::DrawSub(::Lum::OS::DrawInfo* draw, int x, int y, size_t w, size_t h, int dx, int dy)
      {
        if (w==0 || h==0 || dynamic_cast<DrawInfo*>(draw)==NULL) {
          return;
        }

#if defined(HAVE_LIB_XFT2)
        if (picture==0) {
          CreateOSStructures();
        }

        if (picture!=0) {
          ::Picture                  dstPic;
          ::XRenderPictureAttributes attr;
          ::Pixmap                   scratch;
          ::XGCValues                values;
          ::GC                       gc;

          scratch=XCreatePixmap(display,XDefaultRootWindow(display),w,h,depth);

          gc=XCreateGC(display,scratch,0,&values);

          XCopyArea(display,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    scratch,
                    gc,
                    dx,dy,w,h,
                    0,0);

          dstPic=XRenderCreatePicture(display,scratch,format,0,&attr);
          XRenderComposite(display,
                           PictOpOver,
                           picture,
                           0,
                           dstPic,
                           x,y,
                           0,0,
                           0,0,
                           w,h);

          XCopyArea(display,
                    scratch,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    dynamic_cast<DrawInfo*>(draw)->gc,
                    0,0,w,h,
                    dx,dy);

          XRenderFreePicture(display,dstPic);
          XFreeGC(display,gc);
          XFreePixmap(display,scratch);
        }
        return;
#endif

        if (pixmap==0) {
          CreateOSStructures();
        }

        if (pixmap!=0) {
          XSetGraphicsExposures(display,dynamic_cast<DrawInfo*>(draw)->gc,False);
          XCopyArea(display,
                    pixmap,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    dynamic_cast<DrawInfo*>(draw)->gc,
                    x,y,w,h,
                    dx,dy);
          XSetGraphicsExposures(display,dynamic_cast<DrawInfo*>(draw)->gc,True);
        }
      }


      void Image::DrawSubCliped(::Lum::OS::DrawInfo* draw, int x, int y, size_t w, size_t h, int dx, int dy)
      {
        if (w==0 || h==0 || dynamic_cast<DrawInfo*>(draw)==NULL) {
          return;
        }

#if defined(HAVE_LIB_XFT2)
        if (picture==0) {
          CreateOSStructures();
        }

        if (picture!=0) {
          ::Picture                  dstPic;
          ::XRenderPictureAttributes attr;
          ::Pixmap                   scratch;
          ::XGCValues                values;
          ::GC                       gc;

          scratch=XCreatePixmap(display,XDefaultRootWindow(display),w,h,depth);

          gc=XCreateGC(display,scratch,0,&values);

          XCopyArea(display,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    scratch,
                    gc,
                    dx,dy,w,h,
                    0,0);

          dstPic=XRenderCreatePicture(display,scratch,format,0,&attr);
          XRenderComposite(display,
                           PictOpOver,
                           picture,
                           0,
                           dstPic,
                           x,y,
                           0,0,
                           0,0,
                           w,h);

          XCopyArea(display,
                    scratch,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    dynamic_cast<DrawInfo*>(draw)->gc,
                    0,0,w,h,
                    dx,dy);

          XRenderFreePicture(display,dstPic);
          XFreeGC(display,gc);
          XFreePixmap(display,scratch);
        }

        return;
#endif

        if (pixmap==0) {
          CreateOSStructures();
        }

        if (pixmap!=0) {
          if (shape!=0) {
            XSetClipMask(display,dynamic_cast<DrawInfo*>(draw)->gc,shape);
            XSetClipOrigin(display,dynamic_cast<DrawInfo*>(draw)->gc,dx-x,dy-y);
          }

          XSetGraphicsExposures(display,dynamic_cast<DrawInfo*>(draw)->gc,False);
          XCopyArea(display,
                    pixmap,
                    dynamic_cast<DrawInfo*>(draw)->drawable,
                    dynamic_cast<DrawInfo*>(draw)->gc,
                    x,y,w,h,
                    dx,dy);
          XSetGraphicsExposures(display,dynamic_cast<DrawInfo*>(draw)->gc,True);

          if (shape!=0) {
            dynamic_cast<DrawInfo*>(draw)->ReinstallClip();
          }
        }
      }

      void Image::DrawTiled(::Lum::OS::DrawInfo* draw, int x, int y, size_t w, size_t h, int dx, int dy)
      {
        if (w==0 || h==0 || dynamic_cast<DrawInfo*>(draw)==NULL) {
          return;
        }

        if (pixmap==0) {
          CreateOSStructures();
        }

        if (pixmap!=0) {
          XSetTile(display,dynamic_cast<DrawInfo*>(draw)->gc,pixmap);
          XSetTSOrigin(display,dynamic_cast<DrawInfo*>(draw)->gc,dx,dy);
          XSetFillStyle(display,dynamic_cast<DrawInfo*>(draw)->gc,FillTiled);
          XFillRectangle(display,dynamic_cast<DrawInfo*>(draw)->drawable,dynamic_cast<DrawInfo*>(draw)->gc,x,y,w,h);
          /* TODO: Restore correct fill mode */
          XSetFillStyle(display,dynamic_cast<DrawInfo*>(draw)->gc,FillSolid);
        }
      }

      ImageFactory::ImageFactory(::Lum::OS::Display* d)
      : Factory(d)
      {
        XVisualInfo info;
#if defined(HAVE_LIB_XFT2)
          int       tmp1,tmp2;
#endif

        display=dynamic_cast< ::Lum::OS::X11::Display*>(d)->display;
        scrNum=dynamic_cast< ::Lum::OS::X11::Display*>(d)->scrNum;
        visual=dynamic_cast< ::Lum::OS::X11::Display*>(d)->visual;
        depth=dynamic_cast< ::Lum::OS::X11::Display*>(d)->GetColorDepth();
        colorMap=dynamic_cast< ::Lum::OS::X11::Display*>(d)->colorMap;

        if (depth>=24 && XMatchVisualInfo(display,scrNum,24,TrueColor,&info)!=0) {
          visual=info.visual;
          depth=24;
          colorMap=XCreateColormap(display,
                                   XDefaultRootWindow(display),
                                   visual,
                                   AllocNone);

          dynamic_cast< ::Lum::OS::X11::Display*>(d)->visual=visual;
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->SetColorDepth(depth);
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->colorMap=colorMap;
        }
        else if (depth>=16 && XMatchVisualInfo(display,scrNum,16,TrueColor,&info)!=0) {
          visual=info.visual;
          depth=16;
          colorMap=XCreateColormap(display,
                                   XDefaultRootWindow(display),
                                   visual,
                                   AllocNone);

          dynamic_cast< ::Lum::OS::X11::Display*>(d)->visual=visual;
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->SetColorDepth(depth);
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->colorMap=colorMap;
        }
        else if (depth>=8 && XMatchVisualInfo(display,scrNum,8,TrueColor,&info)!=0) {
          // untested!!!
          visual=info.visual;
          depth=8;
          colorMap=XCreateColormap(display,
                                   XDefaultRootWindow(display),
                                   visual,
                                   AllocNone);

          dynamic_cast< ::Lum::OS::X11::Display*>(d)->visual=visual;
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->SetColorDepth(depth);
          dynamic_cast< ::Lum::OS::X11::Display*>(d)->colorMap=colorMap;
        }

#if defined(HAVE_LIB_XFT2)
        render=XRenderQueryExtension(display,&tmp1,&tmp2)==True;

        if (render) {
          ::XRenderPictFormat pFormat;

          format=XRenderFindVisualFormat(display,visual);

          pFormat.id=0;
          pFormat.type=PictTypeDirect;
          pFormat.depth=32;
          pFormat.direct.red=16;
          pFormat.direct.redMask=0xff;
          pFormat.direct.green=8;
          pFormat.direct.greenMask=0xff;
          pFormat.direct.blue=0;
          pFormat.direct.blueMask=0xff;
          pFormat.direct.alpha=24;
          pFormat.direct.alphaMask=0xff;
          pFormat.colormap=0;

          alphaFormat=XRenderFindFormat(display,
                                        PictFormatType |
                                        PictFormatDepth |
                                        PictFormatRed |
                                        PictFormatRedMask |
                                        PictFormatGreen |
                                        PictFormatGreenMask |
                                        PictFormatBlue |
                                        PictFormatBlueMask |
                                        PictFormatAlpha |
                                        PictFormatAlphaMask,
                                        &pFormat,0);
        }
#endif
      }

      void ImageFactory::Deinit(::Lum::OS::Display* /*d*/)
      {
        // TODO
      }


      ::Lum::Images::Image* ImageFactory::CreateImage() const
      {
        return new Image();
      }
    }
  }
}
