/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/X11/Font.h>

#include <unistd.h>

#include <X11/X.h>

#include <Lum/Base/String.h>

#include <Lum/OS/X11/Display.h>

namespace Lum {
  namespace OS {
    namespace X11 {

      Font::Font()
      : loaded(false)
      {
        for (size_t x=0; x<maxStyleNum ; x++) {
          fonts[x]=NULL;
#if defined(HAVE_LIB_XFT2)
          xftfonts[x]=NULL;
#endif
        }
      }

      Font::~Font()
      {
        for (size_t x=0; x<maxStyleNum; x++) {
          if (fonts[x]!=NULL) {
            XFreeFont(dynamic_cast<Display*>(display)->display,fonts[x]);
          }
#if defined(HAVE_LIB_XFT2)
          if (xftfonts[x]!=NULL) {
            if (x==bold) {
            }
            XftFontClose(dynamic_cast<Display*>(display)->display,xftfonts[x]);
          }
#endif
        }
      }

      size_t Font::StringWidth(const std::wstring& text, unsigned long style)
      {
        unsigned long pos=GetFontPos(style);

#if defined(HAVE_LIB_XFT2)
        XGlyphInfo info;

        if (xftfonts[pos]!=NULL) {
#if SIZEOF_WCHAR_T == 4
          XftTextExtents32(dynamic_cast<Display*>(display)->display,
                           xftfonts[pos],
                           (XftChar32*)text.c_str(),text.length(),&info);
#else
          XftTextExtents16(dynamic_cast<Display*>(display)->display,
                           xftfonts[pos],
                           (XftChar16*)text.c_str(),text.length(),&info);
#endif

          return info.xOff;
        }
#endif

        char   *buffer;
        size_t res;
        size_t length;

        buffer=::Lum::Base::WStringToChar16BE(text,length);

        res=XTextWidth16(fonts[pos],(XChar2b*)buffer,length);

        delete [] buffer;

        return res;
      }

      void Font::StringExtent(const std::wstring& text, FontExtent& extent, unsigned long style)
      {
        unsigned long pos=GetFontPos(style);

#if defined(HAVE_LIB_XFT2)
        XGlyphInfo info;

        if (xftfonts[pos]!=NULL) {
#if SIZEOF_WCHAR_T == 4
          XftTextExtents32(dynamic_cast<Display*>(display)->display,
                           xftfonts[pos],
                           (XftChar32*)text.c_str(),text.length(),&info);
#else
          XftTextExtents16(dynamic_cast<Display*>(display)->display,
                           xftfonts[pos],
                           (XftChar16*)text.c_str(),text.length(),&info);
#endif

          extent.left=-info.x;
          extent.right=info.xOff-info.width+info.x;
          extent.width=info.xOff;

          extent.height=info.height;
          extent.ascent=info.y;
          extent.descent=info.height-info.y;

          return;
        }
#endif

        char*        buffer;
        int          direction=0,ascent=0,descent=0;
        XCharStruct  res;
        size_t       length=0;

        buffer=::Lum::Base::WStringToChar16BE(text,length);

        XTextExtents16(fonts[pos],(XChar2b*)buffer,length,&direction,&ascent,&descent,&res);

        delete [] buffer;

        extent.left=res.lbearing;
        extent.right=res.width-res.rbearing;
        extent.width=res.width;
        extent.ascent=res.ascent;
        extent.descent=res.descent;
        extent.height=res.ascent+res.descent;
      }

      std::string Font::BuildFontName() const
      {
        char buffer[128];

        std::string name="-*-";

        // name
        if (attrName & features) {
          name.append(this->name);
        }
        else {
          name.append("*");
        }

        name.append("-medium-r-");

        // setWidth
      /*    IF D.fontSetWidth IN font.features THEN
            CASE font.setWidth OF
              D.fontNormal:
                str.Append("normal",name);
            | D.fontCondensed:
                str.Append("semicondensed",name);
            | D.fontNarrow:
                str.Append("narrow",name);
            | D.fontDouble:
                str.Append("double",name);
            END;
          ELSE
            str.Append("*",name);
        END;*/
        name.append("normal");

        name.append("--");

        // height
        if (attrHeight & features) {
          sprintf(buffer,"%u",pixelHeight);
          name.append(buffer);
        }
        else {
          name.append("*");
        }

        name.append("-");

        // pointHeight
      /*    IF D.fontPointHeight IN font.features THEN
            co.IntToStr(font.pointHeight,buffer);
            str.Append(buffer,name);
          ELSE
            str.Append("*",name);
        END;*/
        name.append("*");

        name.append("-");

        // horizRes
        name.append("*-");

        // vertRes
        name.append("*-");

        // spacing
        if (attrSpacing & features) {
          switch (spacing) {
          case spacingMonospace:
            name.append("*"); // 'm' or 'c'?
            break;
          case spacingProportional:
            name.append("p");
            break;
          case spacingDefault:
            name.append("*");
            break;
          }
        }
        else {
          name.append("*");
        }

        name.append("-");

        // avWidth
       /*   IF D.fontAvWidth IN font.features THEN
            co.IntToStr(font.avWidth,buffer);
            str.Append(buffer,name);
          ELSE
            str.Append("*",name);
        END;*/

        name.append("*");

        name.append("-");

        if (dynamic_cast<Display*>(display)->xim!=0 &&  attrCharSet & features) {
          name.append(charSet);
        }
        else {
          name.append("*-*");
        }

        return name;
      }

#if defined(HAVE_LIB_XFT2)
      std::string Font::BuildXftFontName(unsigned long style) const
      {
        if (!(attrName & features)) {
          return "";
        }

        std::string result;

        result.append(name);
        result.append(":weight=medium");

        if (attrHeight & features) {
          result.append(":pixelsize=");
          result.append(Lum::Base::NumberToString(pixelHeight));
        }



        // spacing
        if (attrSpacing & features) {
          switch (spacing) {
          case spacingMonospace:
            result.append(":spacing=mono");
            break;
          case spacingProportional:
          case spacingDefault:
            result.append(":spacing=propertional");
            break;
          }
        }

        if (style & bold) {
          result.append(":weight=bold");
        }

        if (style & italic) {
          result.append(":slant=italic");
        }

        if (style & slanted) {
          result.append(":slant=oblique");
        }

        return result;
      }
#endif

      /**
        Loads the font. The OS specific font will be matched by evaluating the
        handed features. Every font loaded must be free using Free.

        RETURNS
        If the font can be loaded the method returns a new instance of font that
        must be used for future uses exspecially when calling Free.
      */
      OS::Font* Font::Load()
      {
#if defined(HAVE_LIB_XFT2)
        fullName=BuildXftFontName(normal);
        xftfonts[0]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                    dynamic_cast<Display*>(display)->scrNum,
                                    fullName.c_str());
        if (xftfonts[0]!=NULL) {
          height=xftfonts[0]->ascent+xftfonts[0]->descent;
          ascent=xftfonts[0]->ascent;
          descent=xftfonts[0]->descent;
          return this;
        }
#endif

        fullName=BuildFontName();

        fonts[0]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                fullName.c_str());
        if (fonts[0]!=NULL) {
          height=fonts[0]->ascent+fonts[0]->descent;
          ascent=fonts[0]->ascent;
          descent=fonts[0]->descent;
          return this;
        }

        return NULL;
      }

      std::string ConvertField(const std::string& name,
                               size_t pos,
                               const std::string& string)
      {
        size_t x1,x2;

        std::string res=name;

        x1=0;
        while (x1<res.length() && pos>0) {
          if (res[x1]=='-') {
            pos--;
          }
          x1++;
        }

        x2=x1;

        while (x2+1<res.length() && res[x2+1]!='-') {
          x2++;
        }

        res.erase(x1,x2-x1+1);
        res.insert(x1,string);

        return res;
      }

      void Font::Complete()
      {
        if (loaded) {
          return;
        }

#if defined(HAVE_LIB_XFT2)
        if (xftfonts[0]!=NULL) {
          std::string name;

          /* bold */
          name=BuildXftFontName(bold);
          xftfonts[bold]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                         dynamic_cast<Display*>(display)->scrNum,
                                         name.c_str());

          /* italic */
          name=BuildXftFontName(italic);
          xftfonts[italic]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                           dynamic_cast<Display*>(display)->scrNum,
                                           name.c_str());

          /* slanted */
          name=BuildXftFontName(slanted);
          xftfonts[slanted]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                            dynamic_cast<Display*>(display)->scrNum,
                                            name.c_str());

          /* bold + italic */
          name=BuildXftFontName(bold|italic);
          xftfonts[bold|italic]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                                dynamic_cast<Display*>(display)->scrNum,
                                                name.c_str());

          /* bold + slanted */
          name=BuildXftFontName(bold|slanted);
          xftfonts[bold|slanted]=XftFontOpenName(dynamic_cast<Display*>(display)->display,
                                                 dynamic_cast<Display*>(display)->scrNum,
                                                 name.c_str());

          loaded=true;
          return;
        }

#endif
        std::string help;

        /* bold */
        help=ConvertField(fullName,3,"bold");
        fonts[bold]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                   help.c_str());

        /* italic */
        help=ConvertField(fullName,4,"i");
        fonts[italic]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                     help.c_str());

        /* slanted */
        help=ConvertField(fullName,4,"o");
        fonts[slanted]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                      help.c_str());

        /* bold + italic */
        help=ConvertField(fullName,3,"bold");
        help=ConvertField(help,4,"i");
        fonts[bold|italic]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                          help.c_str());

        /* bold + slanted */
        help=ConvertField(fullName,3,"bold");
        help=ConvertField(help,4,"o");
        fonts[bold|slanted]=XLoadQueryFont(dynamic_cast<Display*>(display)->display,
                                           help.c_str());

        loaded=true;
      }

      /**
        Returns the offset in the font array, that corresponds to the given style.

        \note
        GetFontPos tries to find a "close" font if a font with the given style bits
        does not exist. Currently it sees italic and slanted as equal, so if no italic
        font does exist, it tries a slanted one and vice versa.
      */
      unsigned long Font::GetFontPos(unsigned long style)
      {

        unsigned long tmp=(style&(maxStyleNum-1));

        if (tmp!=normal && !loaded) {
          Complete();
        }

#if defined(HAVE_LIB_XFT2)
        if (xftfonts[tmp]!=NULL) {
          return tmp;
        }

        if (slanted & tmp) {
          tmp=(tmp&~slanted);
          tmp=(tmp|italic);
        }
        else if (italic & tmp) {
          tmp=(tmp&~italic);
          tmp=(tmp|slanted);
        }

        if (xftfonts[tmp]!=NULL) {
          return tmp;
        }

        tmp=(style&(maxStyleNum-1));
#endif

        if (fonts[tmp]!=NULL) {
          return tmp;
        }

        if (slanted & tmp) {
          tmp=(tmp&~slanted);
          tmp=(tmp|italic);
        }
        else if (italic & tmp) {
          tmp=(tmp&~italic);
          tmp=(tmp|slanted);
        }

        if (fonts[tmp]!=NULL) {
          return tmp;
        }

        return normal;
      }
    }
  }
}
