/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Menu.h>

#include <cctype>

#include <Lum/Base/Util.h>

#include <Lum/OS/Driver.h>

#include <Lum/Dialog.h>
#include <Lum/Text.h>

namespace Lum {

  class MenuStripPrefs : public List::Prefs
  {
  public:
    MenuStripPrefs(const std::wstring& name)
    : List::Prefs(name)
    {
      // no code
    }

    void Initialize()
    {
      List::Prefs::Initialize();

      background=OS::display->GetFill(OS::Display::menuStripBackgroundFillIndex);
      frame=OS::display->GetFrame(OS::Display::menuStripFrameIndex);
    }
  };

  class PullDownMenuPrefs : public Object::Prefs
  {
  public:
    PullDownMenuPrefs(const std::wstring& name)
    : Object::Prefs(name)
    {
      // no code
    }

    void Initialize()
    {
      Object::Prefs::Initialize();

      background=OS::display->GetFill(OS::Display::menuPulldownBackgroundFillIndex);
      frame=OS::display->GetFrame(OS::Display::menuPulldownFrameIndex);
    }
  };

  class MenuEntryPrefs : public Control::Prefs
  {
  public:
    OS::ImageRef        arrow;      //! Image used for sub menues
    OS::ImageRef        check;      //! Image used for boolean menues
    OS::ImageRef        radio;      //! Image used for radio menues
    OS::ImageRef        divider;    //! Image used as menu divider

  public:
    MenuEntryPrefs(const std::wstring& name)
    : Control::Prefs(name)
    {
      // no code
    }

    void Initialize()
    {
      Control::Prefs::Initialize();

      background=OS::display->GetFill(OS::Display::menuEntryBackgroundFillIndex);

      arrow=OS::display->GetImage(OS::Display::menuSubImageIndex);
      check=OS::display->GetImage(OS::Display::checkImageIndex);
      radio=OS::display->GetImage(OS::Display::radioImageIndex);
      divider=OS::display->GetImage(OS::Display::menuDividerImageIndex);
    }
  };

  class MenuPrefs : public Object::Prefs
  {
  public:
    MenuPrefs(const std::wstring& name)
    : Object::Prefs(name)
    {
      // no code
    }

    void Initialize()
    {
      Object::Prefs::Initialize();

      frame=OS::display->GetFrame(OS::Display::menuWindowFrameIndex);
    }
  };

  static MenuStripPrefs    *menuStripPrefs=new MenuStripPrefs(L"MenuStrip");
  static PullDownMenuPrefs *pullDownMenuPrefs=new PullDownMenuPrefs(L"PullDownMenu");
  static MenuEntryPrefs    *menuEntryPrefs=new MenuEntryPrefs(L"MenuEntry");
  static MenuPrefs         *menuPrefs=new MenuPrefs(L"Menu");

  /* ---------- Menustrip stuff ------------------*/

  MenuStrip::MenuStrip()
  : selection(NULL),stickyMode(false)
  {
    SetFlex(true,false);

    SetPrefs(menuStripPrefs);
  }

  /**
    Finds the menu entry (@otype{PullDownMenu}) that opens the given (sub)menu).
  */
  std::list<Object*>::iterator MenuStrip::FindLabel(Menu* menu)
  {
    std::list<Object*>::iterator iter;

    iter=list.begin();
    while (iter!=list.end()) {
      if (dynamic_cast<PullDownMenu*>(*iter)->subMenu==menu) {
        return iter;
      }

      ++iter;
    }

    return list.end();
  }

  /**
    Select the given (direct) sub menu.
  */
  void MenuStrip::SetSelection(Menu* newMenu)
  {
    if (selection!=NULL && selection->IsOpen()) {
      selection->Close();
    }
    selection=newMenu;
    if (selection!=NULL) {
      selection->Open();
    }

    Redraw();
  }

  /**
    Select the previous entry in the menu strip.
  */
  void MenuStrip::SelectPrevious()
  {
    std::list<Object*>::iterator object;

    if (selection==NULL) {
      return;
    }

    object=FindLabel(selection);

    if (object==list.begin()) {
      SetSelection(dynamic_cast<PullDownMenu*>(*list.rbegin())->subMenu);
    }
    else {
      object--;
      SetSelection(dynamic_cast<PullDownMenu*>(*object)->subMenu);
    }
  }

  /**
    Select the next entry in the menu strip.
  */
  void MenuStrip::SelectNext()
  {
    std::list<Object*>::iterator object;

    if (selection==NULL) {
      return;
    }

    object=FindLabel(selection);

    object++;

    if (object!=list.end()) {
      SetSelection(dynamic_cast<PullDownMenu*>(*object)->subMenu);
    }
    else if (list.begin()!=list.end()) {
      SetSelection(dynamic_cast<PullDownMenu*>(*list.begin())->subMenu);
    }
  }

  /**
    Return the (PullDownMenu) object currently under the mouse or
    NULL, if there is no child object under the mouse.
  */
  Object* MenuStrip::GetMouseSelected() const
  {
    std::list<Object*>::const_iterator object;

    object=list.begin();
    while (object!=list.end()) {
      if ((*object)->MouseIsIn()) {
        return *object;
      }

      ++object;
    }

    return NULL;
  }

  void MenuStrip::AddPullDownMenu(Object* label, wchar_t sc, Menu* subMenu)
  {
    PullDownMenu *menu;

    menu=new PullDownMenu();
    menu->SetFlex(false,true);
    menu->SetLabel(label);
    menu->SetHotkey(sc);
    menu->subMenu=subMenu;
    menu->strip=this;

    if (subMenu!=NULL) {
      subMenu->strip=this;
      subMenu->SetReference(menu);
    }

    Add(menu);
  }

  void MenuStrip::AddPullDownMenu(const std::wstring& label, Menu* subMenu)
  {
    Text    *text;
    wchar_t shortcut;
    size_t  pos;

    text=new Text();
    text->SetAlignment(Text::centered);
    text->SetFlex(true,true);

    pos=label.find('_');
    if (pos!=std::wstring::npos && pos+1<label.length()) {
      shortcut=label[pos+1];
      text->AddText(label.substr(0,pos));
      text->AddText(label.substr(pos+1,1),OS::Font::underlined);
      text->AddText(label.substr(pos+2));
    }
    else {
      shortcut='\0';
      text->AddText(label);
    }

    AddPullDownMenu(text,shortcut,subMenu);
  }

  void MenuStrip::CalcSize()
  {
    std::list<Object*>::const_iterator object;

    width=0;
    height=0;

    object=list.begin();
    while (object!=list.end()) {
      (*object)->CalcSize();
      height=std::max(height,(*object)->GetOHeight());
      width+=(*object)->GetOWidth();

      ++object;
    }

    List::CalcSize();
  }

  void MenuStrip::Layout()
  {
    std::list<Object*>::const_iterator object;
    int                                  pos;

    pos=x;
    object=list.begin();
    while (object!=list.end()) {
      (*object)->ResizeHeight(height);
      (*object)->Move(pos,y + (height-(*object)->GetOHeight()) / 2);
      pos+=(*object)->GetOWidth();

      ++object;
    }

    List::Layout();
  }

  /**
    Before drawing a child, find out, if it is currently selected and make it
    drawn selected, if it is.
  */
  void MenuStrip::PreDrawChild(Object* child)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    std::list<Object*>::iterator label=FindLabel(selection);
    if (label!=list.end() && child==*label) {
      draw->selected=true;
    }
  }

  /**
    Reverts @oproc{MenuStrip.PreDrawChild}.
  */
  void MenuStrip::PostDrawChild(Object* /*child*/)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    draw->selected=false;
  }

  /* ---------- PullDownMenu stuff ------------------*/

  PullDownMenu::PullDownMenu()
  : label(NULL),sc('\0'),
    shortcutAction(new Model::Action),
    subMenu(NULL)
  {
    SetPrefs(pullDownMenuPrefs);

    SetFlex(true,true);

    SetCanDrawMouseActive(true);

    AttachModel(shortcutAction);
  }

  PullDownMenu::~PullDownMenu()
  {
    UnattachModel(shortcutAction);

    delete label;
    delete subMenu;
  }


  void PullDownMenu::SetLabel(Object* object)
  {
    label=object;
    label->SetParent(this);
  }

  void PullDownMenu::SetHotkey(wchar_t sc)
  {
    this->sc=sc;
  }

  void PullDownMenu::CalcSize()
  {
    label->CalcSize();

    width=label->GetOWidth()+2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
    height=label->GetOHeight()+2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);
    minWidth=width;
    minHeight=height;

    if ((sc!=L'\0') && (dynamic_cast<Dialog*>(GetWindow()->GetMaster())!=NULL)) {
      dynamic_cast<Dialog*>(GetWindow()->GetMaster())->RegisterShortcut(this,
                                                                        OS::qualifierAlt,std::wstring(1,sc),
                                                                        shortcutAction);
    }

    if (subMenu!=NULL) {
      subMenu->SetParent(GetWindow());
    }

    Object::CalcSize();
  }

  void PullDownMenu::Layout()
  {
    label->Resize(width-2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder),
                  height-2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));
    label->Move(x+OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder),
                y+OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));

    Object::Layout();
  }

  bool PullDownMenu::HandleMouseEvent(const OS::MouseEvent& event)
  {
    if (!visible) {
      return false;
    }

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && (event.button==OS::MouseEvent::button1 || event.button==OS::MouseEvent::button3)) {
      strip->SetSelection(subMenu);
      return true;
    }
    else if (event.type==OS::MouseEvent::move && subMenu!=NULL && !subMenu->IsOpen() && PointIsIn(event) && (event.qualifier==OS::qualifierButton1 || event.qualifier==OS::qualifierButton3 || strip->stickyMode)) {
      strip->SetSelection(subMenu);
      return true;
    }

    return false;
  }

  void PullDownMenu::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    draw->activated=IsMouseActive();

    Object::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      draw->activated=false;
      return;
    }

    /* ---- */

    DrawBackground(x,y,w,h);

    label->Draw(x,y,w,h);

    draw->activated=false;
  }

  void PullDownMenu::Hide()
  {
    if (visible) {
      label->Hide();
    }

    Object::Hide();
  }

  void PullDownMenu::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->shortcutAction && this->shortcutAction->IsFinished()) {
      strip->stickyMode=true;
      strip->SetSelection(subMenu);
    }

    Object::Resync(model,msg);
  }

  /* ---------- Menu entry stuff ------------------*/

  MenuEntry::MenuEntry()
  : selectable(false),
    shortcutAction(new Model::Action),
    image(NULL),label(NULL),shortcut(NULL),subImage(NULL),
    x1(0),x2(0),x3(0),x4(0),
    sc(L'\0'),
    w1(0),w2(0),w3(0),w4(0),
    menuWindow(NULL)
  {
    SetPrefs(menuEntryPrefs);

    shortcutAlways=true;

    AttachModel(shortcutAction);
  }

  MenuEntry::~MenuEntry()
  {
    delete label;
    delete shortcut;

    UnattachModel(shortcutAction);
  }

  void MenuEntry::SetOffsets(int x1, int x2, int x3, int x4)
  {
    this->x1=x1;
    this->x2=x2;
    this->x3=x3;
    this->x4=x4;
  }

  void MenuEntry::Hide()
  {
    if (label!=NULL) {
      label->Hide();
    }
    if (shortcut!=NULL) {
      shortcut->Hide();
    }

    Control::Hide();
  }

  bool MenuEntry::IsSelectable() const
  {
    return selectable && GetModel()!=NULL && GetModel()->IsEnabled();
  }

  void MenuEntry::SetHotkey(wchar_t key)
  {
    sc=key;
  }

  void MenuEntry::RegisterToParentWindow(OS::Window* parent)
  {
    if (!key.empty()) {
      assert(parent!=NULL);
      if (dynamic_cast<Dialog*>(parent->GetMaster())!=NULL) {
        dynamic_cast<Dialog*>(parent->GetMaster())->RegisterShortcut(this,qualifier,key,shortcutAction);
      }
    }
  }

  void MenuEntry::CalcLabelSize()
  {
    if (label!=NULL) {
      if (!label->IsInited()) {
        label->SetParent(this);
        label->SetFlex(true,true);
        label->CalcSize();
      }
      w2=label->GetOWidth();
      height=std::max(height,label->GetOHeight()+2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));
    }
  }

  void MenuEntry::CalcShortcutSize()
  {
    if (shortcut!=NULL) {
      if (!shortcut->IsInited()) {
        shortcut->SetParent(this);
        shortcut->SetFlex(true,true);
        shortcut->CalcSize();
      }
      w3=shortcut->GetOWidth();
      height=std::max(height,shortcut->GetOHeight()+2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));
    }
    else {
      w3=0;
    }
  }

  void MenuEntry::DrawImage(OS::DrawInfo* draw)
  {
    if (image.Valid()) {
      // Left align image
      image->Draw(draw,
                   x1,y+(height-image->GetHeight()) / 2,
                   image->GetWidth(),image->GetHeight());
    }
  }

  void MenuEntry::DrawLabel(OS::DrawInfo* /*draw*/)
  {
    if (label!=NULL) {
      // Left align label
      label->Move(x2,y+(height-label->GetOHeight()) / 2);
      label->Draw(x,y,width,height);
    }
  }

  void MenuEntry::DrawShortcut(OS::DrawInfo* /*draw*/)
  {
    if (shortcut!=NULL) {
      // Right align shortcut
      shortcut->Move(x3+w3-shortcut->GetOWidth(),
                     y+(height-shortcut->GetOHeight()) / 2);
      shortcut->Draw(x,y,width,height);
    }
  }

  void MenuEntry::DrawSubImage(OS::DrawInfo* draw)
  {
    if (subImage.Valid()) {
      // Left align sub image
      subImage->Draw(draw,
                     x4,y+(height-subImage->GetHeight()) / 2,
                     subImage->GetWidth(),subImage->GetHeight());
    }
  }

  Menu* MenuEntry::GetSubMenu() const
  {
    return NULL;
  }

  void MenuEntry::SetLabel(Object* label)
  {
    this->label=label;
  }

  void MenuEntry::SetTextLabel(const std::wstring& label)
  {
    Text    *text;
    size_t  pos;
    wchar_t sc;

    text=new Text();
    text->SetAlignment(Text::centered);
    text->SetFlex(true,true);

    pos=label.find('_');
    if (pos!=std::wstring::npos && pos+1<label.length()) {
      sc=label[pos+1];
      text->AddText(label.substr(0,pos));
      text->AddText(label.substr(pos+1,1),OS::Font::underlined);
      text->AddText(label.substr(pos+2));
      SetHotkey(sc);
    }
    else {
      sc='\0';
      text->AddText(label);
    }

    SetLabel(text);
  }

  void MenuEntry::SetShortcut(unsigned long qualifier, const std::wstring& key)
  {
    std::wstring shortcut;

    OS::display->KeyToDisplayKeyDescription(qualifier,key,shortcut);
    this->shortcut=new Text(shortcut);
    this->qualifier=qualifier;
    this->key=key;
  }

  void MenuEntry::OnSelection()
  {
    // no code
  }

  void MenuEntry::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->shortcutAction && this->shortcutAction->IsFinished()) {
      OnSelection();
    }

    Control::Resync(model,msg);
  }

  /* ---------- Menu item stuff ------------------*/

  ActionItem::ActionItem()
  : action(NULL)
  {
    selectable=true;
  }

  bool ActionItem::SetModel(Base::Model* model)
  {
    action=dynamic_cast<Model::Action*>(model);

    Control::SetModel(action);

    return action.Valid();
  }

  void ActionItem::OnSelection()
  {
    if (action.Valid() && action->IsEnabled()) {
      action->Trigger();
    }
  }

  void ActionItem::CalcSize()
  {
    assert(label!=NULL);

    CalcLabelSize();
    CalcShortcutSize();

    width=w1+w2+w3+w4;

    MenuEntry::CalcSize();
  }

  void ActionItem::Draw(int x, int y, size_t w, size_t h)
  {
    MenuEntry::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    if (!action.Valid() || !action->IsEnabled()) {
      draw->disabled=true;
    }
    if (menuWindow->selected==this) {
      draw->selected=true;
    }

    DrawBackground(x,y,w,h);

    DrawImage(draw);
    DrawLabel(draw);
    DrawShortcut(draw);

    draw->disabled=false;
    draw->selected=false;
  }

  /* ---------- Separator menu stuff ------------------*/

  bool SeparatorItem::SetModel(Base::Model* /*model*/)
  {
    return false;
  }

  void SeparatorItem::CalcSize()
  {
    height=dynamic_cast<MenuEntryPrefs*>(prefs)->divider->GetHeight();
    height+=2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder);

    MenuEntry::CalcSize();
  }

  void SeparatorItem::Draw(int x, int y, size_t w, size_t h)
  {
    MenuEntry::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::ImageRef divider=dynamic_cast<MenuEntryPrefs*>(prefs)->divider;
    OS::DrawInfo *draw=GetDrawInfo();

    DrawBackground(x,y,w,h);

    divider->Draw(draw,
                  this->x,this->y+(height-divider->GetHeight())/2,
                  this->width,divider->GetHeight());
  }

  /* ---------- Sub menu stuff ------------------*/

  SubMenu::SubMenu()
  : subMenu(NULL)
  {
    selectable=true;
  }

  SubMenu::~SubMenu()
  {
    delete subMenu;
  }

  bool SubMenu::SetModel(Base::Model* /*model*/)
  {
    return false;
  }

  void SubMenu::SetSubMenu(Menu* menu)
  {
    subMenu=menu;
    subMenu->parentMenu=this;
  }

  bool SubMenu::IsSelectable() const
  {
    return true;
  }

  void SubMenu::RegisterToParentWindow(OS::Window* parent)
  {
    MenuEntry::RegisterToParentWindow(parent);

    subMenu->SetParent(parent);
  }

  void SubMenu::CalcSize()
  {
    CalcLabelSize();

    subImage=dynamic_cast<MenuEntryPrefs*>(prefs)->arrow;
    w4=subImage->GetWidth();

    width=w1+w2+w3+w4;
    height=std::max(height,subImage->GetHeight()+2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));

    MenuEntry::CalcSize();
  }

  void SubMenu::Draw(int x, int y, size_t w, size_t h)
  {
    MenuEntry::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    if (menuWindow->selected==this) {
      draw->selected=true;
    }

    DrawBackground(x,y,w,h);

    if (menuWindow->selected==this) {
      if (!subMenu->IsOpen() && subMenu->Open()) {
        menuWindow->child=subMenu;
      }
    }
    else {
      if (subMenu->IsOpen()) {
        subMenu->Close();
        menuWindow->child=NULL;
      }
    }

    DrawLabel(draw);
    DrawSubImage(draw);

    draw->selected=false;
  }

  Menu* SubMenu::GetSubMenu() const
  {
    return subMenu;
  }

  /* ---------- Bool item stuff ------------------*/

  BoolItem::BoolItem()
  : model(NULL)
  {
    selectable=true;

    // TODO: Use special radio *menu* image!
    image=OS::display->GetImage(OS::Display::checkImageIndex);
  }

  bool BoolItem::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Boolean*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void BoolItem::OnSelection()
  {
    if (model.Valid() && model->IsEnabled()) {
      model->Toggle();
    }
  }

  void BoolItem::CalcSize()
  {
    assert(label!=NULL);

    CalcLabelSize();
    CalcShortcutSize();

    width=w1+w2+w3+w4;

    MenuEntry::CalcSize();
  }

  void BoolItem::Draw(int x, int y, size_t w, size_t h)
  {
    MenuEntry::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    if (!model.Valid() || !model->IsEnabled()) {
      draw->disabled=true;
    }

    /* Background */

    draw->selected=(menuWindow->selected==this);
    DrawBackground(x,y,w,h);
    draw->selected=false;

    /* Image */

    draw->selected=(model.Valid() && model->Get());
    DrawImage(draw);
    draw->selected=false;

    /* Rest */

    draw->selected=(menuWindow->selected==this);

    DrawLabel(draw);
    DrawShortcut(draw);

    draw->disabled=false;
    draw->selected=false;
  }

  /* ---------- Radio item stuff ------------------*/

  RadioItem::RadioItem()
  : model(NULL),
    value(0)
  {
    selectable=true;

    // TODO: Use special radio *menu* image!
    image=OS::display->GetImage(OS::Display::radioImageIndex);
  }

  bool RadioItem::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::SizeT*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void RadioItem::SetValue(size_t value)
  {
    this->value=value;
  }

  void RadioItem::OnSelection()
  {
    if (model.Valid() && model->IsEnabled()) {
      model->Set(value);
    }
  }

  void RadioItem::CalcSize()
  {
    assert(label!=NULL);

    CalcLabelSize();
    CalcShortcutSize();

    width=w1+w2+w3+w4;

    MenuEntry::CalcSize();
  }

  void RadioItem::Draw(int x, int y, size_t w, size_t h)
  {
    MenuEntry::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    OS::DrawInfo *draw=GetDrawInfo();

    if (!model.Valid() || !model->IsEnabled()) {
      draw->disabled=true;
    }

    /* Background */

    draw->selected=(menuWindow->selected==this);
    DrawBackground(x,y,w,h);
    draw->selected=false;

    /* Image */

    draw->selected=(model.Valid() && model->Get()==value);
    DrawImage(draw);
    draw->selected=false;

    /* Rest */

    draw->selected=(menuWindow->selected==this);

    DrawLabel(draw);
    DrawShortcut(draw);

    draw->disabled=false;
    draw->selected=false;
  }

  /* ---------- Menu box stuff ------------------*/

  /**
    Container for a list of menu entries.
  */
  class MenuBox : public List
  {
  public:
    MenuBox();

    void RegisterToParentWindow(OS::Window* parent);
    void CalcSize();
    void Layout();

    MenuEntry* GetSelected();
    MenuEntry* GetFirst() const;
    size_t     GetEntryCount() const;

    MenuEntry* GetPreviousSelectable(MenuEntry* entry);
    MenuEntry* GetNextSelectable(MenuEntry* entry);
    MenuEntry* GetShortcut(wchar_t sc);
  };

  MenuBox::MenuBox()
  {
    SetPrefs(menuPrefs);
  }

  void MenuBox::RegisterToParentWindow(OS::Window* parent)
  {
    std::list<Object*>::iterator iter;

    iter=list.begin();
    while (iter!=list.end()) {
      dynamic_cast<MenuEntry*>(*iter)->RegisterToParentWindow(parent);
      ++iter;
    }
  }

  void MenuBox::CalcSize()
  {
    size_t w1,w2,w3,w4;
    int x1,x2,x3,x4;

    w1=0;
    w2=0;
    w3=0;
    w4=0;
    height=0;

    for (std::list<Object*>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
      MenuEntry* entry=dynamic_cast<MenuEntry*>(*iter);

      entry->CalcSize();
      w1=std::max(w1,entry->w1);
      w2=std::max(w2,entry->w2);
      w3=std::max(w3,entry->w3);
      w4=std::max(w4,entry->w4);
      height+=entry->GetOHeight();
    }

    x1=x+OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
    width+=OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);

    // Image
    if (w1==0) {
      w1=OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    }
    x2=x1+w1+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    width+=w1+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);

    //Label
    if (w2==0) {
      w2=OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    }
    x3=x2+w2+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    width+=w2+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);

    // Shortcut
    if (w3==0) {
      w3=OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    }
    x4=x3+w3+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);
    width+=w3+OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject);

    // Sub Image
    if (w4==0) {
      w4=OS::display->GetSpaceHorizontal(OS::Display::spaceGroupIndent);
    }
    width+=w4+OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);

    for (std::list<Object*>::iterator iter=list.begin(); iter!=list.end(); ++iter) {
      MenuEntry* entry=dynamic_cast<MenuEntry*>(*iter);

      entry->w1=w1;
      entry->w2=w2;
      entry->w3=w3;
      entry->w4=w4;
      entry->SetOffsets(x1,x2,x3,x4);
    }

    minWidth=width;
    minHeight=height;

    List::CalcSize();
  }

  void MenuBox::Layout()
  {
    std::list<Object*>::iterator iter;
    int                          yPos;

    yPos=y;

    iter=list.begin();
    while (iter!=list.end()) {
      (*iter)->Move(x,yPos);
      (*iter)->Resize(width,(*iter)->GetOHeight()); // Does not work?

      yPos+=(*iter)->GetOHeight();

      ++iter;
    }

    List::Layout();
  }

  /**
    Get menu entry the mouse currently is over or NULL.
  */
  MenuEntry* MenuBox::GetSelected()
  {
    std::list<Object*>::iterator entry;
    int                            x,y;

    GetWindow()->GetMousePos(x,y);

    if (x>=0 && x<(int)GetWindow()->GetWidth()) {
      entry=list.begin();
      while (entry!=list.end()) {
        if (y>=(*entry)->GetOY() && y<=(*entry)->GetOY()+(int)(*entry)->GetOHeight()-1) {
          return dynamic_cast<MenuEntry*>(*entry);
        }

        ++entry;
      }
    }

    return NULL;
  }

  MenuEntry* MenuBox::GetFirst() const
  {
    if (list.begin()==list.end()) {
      return NULL;
    }
    else {
      return dynamic_cast<MenuEntry*>(*list.begin());
    }
  }

  size_t MenuBox::GetEntryCount() const
  {
    return list.size();
  }

  MenuEntry* MenuBox::GetPreviousSelectable(MenuEntry* entry)
  {
    std::list<Object*>::reverse_iterator end,current;

    if (list.size()==0) {
      return NULL;
    }

    if (entry==NULL) {
      end=list.rend();
      current=list.rbegin();

    }
    else {
      end=list.rbegin();
      while (*end!=entry) {
        ++end;
      }
      current=end;
      ++current;
    }

    while (current!=end) {
      if (current!=list.rend()) {
        if (dynamic_cast<MenuEntry*>(*current)->IsSelectable()) {
          return dynamic_cast<MenuEntry*>(*current);
        }

        ++current;
      }
      else {
        current=list.rbegin();
      }
    }

    return entry;
  }

  MenuEntry* MenuBox::GetNextSelectable(MenuEntry* entry)
  {
    std::list<Object*>::iterator end,current;

    if (list.size()==0) {
      return NULL;
    }

    if (entry==NULL) {
      end=list.end();
      current=list.begin();

    }
    else {
      end=list.begin();
      while (*end!=entry) {
        ++end;
      }
      current=end;
      ++current;
    }

    while (current!=end) {
      if (current!=list.end()) {
        if (dynamic_cast<MenuEntry*>(*current)->IsSelectable()) {
          return dynamic_cast<MenuEntry*>(*current);
        }

        ++current;
      }
      else {
        current=list.begin();
      }
    }

    return entry;
  }

  MenuEntry* MenuBox::GetShortcut(wchar_t sc)
  {
    std::list<Object*>::iterator iter;

    iter=list.begin();
    while (iter!=list.end())  {
      if (toupper(dynamic_cast<MenuEntry*>(*iter)->sc)==toupper(sc)) {
        return dynamic_cast<MenuEntry*>(*iter);
      }

      ++iter;
    }

    return NULL;
  }

  /* ---------- Menu stuff ------------------*/

  Menu::Menu()
  : window(OS::driver->CreateWindow()),parent(NULL),
    popup(true),registered(false),reference(NULL),
    parentMenu(NULL),child(NULL),selected(NULL),strip(NULL)
  {
    //m.prefs:=prefs;   (* We set the prefs *)

    window->SetType(OS::Window::typeMenu);
    window->SetMaster(this);

    top=new MenuBox();
    top->SetWindow(window);

    AttachModel(window->GetUnmapedAction());
    AttachModel(window->GetHiddenAction());
    AttachModel(window->GetRedrawAction());
    AttachModel(window->GetMouseOutAction());
    AttachModel(window->GetPreInitAction());
    AttachModel(window->GetEventAction());
  }

  Menu::~Menu()
  {
    delete top;

    UnattachModel(window->GetEventAction());
    UnattachModel(window->GetPreInitAction());
    UnattachModel(window->GetMouseOutAction());
    UnattachModel(window->GetRedrawAction());
    UnattachModel(window->GetHiddenAction());
    UnattachModel(window->GetUnmapedAction());

    delete window;
  }

  bool Menu::IsOpen() const
  {
    return window->IsOpen();
  }

  bool Menu::CursorIsIn() const
  {
    return window->CursorIsIn();
  }

  /**
    Private. This method is overloaded to get the parent window once.
    We need the parent window to register our shortcuts.
  */
  void Menu::SetParent(OS::Window* parent)
  {
    if (!registered) {
      top->RegisterToParentWindow(parent);
      registered=true;
      this->parent=parent;
      window->SetParent(parent);
    }
  }

  void Menu::SetReference(Lum::Object* reference)
  {
    this->reference=reference;
  }

  bool Menu::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    return top->VisitChildren(visitor,onlyVisible);
  }

  /**
    Initializes an empty menu.
  */
  void Menu::Add(MenuEntry* entry)
  {
    entry->SetFlex(true,false);
    entry->SetWindow(window);
    entry->menuWindow=this;

    top->Add(entry);
  }

  void Menu::CalcSize()
  {
    top->CalcSize();

    window->SetSize(top->GetOWidth(),top->GetOHeight());
  }

  Lum::Object* Menu::GetObjectByName(const std::wstring& /*name*/)
  {
    return NULL; //return top->GetObjectByName(name);
  }

  /**
    Get menu entry the mouse currently is over or NULL.
  */
  MenuEntry* Menu::GetSelected()
  {
    return top->GetSelected();
  }

  /**
    Return menu strip or @code{NULL}, if there is no menu strip (popup menu).
  */
  MenuStrip* Menu::GetStrip()
  {
    Menu *current;

    current=this;
    while (current!=NULL) {
      if (current->strip!=NULL) {
        return current->strip;
      }

      if (current->parentMenu!=NULL) {
        current=dynamic_cast<Menu*>(current->parentMenu->GetSubMenu());
      }
      else {
        current=NULL;
      }
    }

    return NULL;
  }

  /**
    Return top most menu.
  */
  Menu* Menu::GetTop()
  {
    Menu *current;

    current=this;
    while (current!=NULL && current->parentMenu!=NULL) {
      current=dynamic_cast<Menu*>(current->parentMenu->GetMenu());
    }

    return current;
  }

  Menu* Menu::GetMenuWithSelection()
  {
    Menu *menu;

    menu=this;
    while (menu->parentMenu!=NULL) {
      menu=menu->parentMenu->menuWindow;
    }

    while (menu!=NULL && menu->IsOpen() && menu->selected!=NULL && menu->selected->GetSubMenu()!=NULL && menu->selected->GetSubMenu()->selected!=NULL) {
      menu=menu->selected->GetSubMenu();
    }

    return menu;
  }

  void Menu::PreInit()
  {
    OS::Window *help;
    int        rx,ry,x,y;

    CalcSize();

    selected=NULL;

    if (strip!=NULL && reference!=NULL) {
      help=reference->GetWindow();
      x=help->GetX();
      y=help->GetY();
      x+=reference->GetOX();
      y+=reference->GetOY()+reference->GetOHeight();
      window->Grab(true);
    }
    else if (parentMenu!=NULL) { /* submenu */
      x=parentMenu->GetWindow()->GetX()+parentMenu->GetWindow()->GetWidth();
      y=parentMenu->GetWindow()->GetY()+parentMenu->GetOY();
    }
    else if (parent!=NULL) {
      OS::display->GetMousePos(rx,ry);
      if (popup) {
        x=rx;//+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject);
        y=ry;//+OS::display->GetSpaceVertical(OS::Display::spaceLabelObject);
      }
      else {
        x=rx-selected->GetOX()-selected->GetOWidth() / 2;
        y=ry-selected->GetOY()-selected->GetOHeight() / 2;
      }
    }
    else {
      x=window->GetX();
      y=window->GetY();
    }

    if (strip==NULL) {
      window->Grab(true);
    }

    x=Base::RoundRange(x,0,OS::display->GetScreenWidth()-1-window->GetWidth());
    y=Base::RoundRange(y,0,OS::display->GetScreenHeight()-1-window->GetHeight());

    window->SetPos(x,y);
    window->SetPosition(OS::Window::positionManual,OS::Window::positionManual);
  }

  bool Menu::Open()
  {
    return window->Open(true);
  }

  /**
    Close menu.
  */
  void Menu::Close()
  {
    if (child!=NULL) {
      child->Close();
    }
    if (parentMenu!=NULL) {
      parentMenu->menuWindow->child=NULL;
    }

    window->Close();
  }

  /**
    Close current and all parent menues.
  */
  void Menu::CloseAll()
  {
    Menu      *current;
    MenuStrip *strip;

    current=this;
    while (current!=NULL) {
      current->Close();
      if (current->parentMenu!=NULL) {
        current=dynamic_cast<Menu*>(current->parentMenu->GetMenu());
      }
      else {
        current=NULL;
      }
    }

    strip=GetStrip();
    if (strip!=NULL) {
      strip->SetSelection(NULL);
    }
  }

  /**
    Close down menu and finish menu selection.
  */
  void Menu::EndAction()
  {
    MenuStrip *strip;

    strip=GetStrip();

    if (strip!=NULL) {
      strip->stickyMode=false;
    }

    CloseAll();
  }

  void Menu::Select(MenuEntry* entry)
  {
    MenuEntry* old;

    old=selected;
    if (entry!=old) {
      if ((entry!=NULL) && !entry->IsSelectable()) {
        entry=NULL;
      }

      selected=entry;

      if (old!=NULL) {
        old->Redraw();
      }

      if (entry!=NULL) {
        entry->Redraw();
      }
    }
  }

  /**
    Select next menu item in menu.
  */
  void Menu::SelectDown()
  {
    Menu *menu;

    menu=GetMenuWithSelection();

    if (menu->top->GetEntryCount()==0 || (menu->top->GetEntryCount()==1 && menu->selected==menu->top->GetFirst())) {
      /* nothing to do */
      return;
    }

    menu->Select(menu->top->GetNextSelectable(menu->selected));
  }

  /**
    Select previous menu item in menu.
  */
  void Menu::SelectUp()
  {
    Menu *menu;

    menu=GetMenuWithSelection();

    if (menu->top->GetEntryCount()==0 || (menu->top->GetEntryCount()==1 && menu->selected==menu->top->GetFirst())) {
      /* nothing to do */
      return;
    }

    menu->Select(menu->top->GetPreviousSelectable(menu->selected));
  }

  /**
   Select parent menu if available or previous entry in menu strip.
  */
  void Menu::SelectLeft()
  {
    Menu *menu;

    menu=GetMenuWithSelection();

    if (menu->parentMenu!=NULL) {
      menu->Select(NULL);
    }
    else {
      MenuStrip* strip;

      strip=GetStrip();
      if (strip!=NULL) {
        strip->SelectPrevious();
      }
    }
  }

  /**
    Select sub menu if available or next entry in menu strip.
  */
  void Menu::SelectRight()
  {
    Menu *menu;

    menu=GetMenuWithSelection();

    if (menu->selected!=NULL && menu->selected->GetSubMenu()!=NULL) {
      menu=menu->selected->GetSubMenu();
      menu->Select(menu->top->GetNextSelectable(NULL));
      //menu->selected->GetSubMenu()->SelectDown();
    }
    else {
      MenuStrip* strip;

      strip=GetStrip();
      if (strip!=NULL) {
        strip->SelectNext();
      }
    }
  }
  /**
    Select menu item with the given hotkey.
*/
  void Menu::SelectKey(wchar_t key)
  {
    Menu *menu;

    menu=GetMenuWithSelection();

    if (menu!=NULL) {
      MenuEntry* entry;

      entry=menu->top->GetShortcut(key);

      if (entry!=NULL) {
        if (entry->IsSelectable()) {
          menu->EndAction();
          entry->OnSelection();
        }
        else {
          menu->Select(entry);
        }
      }
    }
  }

  /* ---------------------------------- */

  bool Menu::HandleEvent(OS::Event* event)
  {
    Menu           *menu;
    Object         *object;
    MenuStrip      *strip;
    OS::MouseEvent *mouse;
    OS::KeyEvent   *key;

    if ((mouse=dynamic_cast<OS::MouseEvent*>(event))!=NULL) {
      if (mouse->type==OS::MouseEvent::up && (mouse->button==OS::MouseEvent::button3 || mouse->button==OS::MouseEvent::button1)) {
        menu=GetTop();
        while (menu!=NULL && menu->IsOpen() && menu->selected!=NULL && !menu->CursorIsIn()) {
          menu=menu->selected->GetSubMenu();
        }

        if (menu!=NULL && menu->CursorIsIn()) {
          menu->Select(menu->GetSelected());

          if (menu->selected!=NULL && menu->selected->IsSelectable()) {
            EndAction();
            menu->selected->OnSelection();
          }
        }
        else {
          /* No menu item selected */
          strip=GetStrip();
          if (strip!=NULL && !strip->stickyMode && strip->MouseIsIn()) {
            object=strip->GetMouseSelected();
            if ((object!=NULL) && (dynamic_cast<PullDownMenu*>(object)->subMenu=GetTop())) {
              strip->stickyMode=true;
            }
          }
          else {
            EndAction();
          }
        }

      }
      else if (mouse->type==OS::MouseEvent::down && (mouse->button==OS::MouseEvent::button3 || mouse->button==OS::MouseEvent::button1)) {
        menu=GetTop();
        while (menu!=NULL && menu->IsOpen() && menu->selected!=NULL && !menu->CursorIsIn()) {
          menu=menu->selected->GetSubMenu();
        }

        if (menu==NULL || !menu->CursorIsIn()) {
          EndAction();
        }
      }
      else if (mouse->type==OS::MouseEvent::move) {
        /*
          We got a motion event. We first go to the top of the menu tree
          and then walk down the menu selection tree until we
          find a menu the mouse is currently over.
        */
        menu=GetTop();
        while (menu!=NULL && menu->IsOpen() && menu->selected!=NULL && !menu->CursorIsIn()) {
          menu=menu->selected->GetSubMenu();
        }

        if (menu!=NULL) { /* We found a submenu */
          if (menu->CursorIsIn()) {
            /*
              Select the select entry
            */
            menu->Select(menu->GetSelected());
          }
          else if (this->strip!=NULL && this->strip->MouseIsIn()) {
            /*
              If the mouse is over the strip, check if a new pulldown
              menu has been selected. If, close the old one.
            */
            object=this->strip->GetMouseSelected();
            if (object!=NULL && reference!=object) {
              CloseAll();
            }
          }
        }
        else {
          /*
            The mouse is currently over no menu. Now, find the leaf of the current
            menu selection tree.
          */
          menu=GetMenuWithSelection();

          if (menu!=NULL && menu->selected!=NULL && menu->selected->GetSubMenu()==NULL) {
            /*
              Deselect the last selected entry, because it is not
              selected by the mouse anymore.

              This code sequence is only necessary for driver that do not handle
              OnMouseLeft!
            */
            menu->Select(NULL);
          }
        }
      }
    }
    else if ((key=dynamic_cast<OS::KeyEvent*>(event))!=NULL && key->type==OS::KeyEvent::down) {
      switch (key->key) {
      case OS::keyEscape:
        EndAction();
        break;
      case OS::keyReturn:
        if (selected!=NULL && selected->IsSelectable()) {
          EndAction();
          selected->OnSelection();
        }
        break;
      case OS::keyUp:
        SelectUp();
        break;
      case OS::keyDown:
        SelectDown();
        break;
      case OS::keyLeft:
        SelectLeft();
        break;
      case OS::keyRight:
        SelectRight();
        break;
      default:
        if (key->text[0]!='\0') {
          SelectKey(key->text[0]);
        }
        break;
      }
    }

    return true;
  }

  void Menu::Resync(Lum::Base::Model* model,
                    const Lum::Base::ResyncMsg& msg)
  {
    if (model==window->GetUnmapedAction() && window->GetUnmapedAction()->IsFinished()) {
      top->Hide();
    }
    else if (model==window->GetHiddenAction() && window->GetHiddenAction()->IsFinished()) {
      CloseAll();
    }
    else if (model==window->GetMouseOutAction() && window->GetMouseOutAction()->IsFinished()) {
      if (selected!=NULL && selected->GetSubMenu()==NULL) {
        MenuEntry *oldSelected;

        oldSelected=selected;
        selected=NULL;
        oldSelected->Redraw();
      }
    }
    else if (model==window->GetRedrawAction() && window->GetRedrawAction()->IsFinished()) {
      const OS::Window::RedrawMsg* redrawMsg=dynamic_cast<const OS::Window::RedrawMsg*>(&msg);

      top->Move(0,0);
      top->Draw(redrawMsg->x,redrawMsg->y,redrawMsg->width,redrawMsg->height);
    }
    else if (model==window->GetPreInitAction() && window->GetPreInitAction()->IsFinished()) {
      PreInit();
    }
    else if (model==window->GetEventAction() && window->GetEventAction()->IsFinished()) {
      const OS::Window::EventMsg *eventMsg=dynamic_cast<const OS::Window::EventMsg*>(&msg);

      HandleEvent(eventMsg->event);
    }
  }

  OS::Window* Menu::GetWindow() const
  {
    return window;
  }

  void Menu::AddActionItem(const std::wstring& label,
                           Model::Action* action,
                           unsigned long qualifier,
                           const std::wstring& key)
  {
    ActionItem *item;

    item=new ActionItem();
    item->SetTextLabel(label);

    if (!key.empty()) {
      item->SetShortcut(qualifier,key);
    }
    item->SetModel(action);

    Add(item);
  }

  void Menu::AddBoolItem(const std::wstring& label,
                         Model::Boolean* model,
                         unsigned long qualifier,
                         const std::wstring& key)
  {
    BoolItem *item;

    item=new BoolItem();
    item->SetTextLabel(label);

    if (!key.empty()) {
      item->SetShortcut(qualifier,key);
    }
    item->SetModel(model);

    Add(item);
  }

  void Menu::AddRadioItem(const std::wstring& label,
                         Model::SizeT* model,
                         size_t value,
                         unsigned long qualifier,
                         const std::wstring& key)
  {
    RadioItem *item;

    item=new RadioItem();
    item->SetTextLabel(label);

    if (!key.empty()) {
      item->SetShortcut(qualifier,key);
    }
    item->SetModel(model);
    item->SetValue(value);

    Add(item);
  }

  void Menu::AddSeparator()
  {
    Add(new SeparatorItem());
  }

  void Menu::AddSubMenu(Lum::Object* label, Menu* subMenu)
  {
    SubMenu* sub;

    sub=new SubMenu();
    sub->SetLabel(label);
    sub->SetSubMenu(subMenu);

    Add(sub);
  }

  void Menu::AddSubMenu(const std::wstring& label, Menu* subMenu)
  {
    AddSubMenu(new Text(label),subMenu);
  }
}

