#ifndef LUM_BUTTON_H
#define LUM_BUTTON_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/Size.h>

#include <Lum/Model/Action.h>

#include <Lum/Object.h>

namespace Lum {

  /**
    Implements a powerfull general purpose button class with
    "action on press" semantic. The button label can either be
    text or any other class derived from @otype{G.Object}.

    The button can be on of a number of types that slidly differ
    in their apearance.

    Instead of the "one action on press" semantic, it can be
    configured to retrigger until the button is deselected again.
  */
  class LUMAPI Button : public Control
  {
  public:
    enum Type {
      typeNormal,  //! normal button
      typeDefault,
      typeCommit,
      typeCancel,
      typeImage,   //! A button that holds an image
      typeToolBar, //! a tool bar button
      typeScroll
    };

  public:
    class Prefs : public Object::Prefs
    {
    public:
      OS::FillRef  bgReturn;
      OS::FillRef  bgEscape;
      OS::FillRef  bgDefault;
      OS::FillRef  bgToolbar;
      OS::FillRef  bgScroll;
      Base::Size   sHSpace;
      Base::Size   sVSpace;
      OS::FrameRef returnFrame;
      OS::FrameRef escapeFrame;
      OS::FrameRef defaultFrame;
      OS::FrameRef toolbarFrame;
      OS::FrameRef scrollFrame;
      OS::FrameRef iFrame;

    public:
      Prefs();
      void Initialize();
    };

  private:
    Object           *object;       //! The label of the button
    OS::ImageRef     image;         //! Optional image (depending on type) to be displayed with the object
    OS::FontRef      font;          //! Font used
    Type             type;          //! Type
    bool             pulse;         //! Button creates repeately press events while clicked
    wchar_t          shortCut;
    bool             scAssigned;
    Model::ActionRef model;         //! The action to trigger on press
    int              imageX,imageY; //! Position of image

  public:
    Button();
    Button(Object* object);
    Button(const std::wstring& text);
    ~Button();

    bool HasBaseline() const;
    size_t GetBaseline() const;

    void SetFont(OS::Font* font);
    void SetLabel(Object* object);
    void SetImage(OS::ImageRef& image);
    void SetText(const std::wstring& string);
    void SetType(Type type);
    void SetPulse(bool pulse);

    bool SetModel(Base::Model* model);

    void CalcSize();
    void Layout();
    void Draw(int x, int y, size_t w, size_t h);
    void Hide();

    bool HandleMouseEvent(const OS::MouseEvent& event);
    bool HandleKeyEvent(const OS::KeyEvent& event);

    void Resync(Base::Model* model, const Base::ResyncMsg& msg);
  };
}

#endif
