/*
  This source is part of the Illumination library
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/X11/HildonKeyboardHandler.h>

#include <Lum/Base/String.h>

#include <Lum/OS/X11/Display.h>
#include <Lum/OS/X11/Event.h>
#include <Lum/OS/X11/Window.h>

#include <iostream>

namespace Lum {
  namespace OS {
    namespace X11 {

      HildonKeyboardHandler::HildonKeyboardHandler()
      : window(NULL),
        object(NULL)
      {
        // no code
      }

      HildonKeyboardHandler::~HildonKeyboardHandler()
      {
        // no code
      }

      void HildonKeyboardHandler::OnWindowFocusIn(OS::Window* window)
      {
        this->window=window;
      }

      void HildonKeyboardHandler::OnWindowFocusOut(OS::Window* window)
      {
        if (this->window!=window) {
          return;
        }

        HideKeyboard();

        this->window=NULL;
        object=NULL;
      }

      void HildonKeyboardHandler::OnFocusChange(OS::Window* window,
                                                Lum::Object* oldObject,
                                                Lum::Object* newObject)
      {
        if (this->window!=window) {
          return;
        }

        if (object==newObject) {
          return;
        }

        if (object!=NULL && object->RequestsKeyboard() &&
            newObject!=NULL && newObject->RequestsKeyboard()) { // "do nothing" case
          object=newObject;
        }
        else if (newObject!=NULL && newObject->RequestsKeyboard()) { // "show" case
          object=newObject;
          ShowKeyboard();
        }
        else { // "hide" case
          object=NULL;
          HideKeyboard();
        }
      }

      bool HildonKeyboardHandler::HandleEvent(OS::Window* window, OS::Event* event)
      {
        XEvent x11Event;

        GetX11Event(event,x11Event);

        if (x11Event.type!=ClientMessage) {
          return false;
        }

        if (this->window!=window) {
          return false;
        }

        Display *display=dynamic_cast<Display*>(Lum::OS::display);

        if (display==NULL) {
          return false;
        }

        ::Window imWindow=GetKeyboardWindow();

        if (imWindow==None) {
          return false;
        }

        if (x11Event.xclient.message_type==XInternAtom(display->display,"_HILDON_IM_COM",false)) {
          if (x11Event.xclient.data.l[1]==0x00) { // ENTER
            KeyEvent keyEvent;

            keyEvent.key=keyReturn;
            keyEvent.qualifier=0;
            keyEvent.type=KeyEvent::down;

            window->HandleEvent(&keyEvent);

            keyEvent.type=KeyEvent::up;

            window->HandleEvent(&keyEvent);
            return true;
          }
          else if (x11Event.xclient.data.l[1]==0x01) { // TAB
            KeyEvent keyEvent;

            keyEvent.key=keyTab;
            keyEvent.qualifier=0;
            keyEvent.type=KeyEvent::down;

            window->HandleEvent(&keyEvent);

            keyEvent.type=KeyEvent::up;

            window->HandleEvent(&keyEvent);
            return true;
          }
          else if (x11Event.xclient.data.l[1]==0x02) { // BACKSPACE
            KeyEvent keyEvent;

            keyEvent.key=keyBackspace;
            keyEvent.qualifier=0;
            keyEvent.type=KeyEvent::down;

            window->HandleEvent(&keyEvent);

            keyEvent.type=KeyEvent::up;

            window->HandleEvent(&keyEvent);
            return true;
          }
          else if (x11Event.xclient.data.l[1]==0x03) { // SPACE
            KeyEvent keyEvent;

            keyEvent.key=keySpace;
            keyEvent.qualifier=0;
            keyEvent.type=KeyEvent::down;

            window->HandleEvent(&keyEvent);

            keyEvent.type=KeyEvent::up;

            window->HandleEvent(&keyEvent);
            return true;
          }
          else if (x11Event.xclient.data.l[1]==0x04) {
            //std::cout << "CONFIRM_SENTENCE_START" << std::endl;

            ::XEvent event;

            memset(&event,0,sizeof(event));
            event.xclient.type=ClientMessage;
            event.xclient.window=imWindow;
            event.xclient.message_type=XInternAtom(display->display,"_HILDON_IM_ACTIVATE",False);
            event.xclient.format=8;
            event.xclient.data.l[0]=dynamic_cast<X11::Window*>(window)->GetDrawable();
            event.xclient.data.l[1]=dynamic_cast<X11::Window*>(window)->GetDrawable();
            event.xclient.data.l[2]=0x03;
            event.xclient.data.l[3]=0x00000007;
            event.xclient.data.l[4]=0;
            if (XSendEvent(display->display,imWindow,False,NoEventMask,&event)==0) {
              std::cerr << "Error while sending vk-r-1" << std::endl;
            }
          }
          else if (x11Event.xclient.data.l[1]==0x05) {
            //std::cout << "FLUSH_PREEDIT" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x06) {
            //std::cout << "BUFFERED_MODE" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x07) {
            //std::cout << "DIRECT_MODE" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x08) {
            //std::cout << "REDIRECT_MODE" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x09) {
            //std::cout << "SURROUNDING_MODE" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0a) {
            //std::cout << "CLIPBOARD_COPY" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0b) {
            //std::cout << "CLIPBOARD_CUT" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0c) {
            //std::cout << "CLIPBOARD_PASTE" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0d) {
            //std::cout << "CLIPBOARD_SELECTION_QUERY" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0e) {
            //std::cout << "REQUEST_SURROUNDING" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x0f) {
            //std::cout << "WIDGET_CHANGED" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x10) {
            //std::cout << "OPTION_CHANGED" << std::endl;
          }
          else if (x11Event.xclient.data.l[1]==0x11) {
            //std::cout << "CLEAR_STICKY" << std::endl;
          }
          else {
            std::cout << "got HILDON-COM " << std::hex << x11Event.xclient.data.l[1] << std::endl;
          }
        }
        else if (x11Event.xclient.message_type==XInternAtom(display->display,"_HILDON_IM_INSERT_UTF8",false)) {
          KeyEvent     keyEvent;
          std::string  inputText;
          size_t       i;

          // Decode text
          i=4;
          while (i<20 && x11Event.xclient.data.b[i]!='\0') {
            inputText.append(1,(char)x11Event.xclient.data.b[i]);
            i++;
          }

          // Fake key event
          keyEvent.key=keyUnknown;
          keyEvent.qualifier=0;
          keyEvent.text=Lum::Base::UTF8ToWString(inputText);
          keyEvent.type=KeyEvent::down;

          window->HandleEvent(&keyEvent);

          keyEvent.type=KeyEvent::up;

          window->HandleEvent(&keyEvent);
        }

        return false;
      }

      void HildonKeyboardHandler::ShowKeyboard()
      {
        assert(window!=0);

        Display *display=dynamic_cast<Display*>(Lum::OS::display);

        if (display==NULL) {
          return;
        }

        ::Window imWindow=GetKeyboardWindow();

        if (imWindow==None) {
          return;
        }

        ::XEvent event;

        memset(&event,0,sizeof(event));
        event.xclient.type=ClientMessage;
        event.xclient.window=imWindow;
        event.xclient.message_type=XInternAtom(display->display,"_HILDON_IM_ACTIVATE",False);
        event.xclient.format=8;
        event.xclient.data.l[0]=dynamic_cast<X11::Window*>(window)->GetDrawable();
        event.xclient.data.l[1]=dynamic_cast<X11::Window*>(window)->GetDrawable();
        event.xclient.data.l[2]=0x08; //SETNSHOW
        event.xclient.data.l[3]=(1 << 0) + (1 << 1) + (1<< 2) + (1 << 31);
        event.xclient.data.l[4]=0;
        if (XSendEvent(display->display,imWindow,False,NoEventMask,&event)==0) {
          std::cerr << "Error while sending OpenKeyboard" << std::endl;
        }

        XSync(display->display,false);

        if (!display->WaitTimedForX11Event(dynamic_cast<X11::Window*>(window)->GetDrawable(),
                                           ClientMessage,event)) {
          std::cerr << "Keyboard did not open in time!" << std::endl;
          return;
        }
      }

      void HildonKeyboardHandler::HideKeyboard()
      {
        assert(window!=0);

        Display *display=dynamic_cast<Display*>(Lum::OS::display);

        if (display==NULL) {
          return;
        }

        ::Window imWindow=GetKeyboardWindow();

        if (imWindow==None) {
          return;
        }

        ::XEvent event;

        memset(&event,0,sizeof(event));
        event.xclient.type=ClientMessage;
        event.xclient.window=imWindow;
        event.xclient.message_type=XInternAtom(display->display,"_HILDON_IM_ACTIVATE",False);
        event.xclient.format=8;
        event.xclient.data.l[0]=0;
        event.xclient.data.l[1]=0;
        event.xclient.data.l[2]=0x02;
        event.xclient.data.l[3]=(1 << 0) + (1 << 1) + (1<< 2) + (1 << 31);
        event.xclient.data.l[4]=0;
        if (XSendEvent(display->display,imWindow,False,NoEventMask,&event)==0) {
          std::cerr << "Error while sending 'CloseKeyboard'" << std::endl;
        }

        XSync(display->display,false);
      }

      ::Window HildonKeyboardHandler::GetKeyboardWindow() const
      {
        Display *display=dynamic_cast<Display*>(Lum::OS::display);

        if (display==NULL) {
          return None;
        }

        Atom hildonWindow=XInternAtom(display->display,"_HILDON_IM_WINDOW",true);

        if (hildonWindow==None) {
          return None;
        }

        Atom           atomType;
        int            atomFormat;
        int            res;
        unsigned long  atomItems;
        unsigned long  atomBytes;
        unsigned char* atomProperties;
        ::Window       window;

        res=XGetWindowProperty(display->display,RootWindow(display->display,display->scrNum),
                               hildonWindow,0,1,false,display->atoms[display->windowAtom],
                               &atomType,&atomFormat,&atomItems,&atomBytes,&atomProperties);
        if (res!=Success || atomProperties==NULL) {
          std::cerr << "Cannot get IM window" << std::endl;
          return None;
        }

        window=reinterpret_cast< ::Window*>(atomProperties)[0];

        XFree(atomProperties);

        return window;
      }

    }
  }
}

