/*
  This source is part of the Illumination library
  Copyright (C) 2005  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/X11/DrawInfo.h>

#include <assert.h>

#include <unistd.h>

#include <X11/X.h>
#include <X11/Xatom.h>

#include <Lum/Base/String.h>

#include <Lum/Images/Image.h>

#include <Lum/OS/X11/Bitmap.h>
#include <Lum/OS/X11/Font.h>
#include <Lum/OS/X11/Image.h>
#include <Lum/OS/X11/Window.h>

namespace Lum {
  namespace OS {
    namespace X11 {

      /**
        Initialize an instance of the  DrawInfo class.
      */
      DrawInfo::DrawInfo(Lum::OS::Window* window)
      : ::Lum::OS::Base::DrawInfo(window)
#ifdef HAVE_LIB_GTK
        ,gdkPixmap(NULL)
        ,gdkWindow(NULL)
#endif
      {
        this->drawable=dynamic_cast<Window*>(window)->GetDrawable();
        display=dynamic_cast<Window*>(window)->GetDisplay();

        Initialize();
      }

      DrawInfo::DrawInfo(Lum::OS::Bitmap* bitmap)
      : ::Lum::OS::Base::DrawInfo(bitmap)
 #ifdef HAVE_LIB_GTK
        ,gdkPixmap(NULL)
        ,gdkWindow(NULL)
#endif

      {
        drawable=dynamic_cast<Bitmap*>(bitmap)->GetDrawable();
        display=dynamic_cast<Bitmap*>(bitmap)->GetDisplay();

        Initialize();

#ifdef HAVE_LIB_GTK
        gdkPixmap=gdk_pixmap_foreign_new(drawable);
#endif
      }

      /**
        Deinitializes the drawInfo
      */
      DrawInfo::~DrawInfo()
      {
        /*
        ASSERT(d.fPenPos==-1);
        ASSERT(d.fontPos==-1);
        ASSERT(d.styleStack==NULL);
        ASSERT(d.dashStack==NULL);
        ASSERT(d.clipPos==-1);
        ASSERT(d.patternStack==NULL);
        ASSERT(d.modeStack==NULL);
        */
#if defined(HAVE_LIB_XFT2)
        XftDrawDestroy(xftgc);
#endif

#ifdef HAVE_LIB_GTK
        if (gdkPixmap!=NULL) {
          g_object_unref(gdkPixmap);
          gdkPixmap=NULL;
        }

        if (gdkWindow!=NULL) {
          g_object_unref(gdkWindow);
          gdkWindow=NULL;
        }
#endif

        XFreeGC(display,gc);
      }

      void DrawInfo::Initialize()
      {
        unsigned long mask;
        XGCValues     values;

        mask=0;
        gc=XCreateGC(display,drawable,mask,&values);

#if defined(HAVE_LIB_XFT2)
        xftgc=XftDrawCreate(display,drawable,
                            dynamic_cast<Display*>(::Lum::OS::display)->visual,
                            dynamic_cast<Display*>(::Lum::OS::display)->colorMap);
#endif

        XSetArcMode(display,gc,ArcPieSlice);
      }

      void DrawInfo::OnMap()
      {
#ifdef HAVE_LIB_GTK
        if (gdkPixmap==NULL) {
          gdkWindow=gdk_window_foreign_new(drawable);
        }
#endif
      }

      void DrawInfo::OnUnmap()
      {
#ifdef HAVE_LIB_GTK
        if (gdkWindow!=NULL) {
          g_object_unref(gdkWindow);
          gdkWindow=NULL;
        }
#endif
      }

      void DrawInfo::PushClip(size_t x, size_t y, size_t width, size_t height)
      {
        ClipEntry entry;

        entry.Init(this);

        clipStack.push_back(entry);

        clipStack.back().Add(x,y,width,height);
        RecalcClipRegion();
      }

      void DrawInfo::PushClipBegin(size_t x, size_t y, size_t width, size_t height)
      {
        ClipEntry entry;

        entry.Init(this);

        clipStack.push_back(entry);

        clipStack.back().Add(x,y,width,height);
      }

      void DrawInfo::AddClipRegion(size_t x, size_t y, size_t width, size_t height)
      {
        clipStack.back().Add(x,y,width,height);
      }

      void DrawInfo::SubClipRegion(size_t x, size_t y, size_t width, size_t height)
      {
        clipStack.back().Sub(x,y,width,height);
      }

      void DrawInfo::PushClipEnd()
      {
        RecalcClipRegion();
      }

      void DrawInfo::PopClip()
      {
        clipStack[clipStack.size()-1].Free();

        clipStack.pop_back();

        RecalcClipRegion();
      }

      /**
        Recalces the current clipping regions by analysing the
        current clipping stack.
       */
      void DrawInfo::RecalcClipRegion()
      {
        int    pos;
        Region tmp;

        tmp=XCreateRegion();

        if (clipStack.size()>0) {
          XUnionRegion(clipStack.back().region,tmp,tmp);

          if (clipStack.size()>1) {
            pos=clipStack.size()-2;
            while (pos>=0) {
              XIntersectRegion(clipStack[pos].region,tmp,tmp);
              --pos;
            }
          }
        }
        else {
          XRectangle rect;

          rect.x=0;
          rect.y=0;
          rect.width=32000;  // TODO
          rect.height=32000; // TODO

          XUnionRectWithRegion(&rect,tmp,tmp);
        }

        XSetRegion(display,gc,tmp);
#if defined(HAVE_LIB_XFT2)
        XftDrawSetClip(xftgc,tmp);
#endif
        XDestroyRegion(tmp);
      }

      /**
      Initializes a clip entry object.
      */
      void DrawInfo::ClipEntry::Init(DrawInfo *draw)
      {
        region=XCreateRegion();
        this->draw=draw;
      }

      /**
      Frees the given clip entry.
      */
      void DrawInfo::ClipEntry::Free()
      {
        XDestroyRegion(region);
      }

      /**
      adds the given rectangle to the current clip entry.
      */
      void DrawInfo::ClipEntry::Add(int x, int y, int width, int height)
      {
        XRectangle rect;

        rect.x=x;
        rect.y=y;
        rect.width=width;
        rect.height=height;

        XUnionRectWithRegion(&rect,region,region);
      }

      /**
        adds the given rectangle to the current clip entry.
      */
      void DrawInfo::ClipEntry::Sub(int x, int y, int width, int height)
      {
        XRectangle rect;
        Region     tmp;

        rect.x=x;
        rect.y=y;
        rect.width=width;
        rect.height=height;

        tmp=XCreateRegion();
        XUnionRectWithRegion(&rect,tmp,tmp);
        XSubtractRegion(region,tmp,region);
        XDestroyRegion(tmp);
      }

      void DrawInfo::ReinstallClip()
      {
        if (clipStack.size()>0) {
          RecalcClipRegion();
        }
      }

      void DrawInfo::PushFont(::Lum::OS::Font *font, unsigned long style)
      {
        assert(font!=NULL);

        if (fontStack.size()>0 && fontStack.top().font.Get()==font && fontStack.top().style==style) {
          fontStack.top().count++;
        }
        else {
          FontEntry     entry;
          Font          *f;
          unsigned long pos;

          entry.font=font;
          entry.style=style;
          entry.count=1;

          f=dynamic_cast<Font*>(font);
          pos=f->GetFontPos(style);

          fontStack.push(entry);

          if (f->fonts[pos]!=NULL) {
            XSetFont(display,gc,f->fonts[pos]->fid);
          }
        }
      }

      void DrawInfo::PopFont()
      {
        assert(fontStack.size()>0);

        if (fontStack.top().count>1) {
          fontStack.top().count--;
        }
        else {
          fontStack.pop();
          if (fontStack.size()>0) {
            Font          *font=dynamic_cast<Font*>(fontStack.top().font.Get());
            unsigned long pos=font->GetFontPos(fontStack.top().style);

            if (font->fonts[pos]!=NULL) {
              XSetFont(display,gc,font->fonts[pos]->fid);
            }
          }
        }
      }

      void DrawInfo::DrawString(int x, int y, const std::wstring& text, size_t start, size_t length)
      {
        bool handled=false;

        unsigned long pos;
        Font          *font;
#if defined(HAVE_LIB_XFT2)
        XftColor      color;
        XColor        xcolor;
#endif
        assert(fontStack.size()>0);

        font=dynamic_cast<Font*>(fontStack.top().font.Get());
        pos=font->GetFontPos(fontStack.top().style);

#if defined(HAVE_LIB_XFT2)
        if (font->xftfonts[pos]!=NULL) {
          xcolor.pixel=fPenStack.top().color;
          XQueryColor(dynamic_cast<Display*>(OS::display)->display,
                      dynamic_cast<Display*>(OS::display)->colorMap,
                      &xcolor);

          color.pixel=xcolor.pixel;
          color.color.red=xcolor.red;
          color.color.blue=xcolor.blue;
          color.color.green=xcolor.green;
          color.color.alpha=(short unsigned int)-1; // TODO

#if SIZEOF_WCHAR_T == 4
          XftDrawString32(xftgc,&color,font->xftfonts[pos],x,y,
                          (XftChar32*)text.data()+start,length);
#else
          XftDrawString16(xftgc,&color,font->xftfonts[pos],x,y,
                          (XftChar16*)text.data+start,length);
#endif

          if (Font::underlined & fontStack.top().style) {
            XGlyphInfo info;

#if SIZEOF_WCHAR_T == 4
            XftTextExtents32(dynamic_cast<Display*>(OS::display)->display,
                             font->xftfonts[pos],
                             (XftChar32*)text.data()+start,length,&info);
#else
            XftTextExtents16(dynamic_cast<Display*>(OS::display)->display,
                             font->xftfonts[pos],
                             (XftChar16*)text.data()+start,length),&info);
#endif
            DrawLine(x,y,x-info.x+info.xOff,y);
          }
          handled=true;
        }
#endif

        if (!handled) {
          char   *buffer;
          size_t bufferLength;

          buffer=Lum::Base::WStringToChar16BE(text.substr(start,length),bufferLength);

          if (buffer!=NULL) {
            XDrawString16(display,drawable,gc,x,y,(XChar2b*)buffer,bufferLength);

            if (Font::underlined & fontStack.top().style) {
              XCharStruct res;
              int         direction=0,ascent=0,descent=0;

              XTextExtents16(font->fonts[pos],(XChar2b*)buffer,bufferLength,&direction,&ascent,&descent,&res);
              DrawLine(x,y,x+res.width,y);
            }

            delete [] buffer;
          }
        }
      }

      void DrawInfo::DrawFillString(int x, int y,
                                    const std::wstring& text,
                                    Color background)
      {
        bool handled=false;

        assert(fontStack.size()>0);

#if defined(HAVE_LIB_XFT2)
        XftColor      color;
        XColor        xcolor;
        XGlyphInfo    info;
        unsigned long pos;
        Font          *font;

        font=dynamic_cast<Font*>(fontStack.top().font.Get());
        pos=font->GetFontPos(fontStack.top().style);

        if (font->xftfonts[pos]!=NULL) {
#if SIZEOF_WCHAR_T == 4
          XftTextExtents32(dynamic_cast<Display*>(OS::display)->display,
                           font->xftfonts[pos],
                           (XftChar32*)text.c_str(),text.length(),&info);
#else
          XftTextExtents16(dynamic_cast<Display*>(OS::display)->display,
                           font->xftfonts[pos],
                           (XftChar16*)text.c_str(),text.length(),&info);
#endif

          PushForeground(background);
          FillRectangle(x,y-font->ascent,-info.x+info.xOff,font->height);
          PopForeground();

          xcolor.pixel=fPenStack.top().color;
          XQueryColor(dynamic_cast<Display*>(OS::display)->display,
                      dynamic_cast<Display*>(OS::display)->colorMap,
                      &xcolor);

          color.pixel=xcolor.pixel;
          color.color.red=xcolor.red;
          color.color.blue=xcolor.blue;
          color.color.green=xcolor.green;
          color.color.alpha=(short unsigned int)-1; // TODO

#if SIZEOF_WCHAR_T == 4
          XftDrawString32(xftgc,&color,font->xftfonts[pos],x,y,
                          (XftChar32*)text.c_str(),text.length());
#else
          XftDrawString16(xftgc,&color,font->xftfonts[pos],x,y,
                          (XftChar16*)text.c_str(),text.length());
#endif

          handled=true;
        }
#endif

        if (!handled) {
          char   *buffer;
          size_t length;

          buffer=Lum::Base::WStringToChar16BE(text,length);

          if (buffer!=NULL) {
            XDrawImageString16(display,drawable,gc,x,y,(XChar2b*)buffer,length);
            delete [] buffer;
          }
        }
      }

      void DrawInfo::PushForeground(::Lum::OS::Color color)
      {
        if (fPenStack.size()>0 && fPenStack.top().color==color) {
          fPenStack.top().count++;
        }
        else {
          PenColor entry;

          entry.color=color;
          entry.count=1;

          fPenStack.push(entry);
          XSetForeground(display,gc,color);
        }
      }

      void DrawInfo::PopForeground()
      {
        if (fPenStack.top().count>1) {
          fPenStack.top().count--;
        }
        else {
          fPenStack.pop();
          if (fPenStack.size()>0) {
            XSetForeground(display,gc,fPenStack.top().color);
          }
        }
      }

      void DrawInfo::PushDrawMode(::Lum::OS::DrawInfo::DrawMode mode)
      {
        DrawMode entry;
        int      xMode=GXcopy;

        entry.mode=mode;
        modeStack.push(entry);

        switch (mode) {
        case drawModeCopy:
          xMode=GXcopy;
          break;
        case drawModeInvert:
          xMode=GXinvert;
          break;
        }

        XSetFunction(display,gc,xMode);
      }

      void DrawInfo::PopDrawMode()
      {
        int xMode=GXcopy;

        modeStack.pop();

        if (modeStack.size()>0) {
          switch (modeStack.top().mode) {
          case drawModeCopy:
            xMode=GXcopy;
            break;
          case drawModeInvert:
            xMode=GXinvert;
            break;
          }

          XSetFunction(display,gc,xMode);
        }
        else {
          XSetFunction(display,gc,GXcopy);
        }
      }

      void DrawInfo::PushPen(int size, Pen pen)
      {
        if (penStack.size()>0 && penStack.top().pen==pen && penStack.top().size==size) {
         penStack.top().count++;
        }
        else {
          PenStyle entry;

          entry.size=size;
          entry.pen=pen;
          entry.count=1;

          penStack.push(entry);

          /*
          if (d.dashStack!=NULL) {
            lMode=d.dashStack.mode;
          }
          else {
            lMode=X11.LineSolid;
          }*/

          switch (pen) {
          case penNormal:
            XSetLineAttributes(display,gc,size,LineSolid,CapButt,JoinBevel);
            break;
          case penRound:
            XSetLineAttributes(display,gc,size,LineSolid,CapRound,JoinRound);
            break;
          }
        }
      }

      void DrawInfo::PopPen()
      {
        if (penStack.top().count>1) {
          penStack.top().count--;
        }
        else {
          penStack.pop();

          /*
          if (d.dashStack!=NULL) {
            mode=d.dashStack.mode;
          }
          else {
            mode=X11.LineSolid;
          }*/

          if (penStack.size()>0) {
            switch (penStack.top().pen) {
            case penNormal:
              XSetLineAttributes(display,gc,penStack.top().size,LineSolid,CapButt,JoinBevel);
              break;
            case penRound:
              XSetLineAttributes(display,gc,penStack.top().size,LineSolid,CapRound,JoinRound);
              break;
            }
          }
          else {
            XSetLineAttributes(display,gc,0,LineSolid,CapButt,JoinBevel);
          }
        }
      }

      void DrawInfo::PushDash(const char* dashList, size_t len, DashMode mode)
      {
        DashEntry entry;
        int       size,cap=CapButt,join=JoinBevel,line=0;

        entry.dash=dashList;
        entry.len=len;
        entry.mode=mode;

        switch (mode) {
        case fDash:
          line=LineOnOffDash;
          break;
        case fbDash:
          line=LineDoubleDash;
          break;
        }

        if (penStack.size()>0) {
          size=penStack.top().size;

          switch (penStack.top().pen) {
          case penNormal:
            cap=CapButt;
            join=JoinBevel;
            break;
            case penRound:
            cap=CapRound;
            join=JoinRound;
            break;
          }
        }
        else {
          size=0;
          cap=CapButt;
          join=JoinBevel;
        }

        dashStack.push(entry);

        XSetLineAttributes(display,gc,size,line,cap,join);
        XSetDashes(display,gc,0,dashList,len);
      }

      void DrawInfo::PopDash()
      {
        int size,cap=CapButt,join=JoinBevel;

        dashStack.pop();

        if (penStack.size()>0) {
          size=penStack.top().size;

          switch (penStack.top().pen) {
          case penNormal:
            cap=CapButt;
            join=JoinBevel;
            break;
            case penRound:
            cap=CapRound;
            join=JoinRound;
            break;
          }
        }
        else {
          size=0;
          cap=CapButt;
          join=JoinBevel;
        }

        if (dashStack.size()>0) {
          int line=0;

          switch (dashStack.top().mode) {
          case fDash:
            line=LineOnOffDash;
            break;
          case fbDash:
            line=LineDoubleDash;
            break;
          }

          XSetLineAttributes(display,gc,size,line,cap,join);
          XSetDashes(display,gc,0,dashStack.top().dash,
                     dashStack.top().len);
        }
        else {
          XSetLineAttributes(display,gc,size,LineSolid,cap,join);
        }
      }

      void DrawInfo::PushPattern(const char */*pattern*/, int /*width*/, int /*height*/, int /*mode*/)
      {/*
        Pattern pat;


        pat=new Pattern;
        pat.pixMap=X11.XCreateBitmapFromData(D.display(Display).display,X11.XRootWindow(D.display(Display).display,D.display(Display).scrNum),pattern,width,height);
        if (pat.pixMap==0) {
          Err.String("Cannot create pimap!");
          Err.Ln;
        }
        pat.mode=mode;
        X11.XSetStipple(D.display(Display).display,d.gc,pat.pixMap);
        CASEmodeOFD.fgPattern:X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillStippled);|D.fbPattern:X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillOpaqueStippled);
      }
      else {
        Err.String("Unsuported patternMode!");
        Err.Ln;*/
      }

      void DrawInfo::PopPattern()
      {
       /*
        X11.XFreePixmap(D.display(Display).display,d.patternStack.pixMap);
        d.patternStack=d.patternStack.next;
        if (d.patternStack!=NULL) {
          CASEd.patternStack.modeOFD.fgPattern:X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillStippled);|D.fbPattern:X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillOpaqueStippled);
        }
        else {
          Err.String("Unsuported patternMode!");
          Err.Ln;
        }
        X11.XSetStipple(D.display(Display).display,d.gc,d.patternStack.pixMap);
      }
      else {
        X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillSolid);*/
      }

      void DrawInfo::PushBitmap(::Lum::OS::Bitmap* bitmap, BitmapMode mode)
      {
        PatternEntry pattern;

        pattern.pixmap=dynamic_cast<Bitmap*>(bitmap)->pixmap;
        pattern.mode=mode;

        XSetStipple(display,gc,pattern.pixmap);

        switch (mode) {
        case bitmapModeFg:
          XSetFillStyle(display,gc,FillStippled);
          break;
        case bitmapModeFgBg:
          XSetFillStyle(display,gc,FillOpaqueStippled);
          break;
        }

        patternStack.push(pattern);
      }

      void DrawInfo::PopBitmap()
      {
        patternStack.pop();

        if (patternStack.size()>0) {
          XSetStipple(display,gc,patternStack.top().pixmap);

          switch (patternStack.top().mode) {
          case bitmapModeFg:
            XSetFillStyle(display,gc,FillStippled);
            break;
          case bitmapModeFgBg:
            XSetFillStyle(display,gc,FillOpaqueStippled);
            break;
          }
        }
        else {
          XSetFillStyle(display,gc,FillSolid);
        }
      }

      void DrawInfo::DrawPoint(int x, int y)
      {
        XDrawPoint(display,drawable,gc,x,y);
      }

      void DrawInfo::DrawPointWithColor(int x, int y, ::Lum::OS::Color color)
      {
        XSetForeground(display,gc,color);
        XDrawPoint(display,drawable,gc,x,y);
        /*
        if (d.fPenPos>=0) {
          X11.XSetForeground(D.display(Display).display,d.gc,d.fPenStack[d.fPenPos].color);
        } POP */
      }

      void DrawInfo::DrawLine(int x1, int y1, int x2, int y2)
      {
        XDrawLine(display,drawable,gc,x1,y1,x2,y2);
      }

      void DrawInfo::DrawRectangle(int x, int y, int width, int height)
      {
        DrawLine(x,y+height-1,x,y+1);
        DrawLine(x,y,x+width-1,y);
        DrawLine(x+width-1,y+1,x+width-1,y+height-1);
        DrawLine(x+width-1,y+height-1,x,y+height-1);
      }

      void DrawInfo::FillRectangle(int x, int y, int width, int height)
      {
        XFillRectangle(display,drawable,gc,x,y,width,height);
      }

      void DrawInfo::DrawArc(int x, int y, int width, int height, int angle1, int angle2)
      {
        XDrawArc(display,drawable,gc,x,y,width-1,height-1,angle1,angle2);
      }

      void DrawInfo::FillArc(int x, int y, int width, int height, int angle1, int angle2)
      {
        XFillArc(display,drawable,gc,x,y,width,height,angle1,angle2);
      }

      void DrawInfo::FillPolygon(const Point points[], size_t count)
      {
        for (size_t x=0; x< count; x++) {
          pointArray[x].x=points[x].x;
          pointArray[x].y=points[x].y;
        }
        XFillPolygon(display,drawable,gc,pointArray,count,Complex,CoordModeOrigin);
      }

      void DrawInfo::CopyArea(int sX, int sY, int width, int height, int dX, int dY)
      {
        //XSetGraphicsExposures(display,gc,True);
        XCopyArea(display,drawable,drawable,gc,sX,sY,width,height,dX,dY);
        //XSetGraphicsExposures(display,gc,False);
      }

      /**
        Copy the rectangle sY,sY-width,height
        to the current DrawInfo starting at position dX,dY.
      */
      void DrawInfo::CopyFromBitmap(::Lum::OS::Bitmap* bitmap, int sX, int sY, int width, int height, int dX, int dY)
      {
        XCopyArea(display,dynamic_cast<Bitmap*>(bitmap)->pixmap,
                  drawable,gc,
                  sX,sY,width,height,
                  dX,dY);
      }

      void DrawInfo::CopyToBitmap(int sX, int sY, int width, int height, int dX, int dY, ::Lum::OS::Bitmap* bitmap)
      {
        //XSetGraphicsExposures(display,gc,True);
        XCopyArea(display,drawable,
                  dynamic_cast<Bitmap*>(bitmap)->pixmap,
                  dynamic_cast<DrawInfo*>(bitmap->GetDrawInfo())->gc,
                  sX,sY,width,height,
                  dX,dY);
        //XSetGraphicsExposures(display,gc,False);
      }
    }
  }
}
