#ifndef LUM_OS_DISPLAY_H
#define LUM_OS_DISPLAY_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <memory>
#include <string>

#include <Lum/Private/ImportExport.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/DragDrop.h>
#include <Lum/Base/Object.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Color.h>
#include <Lum/OS/Event.h>
#include <Lum/OS/Fill.h>
#include <Lum/OS/Font.h>
#include <Lum/OS/Frame.h>
#include <Lum/OS/Image.h>

#include <Lum/Prefs/Base.h>

namespace Lum {
  namespace OS {

    class Display;
    class KeyboardHandler;
    class Theme;
    class Window;

    /**
      Display represents the graphical display with all its global attributes.
    */
    class LUMAPI Display : public Lum::Base::MsgObject
    {

    public:
      static const size_t colorCount = 12; //! Number of predefined colors
      static const size_t fontCount  =  2; //! Number of predefined fonts
      static const size_t fillCount  = 27; //! Number of predefined fills
      static const size_t frameCount = 38; //! Number of predefined frames
      static const size_t imageCount = 30; //! Number of predefined images

    protected:
      class LUMAPI Timer
      {
      public:
        Lum::Base::SystemTime time;
        Model::ActionRef      action;

      public:
        bool operator<(const Timer& other) const;
      };

    public:
      enum ColorIndex {
        backgroundColor               =  0, // To be removed!
        tableTextColor                =  1, //! Color of text in tables and edit fields
        textColor                     =  2, //! Color of standard text (window, button...)
        textSelectColor               =  3, //! Color of standard text in pressed objects
        textDisabledColor             =  4, //! Color of disabled text
        fillColor                     =  5, // To be removed!
        fillTextColor                 =  6, //! Color of text in fills (tables, entries, progress...)
        graphScaleColor               =  7, //! Color of scaling marks in a graph
        blackColor                    =  8, // To be removed!
        whiteColor                    =  9, // To be removed!
        tabTextColor                  = 10, //! Font color for a textual tab rider
        tabTextSelectColor            = 11  //! Font color of a selected textual tab rider
      };

      enum FillIndex {
        backgroundFillIndex                =  0, //! Normal window background
        tableBackgroundFillIndex           =  1, //! Table background for even rows
        tableBackground2FillIndex          =  2, //! Table background for uneven rows
        buttonBackgroundFillIndex          =  3, //! Background of normal buttons
        positiveButtonBackgroundFillIndex  =  4,
        negativeButtonBackgroundFillIndex  =  5,
        defaultButtonBackgroundFillIndex   =  6,
        toolbarButtonBackgroundFillIndex   =  7, //! Background of toolbar button
        scrollButtonBackgroundFillIndex    =  8, //! Background of scroll button
        comboBackgroundFillIndex           =  9, //! Background of combobox
        entryBackgroundFillIndex           = 10, //! Background of entry fields
        helpBackgroundFillIndex            = 11, //! Background of tooltips
        hprogressBackgroundFillIndex       = 12, //! Background of progress bar
        hprogressFillIndex                 = 13, //! Filled part of progress bar
        vprogressBackgroundFillIndex       = 14, //! Background of progress bar
        vprogressFillIndex                 = 15, //! Filled part of progress bar
        hknobBoxFillIndex                  = 16, //! Background of horizontal scoll box
        vknobBoxFillIndex                  = 17, //! Background of vertical scoll box
        tabBackgroundFillIndex             = 18, //! Background of tab control
        tabRiderBackgroundFillIndex        = 19, //! Background of tab rider
        columnBackgroundFillIndex          = 20, //! Background of one table column header
        columnEndBackgroundFillIndex       = 21, //! Background of the area behind the last table column
        menuStripBackgroundFillIndex       = 22, //! Background of a menu strip
        menuPulldownBackgroundFillIndex    = 23, //! Background of a pulldown menu (entry in a menu strip)
        menuEntryBackgroundFillIndex       = 24, //! Background of a menu entry
        graphBackgroundFillIndex           = 25, //! Background of graph-like controls
        toolbarBackgroundFillIndex         = 26  //! Background of a toolbar
      };

      enum FrameIndex {
        buttonFrameIndex          =  0, //! frame for normal buttons
        positiveButtonFrameIndex  =  1,
        negativeButtonFrameIndex  =  2,
        defaultButtonFrameIndex   =  3,
        toolbarButtonFrameIndex   =  4,
        scrollButtonFrameIndex    =  5, //! frame for scroll buttons
        editFrameIndex            =  6, //! frame for single line text fields
        hprogressFrameIndex       =  7, //! frame of a progress bar
        vprogressFrameIndex       =  8, //! frame of a progress bar
        hknobBoxFrameIndex        =  9, //! frame around the knob area of a horizontal scroller element
        vknobBoxFrameIndex        = 10, //! frame around the knob area of a vertical scroller element
        hscrollFrameIndex         = 11, //! frame around a horizontal scrollbar
        vscrollFrameIndex         = 12, //! frame around a vertical scrollbar
        tabFrameIndex             = 13, //! tab frame
        tabRiderFrameIndex        = 14, //! frame around a single rider of a tab
        listboxFrameIndex         = 15, //! frame a round a list box and similar controls
        scrolledFrameIndex        = 16, //! Frame around scrolled areas
                                        //! (frame around the scrollable object and the corresponding scrollbars)
        columnFrameIndex          = 17, //! frame of a single table column header element
        columnLeftFrameIndex      = 18, //! frame of the first table column header element in a row
        columnMiddleFrameIndex    = 19, //! frame of not corner table column header element
        columnRightFrameIndex     = 20, //! frame of the last single table column header element in a row
        columnEndFrameIndex       = 21, //! frame for the area bhind the last object
        comboFrameIndex           = 22, //! Display-only combobox frame
        editComboFrameIndex       = 23, //! Editable combobox frame
        mainWindowFrameIndex      = 24, //! Frame around main windows
        menuWindowFrameIndex      = 25, //! Frame around menu windows
        popupWindowFrameIndex     = 26, //! Frame around popup windows
        tooltipWindowFrameIndex   = 27, //! Frame around tooltip windows
        hscaleFrameIndex          = 28, //! Frame around a horizontal slider
        vscaleFrameIndex          = 29, //! Frame around a vertical slider
        focusFrameIndex           = 30, //! Frame for showing focus
        statusbarFrameIndex       = 31, //! Frame around the whole status line
        statuscellFrameIndex      = 32, //! Frame around a single status cell
        plateFrameIndex           = 33, //! General frame for control presenting some simple content (color chooser)
        valuebarFrameIndex        = 34, //! Frame around value bars
        groupFrameIndex           = 35, //! Frame for grouping a number of elements
        menuStripFrameIndex       = 36, //! Frame around a menu strip
        menuPulldownFrameIndex    = 37  //! Frame around a pull down menu (entry ina menu strip)
      };

      enum ImageIndex {
        checkImageIndex            =  0, //! Image of a check box
        radioImageIndex            =  1, //! Image of a single radio button
        hknobImageIndex            =  2, //! Image of a horizontal scrolling knob
        vknobImageIndex            =  3, //! Image of a vertical scrolling knob
        arrowLeftImageIndex        =  4, //! Image of a arrow pointing to the left side
        arrowRightImageIndex       =  5, //! Image of a arrow pointing to the right side
        arrowUpImageIndex          =  6, //! Image of a arrow pointing up
        arrowDownImageIndex        =  7, //! Image of a arrow pointing down
        arrowScrollLeftImageIndex  =  8, //! Image of a arrow for scroll bars pointing to the left side
        arrowScrollRightImageIndex =  9, //! Image of a arrow for scroll bars pointing to the right side
        arrowScrollUpImageIndex    = 10, //! Image of a arrow for scroll bars pointing up
        arrowScrollDownImageIndex  = 11, //! Image of a arrow for scroll bars pointing down
        comboImageIndex            = 12, //! Image for combo box popup (normally just an arrow)
        comboEditButtonImageIndex  = 13, //! Image for combo box popup (normally just an arrow)
        comboDividerImageIndex     = 14, //! Divider between content and image
        treeExpanderImageIndex     = 15, //! Image for tree node (de-)expansion
        hscaleKnobImageIndex       = 16, //! Horizontal slider knob
        vscaleKnobImageIndex       = 17, //! Vertical slider knob
        leftSliderImageIndex       = 18, //! Left (filled) part of the slider control
        rightSliderImageIndex      = 19, //! Right (unfilled) part of the slider control
        topSliderImageIndex        = 20, //! Top (unfilled) part of the slider control
        bottomSliderImageIndex     = 21, //! bottom (filled) part of the slider control
        menuDividerImageIndex      = 22, //! Divider between menu entry groups
        menuSubImageIndex          = 23, //! Image for signaling a sub menu
        positiveImageIndex         = 24, //! Image for marking a positive action (e.g. for a button)
        negativeImageIndex         = 25, //! Image for marking a negative action (e.g. for a button)
        defaultImageIndex          = 26, //! Image for marking a default action (e.g. for a button)
        helpImageIndex             = 27, //! Image for marking a help action (e.g. for a button)
        closeImageIndex            = 28, //! Image for closing widget
        ledImageIndex              = 29  //! An image for a simple on/off LED
      };

      enum FontScale {
        fontScaleFootnote = 85,
        fontScaleNormal   = 100,
        fontScaleCaption2 = 115,
        fontScaleCaption1 = 135,
        fontScaleLogo     = 200
      };

      enum FontType {
        fontTypeProportional,
        fontTypeFixed
      };

      enum TextDirection {
        textDirectionLeftToRight,
        textDirectionRightToLeft
      };

      enum Type {
        typeGraphical = 0,
        typeTextual   = 1
      };

      enum ColorMode {
        colorModeMonochrome = 0,
        colorModeGreyScale  = 1,
        colorModeColor      = 2
      };

      enum Size {
        sizeTiny   = 0,
        sizeSmall  = 1,
        sizeNormal = 2,
        sizeLarge  = 3,
        sizeHuge   = 4
      };

      enum Space {
        spaceWindowBorder      =  0, //! Space between window border and its content
        spaceInterGroup        =  1, //! Space between groups of elements
        spaceGroupIndent       =  2, //! Indention between groups
        spaceInterObject       =  3, //! Space between individual objects
        spaceObjectBorder      =  4, //! Space between object border and its elements
        spaceIntraObject       =  5, //! Space between elements within an object
        spaceLabelObject       =  6, //! Space between an object and its label
        spaceObjectSensibility =  7, //! Size of sensitive area around an object
        spaceObjectMouseMove   =  8, //! Amount of space the mouse has to move before
                                     //! a movement is detected (=> drag & drop).
        spaceObjectDetail      =  9  //! Space for object visualizing details
      };

      enum ByteOrder {
        littleEndian,
        bigEndian
      };

      /**
        Display-specific preferences
      */
      class Prefs : public ::Lum::Prefs::Prefs
      {
      public:
        std::string  propFont;
        std::string  fixedFont;
        size_t       propFontSize;
        size_t       fixedFontSize;
        std::wstring theme;

      public:
        Prefs();
      };

    public:
      /**
        @name (De)initialisation
        Initialisation and deinitialisation of Display instances.
      */
      //@{
      /**
        Open the display. Opening a display may fail.
      */
      virtual bool Open() = 0;

      /**
        Close the display.
      */
      virtual void Close() = 0;
      //@}

      /**
        @name Screen and workarea dimensions and resolution
        Returns the dimensions of the screen and the workarea, which is the useable part
        of the screen not covered by task bars and similar. LAlso returns the resolution of the
        screen and DPI.
      */
      //@{
      virtual double GetDPI() const = 0;

      /**
        Return the screen width in pixel.
      */
      virtual size_t GetScreenWidth() const = 0;

      /**
        Return the screen height in pixel.
      */
      virtual size_t GetScreenHeight() const = 0;

      /**
        Return the work area width in pixel. Work area is the part
        of the desktop where you can place application window on.
      */
      virtual size_t GetWorkAreaWidth() const = 0;

      /**
        Return the work area height in pixel. Work area is the part
        of the desktop where you can place application window on.
      */
      virtual size_t GetWorkAreaHeight() const = 0;
      virtual Size GetSize() const = 0;
      //@}

      /**
        @name Fonts
        Method for font management.
      */
      //@{
      /**
        Get a font of the given type and the given size, where
        the size is handed to the function as percent points in relation
        to the default font size of the given font type.
      */
      virtual Font* GetFont(FontType type=fontTypeProportional,
                            size_t scale=fontScaleNormal) const = 0;

      /**
        Get a font of the given type and the given size, where
        the size is given in pixel. If no font with the exact pixel
        size can be found, NULL will be returned.
      */
      virtual Font* GetFontByPixel(FontType type, size_t size) const = 0;
      /**
        Get a font of the given type and the given maximum size, where
        the maxium size is given in pixel. Note that the gven pixel
        size must be at least the height in pixel of the given default
        font size for the given font type, else it cannot be assured
        that a font can be found!
      */
      virtual Font* GetFontByMaxPixel(FontType type, size_t size) const = 0;
      //FontList GetFontList() = 0;
      //@}

      /**
        @name Colors
        Method for handling of colors and fills.
      */
      //@{
      virtual ColorMode GetColorMode() const = 0;

      /**
        Return the number of bits available for defining a color.
      */
      virtual size_t GetColorDepth() const = 0;

      /**
        Return the given predefined color.
      */
      virtual Color GetColor(ColorIndex color) const = 0;
      virtual Color GetColorByName(const std::string& name) = 0;
      virtual bool AllocateColor(double red, double green, double blue,
                                 Color& color) = 0;
      virtual bool AllocateNamedColor(const std::string& name,
                                      Color& color) = 0;
      virtual void FreeColor(Color color) = 0;

      /**
        @name Objects
        Method for handling of various highlevel objects like fills and frames.
      */
      //@{
      virtual Fill* GetFill(FillIndex fill) const = 0;
      virtual Frame* GetFrame(FrameIndex frame) const = 0;
      virtual Image* GetImage(ImageIndex image) const = 0;
      //@}

      /**
        @name Event loop
        Methods for dealing with the global event loop.
      */
      //@{
      virtual void PutBackEvent(Event* event, Window* destWin) = 0;
      virtual void EventLoop() = 0;
      virtual void SignalEventLoop() = 0;
      virtual void Exit() = 0;
      virtual void QueueActionForAsyncNotification(Model::Action *action) = 0;
      virtual void QueueActionForEventLoop(Model::Action *action) = 0;
      //@}

      // Mouse
      virtual bool GetMousePos(int& x, int& y) const = 0;

      /**
        @name Selection
        Methods for dealing with OS specific global selection support. Whil clipboard must
        exlipcitely be copied a selection is imediately active.
      */
      //@{
      virtual bool RegisterSelection(Lum::Base::DnDObject* object, Window* window) = 0;
      virtual void CancelSelection() = 0;
      virtual ::Lum::Base::DnDObject* GetSelectionOwner() const = 0;
      virtual bool QuerySelection(Window* window, Lum::Base::DnDObject* object) = 0;
      //@}

      /**
        @name Clipboard
        Methods dealing with the OS clipboard.
      */
      //@{
      /**
        Set the operating system clipboard to the given string value.

        Note, that the interface of this method will change to support more
        datatypes in the future.
      */
      virtual bool SetClipboard(const std::wstring& content) = 0;

      /**
        Request the current value of the operating system clipboard.

        If Illumination can get the clipboard and the clipboard does have
        a texttual value, it is returned. Otherwise an empty string is returned.
      */

      virtual std::wstring GetClipboard() const = 0;

      /**
        Clear the operating system global clipboard.
      */
      virtual void ClearClipboard() = 0;
      //@}

      /**
        @name Timer
        Methods for dealing with timer.
      */
      //@{
      virtual void AddTimer(long seconds, long microseconds, Model::Action* action) = 0;
      virtual void RemoveTimer(Model::Action* action) = 0;

      /**
        Check the list of registered timeouts and send a message for every elapsed
        timeout.
       */
      virtual void CheckTimers() = 0;
      //@}

      /*
      Sleep AddSleep(::VO::Base::Object::MsgObject object) = 0;
      void RemoveSleep(Sleep sleep) = 0;

      Channel AddChannel(Channel channel, unsigned long ops, ::VO::Base::Object::MsgObject object) = 0;
      void RemoveChannel(Channel channel) = 0;*/

      /**
        @name Textual representation of keyboard shortcuts.
        Methods for converting textual representations of keyboard shortcuts to strings
        and vice versa.
      */
      //@{
      /**
        Converts a given key description including qualifier and
        key into a textual description. This routine does
        not differentiate between logical identical qualifiers, that apear
        mutliple time, like Lum::OS::qualifierShiftLeft and Lum::OS::qualifierShiftRight.
        The will be maped to the same string.

        The current implementation does only handle printable keys
        correctly.
      */
      virtual void KeyToKeyDescription(unsigned long qualifier,
                                       const std::wstring& key,
                                       std::wstring& description) = 0;

      /**
        Converts a given key description including qualifier and
        key into a textual description for displaying them.
        The resulting string tries to mimic the syntax of the underlying driver
        instead of Display::KeyToKeyDescription which uses the internal
        format and thus is reversible. The result of this method cannot be
        reversed by Display::KeyDescriptionToKey.

        This method can be overloaded by display drivers.

        This routine does not differentiate between logical identical qualifiers,
        that apear mutliple time, like Lum::OS::qualifierShiftLeft and
        Lum::OS::qualifierShiftRight. The will be maped to the same string.

        The current implementation does only handle printable keys
        correctly.
       */
      virtual void KeyToDisplayKeyDescription(unsigned long qualifier,
                                              const std::wstring& key,
                                              std::wstring& description) = 0;

      /**
        Converts a given textual key \p description like it is returned
        by Display::KeyToKeyDescription into a keydescription containing of
        \p qualifier and \p key.

        The method returns 'true' if the conversion succeeds, 'false'
        if the conversion was not possible because of syntactical error in the
        description format.
      */
      virtual bool KeyDescriptionToKey(const std::wstring& description,
                                       unsigned long& qualifier,
                                       std::wstring& key) = 0;

      //@}


      virtual Type GetType() const = 0;

      virtual void SetProgramName(const std::wstring& name) = 0;
      virtual std::wstring GetProgramName() const = 0;

      virtual void SetAppName(const std::wstring& name) = 0;
      virtual std::wstring GetAppName() const = 0;

      virtual void SetArgs(const std::vector<std::wstring>& args) = 0;
      virtual size_t GetArgCount() const = 0;
      virtual std::wstring GetArg(size_t pos) const = 0;

      /**
        @name references
        Methods for dealing with preferences.
      */
      //@{
      virtual std::wstring GetPrefsPath() const = 0;
      virtual std::wstring GetPrefsName() const = 0;
      //@}

      /**
        @name Theming
        Methods for dealing with themes.
      */
      //@{
      virtual std::wstring GetThemePath() const = 0;
      virtual std::wstring GetThemeName() const = 0;
      virtual Theme* GetTheme() const = 0;
      //@}

      virtual size_t GetSpaceHorizontal(Space space) const = 0;
      virtual size_t GetSpaceVertical(Space space) const = 0;

      virtual TextDirection GetDefaultTextDirection() const = 0;

      /**
        Set the maximum time between mouse clicks that are detected as double click
        (or tripple click). Time is measured in milliseconds.
      */
      virtual void SetMultiClickTime(unsigned long time) = 0;
      virtual unsigned long GetMultiClickTime() const = 0;

      /**
        @name Virtual keyboards
        Mehods for dealing with support for virtual keyboards and other advanced
        input methods.
      */
      //@{
      virtual KeyboardHandler* GetKeyboardHandler() const = 0;
      //@}


      virtual void Beep() = 0 ;

      virtual void ReinitWindows() = 0;

      virtual ByteOrder GetSystemByteOrder() const = 0;

    };

    extern "C" {
      extern Display::ColorIndex GetColorIndexByName(const std::string& name);
    }

    extern LUMAPI const char* colorNames[Display::colorCount];
    extern LUMAPI const char* fillNames[Display::fillCount];
    extern LUMAPI const char* frameNames[Display::frameCount];
/*
      bool GetRGB32ByColorName(char name[], int r, int g, int b);
*/
    extern LUMAPI Display* display;
  }
}

#endif
