/*
  This source is part of the Illumination library
  Copyright (C) 2006  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/PopupControl.h>

namespace Lum {

  PopupControl::Prefs::Prefs(const std::wstring& name)
  : Control::Prefs(name)
  {
    // no code
  }

  PopupControl::PopupControl(Control* control)
  : control(control),button(new Button()),popupAction(new Model::Action())
  {
    button->RequestFocus();
    button->SetModel(popupAction);
    //button->SetType(Button::typeSmall);
    button->SetText(L"...");

    popupAction->Disable();

    AttachModel(popupAction);
  }

  PopupControl::~PopupControl()
  {
    UnattachModel(popupAction);

    if (control->GetModel()) {
      UnattachModel(control->GetModel());
    }

    delete button;
    delete control;
  }

  bool PopupControl::HasBaseline() const
  {
    return control->HasBaseline();
  }

  size_t PopupControl::GetBaseline() const
  {
    if (control->HasBaseline()) {
      return frame->topBorder+(height-control->GetOHeight())/2+control->GetBaseline();
    }
    else {
      return 0;
    }
  }

  bool PopupControl::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    if (control!=NULL && !visitor.Visit(control)) {
      return false;
    }

    if (button!=NULL && !visitor.Visit(button)) {
      return false;
    }

    return true;
  }

  bool PopupControl::SetModel(Base::Model* model)
  {
    bool res;

    if (control->GetModel()!=NULL) {
      UnattachModel(control->GetModel());
    }

    res=control->SetModel(model);

    if (control->GetModel()) {
      AttachModel(control->GetModel());
    }

    return res;
  }

  Base::Model* PopupControl::GetModel() const
  {
    return control->GetModel();
  }

  void PopupControl::CalcSize()
  {
    control->SetParent(this);
    button->SetParent(this);

    control->SetFlex(true,false);

    control->CalcSize();
    button->CalcSize();

    minWidth=control->GetOMinWidth()+button->GetOMinWidth();
    width=control->GetOWidth()+button->GetOWidth();

    minHeight=std::max(control->GetOMinHeight(),button->GetOMinHeight());
    height=std::max(control->GetOHeight(),button->GetOHeight());

    Control::CalcSize();
  }

  void PopupControl::Layout()
  {
    control->ResizeWidth(width-button->GetOWidth());

    control->Move(x,y+(height-control->GetOHeight())/2);
    button->Move(x+control->GetOWidth(),y+(height-button->GetOHeight())/2);

    Control::Layout();
  }

  void PopupControl::Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!Intersect(x,y,w,h)) {
      return;
    }

    /* --- */

    DrawBackground(x,y,w,h);

    control->Draw(x,y,w,h);
    button->Draw(x,y,w,h);
  }

  void PopupControl::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (control!=NULL && popupAction.Valid() && control->GetModel()==model) {
      if (model->IsEnabled()) {
        popupAction->Enable();
      }
      else {
        popupAction->Disable();
      }
    }

    Control::Resync(model,msg);
  }
}
