/*
  This source is part of the Illumination library
  Copyright (C) 2006  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/X11/GtkTheme.h>

#include <Lum/Private/Config.h>

#include <gdk/gdkx.h>
#include <gdk-pixbuf-xlib/gdk-pixbuf-xlib.h>

#if defined(HAVE_LIB_HILDON)
  #if HILDON_VERSION == 1
    #include <hildon/hildon-window.h>
  #else
    #include <hildon-widgets/hildon-window.h>
  #endif
#endif

#include <Lum/Base/Size.h>

#if defined(HAVE_LIB_CAIRO)
  #include <Lum/OS/Cairo/DrawInfo.h>
#endif

#include <Lum/OS/X11/Display.h>
#include <Lum/OS/X11/DrawInfo.h>

#if defined(HAVE_LIB_GCONF)
  #include <gconf/gconf.h>
  #include <gconf/gconf-client.h>
#endif

#include <cmath>

#include <Lum/Base/String.h>

#include <iostream>
namespace Lum {
  namespace OS {
    namespace X11 {

      GtkWidgets *gtkWidgets=NULL;

#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
      static const char* buttonDetail="buttondefault";
#else
      static const char* buttonDetail="button";
#endif

      GtkStateType GetGtkStateType(OS::DrawInfo *draw)
      {
        if (draw->disabled) {
          return GTK_STATE_INSENSITIVE;
        }
        else if (draw->selected) {
          return GTK_STATE_ACTIVE;
        }
        else if (draw->activated) {
          return GTK_STATE_PRELIGHT;
        }

        return GTK_STATE_NORMAL;
      }

      GtkStateType GetGtkRangeStateType(OS::DrawInfo *draw, GtkWidget* widget)
      {
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
	if (true) {
#else	
        if (GetWidgetStyleBoolValue(widget,"activate-slider")) {
#endif		
          if (draw->disabled) {
            return GTK_STATE_INSENSITIVE;
          }
          else if (draw->selected) {
            return GTK_STATE_ACTIVE;
          }
          else if (draw->activated) {
            return GTK_STATE_PRELIGHT;
          }

          return GTK_STATE_NORMAL;
        }
        else {
          if (draw->disabled) {
            return GTK_STATE_INSENSITIVE;
          }
          else if (draw->selected || draw->activated) {
            return GTK_STATE_PRELIGHT;
          }

          return GTK_STATE_NORMAL;
        }
      }

      void SetFocusIfRequested(OS::DrawInfo *draw, GtkWidgets::GtkStyles style)
      {
        if (draw->focused) {
          GTK_WIDGET_SET_FLAGS(gtkWidgets->widget[style],GTK_HAS_FOCUS);
        }
        else {
          GTK_WIDGET_UNSET_FLAGS(gtkWidgets->widget[style],GTK_HAS_FOCUS);
        }
      }

      GtkWidgets::GtkWidgets()
      {
        GtkListStore      *listStore;
        GtkCellRenderer   *cellRenderer;
        GtkTreeIter       treeIter;
        GtkTreeViewColumn *column;
        GtkTreeView       *list2;

#ifdef HAVE_LIB_HILDON
        widget[gtkStyleWindow]=hildon_window_new();
#else
        widget[gtkStyleWindow]=gtk_window_new(GTK_WINDOW_POPUP);
#endif
        gtk_widget_realize(widget[gtkStyleWindow]);

        layout=gtk_fixed_new();
        gtk_container_add(GTK_CONTAINER(widget[gtkStyleWindow]),layout);

        adjustment=GTK_ADJUSTMENT(gtk_adjustment_new(50,1,100,1,1,20));

        widget[gtkStyleButton]=gtk_button_new();
        widget[gtkStyleToolButton]=gtk_button_new();
        gtk_button_set_focus_on_click(GTK_BUTTON(widget[gtkStyleToolButton]),FALSE);
        widget[gtkStyleToolbar]=gtk_toolbar_new();
        gtk_button_set_relief(GTK_BUTTON(widget[gtkStyleToolButton]),
                              gtk_toolbar_get_relief_style(GTK_TOOLBAR(widget[gtkStyleToolbar])));
        gtk_toolbar_append_widget(GTK_TOOLBAR(widget[gtkStyleToolbar]),widget[gtkStyleToolButton],NULL,NULL);
        widget[gtkStyleEdit]=gtk_entry_new();
        widget[gtkStyleProgress]=gtk_progress_bar_new();
        widget[gtkStyleHKnob]=gtk_hscrollbar_new(adjustment);
        widget[gtkStyleVKnob]=gtk_vscrollbar_new(adjustment);
        widget[gtkStyleCheck]=gtk_check_button_new();
        widget[gtkStyleRadio]=gtk_radio_button_new(NULL);
        widget[gtkStyleRadio2]=gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(widget[gtkStyleRadio]));
        widget[gtkStyleTab]=gtk_notebook_new();

        widget[gtkStyleList]=gtk_tree_view_new();
        // Add column to table and pack cell renderer into tree view column
        column=gtk_tree_view_column_new();
        cellRenderer=gtk_cell_renderer_text_new();
        gtk_tree_view_column_pack_start(column,cellRenderer,TRUE);
        gtk_tree_view_append_column(GTK_TREE_VIEW(widget[gtkStyleList]),column);

        // Create list model with one column and one row
        listStore=gtk_list_store_new(1,G_TYPE_STRING);
        gtk_list_store_append(listStore,&treeIter);
        gtk_list_store_set(listStore,&treeIter,0,"Blub",-1);

        // Assign listmodel to table
        gtk_tree_view_set_model(GTK_TREE_VIEW(widget[gtkStyleList]),GTK_TREE_MODEL(listStore));
        g_object_unref(listStore);
        gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(widget[gtkStyleList])),
                                    GTK_SELECTION_SINGLE);
        gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(widget[gtkStyleList])),&treeIter);
        widget[gtkStyleColumn]=column->button;

        list2=GTK_TREE_VIEW(gtk_tree_view_new());
        column=gtk_tree_view_column_new();
        gtk_tree_view_append_column(list2,column);
        widget[gtkStyleColumnLeft]=column->button;
        column=gtk_tree_view_column_new();
        gtk_tree_view_append_column(list2,column);
        widget[gtkStyleColumnMiddle]=column->button;
        column=gtk_tree_view_column_new();
        gtk_tree_view_append_column(list2,column);
        widget[gtkStyleColumnRight]=column->button;

        widget[gtkStyleComboBox]=gtk_combo_box_new();
        widget[gtkStyleEditComboBox]=gtk_combo_box_entry_new();
        widget[gtkStyleHLine]=gtk_hseparator_new();
        widget[gtkStyleVLine]=gtk_vseparator_new();

        widget[gtkStyleHScale]=gtk_hscale_new(adjustment);
        widget[gtkStyleVScale]=gtk_vscale_new(adjustment);
        gtk_range_set_inverted(GTK_RANGE(widget[gtkStyleVScale]),true);

        widget[gtkStyleStatus]=gtk_statusbar_new();
        widget[gtkStyleScrolled]=gtk_scrolled_window_new(adjustment,adjustment);

        widget[gtkStyleDummy]=gtk_button_new();

        widget[gtkStyleMenuBar]=gtk_menu_bar_new();
        widget[gtkStyleMenuPullDown]=gtk_menu_item_new();
        widget[gtkStyleMenu]=gtk_menu_new();
        widget[gtkStyleMenuItem]=gtk_menu_item_new();

        for (size_t i=gtkStyleButton; i<gtkStyleMenuPullDown; i++) {
          if (i!=gtkStyleToolButton) {
            gtk_container_add(GTK_CONTAINER(layout),widget[i]);
          }
          gtk_widget_realize(widget[i]);
        }

        gtk_menu_shell_append(GTK_MENU_SHELL(widget[gtkStyleMenuBar]),widget[gtkStyleMenuPullDown]);
        gtk_widget_realize(widget[gtkStyleMenu]);
        gtk_menu_shell_append(GTK_MENU_SHELL(widget[gtkStyleMenu]),widget[gtkStyleMenuItem]);

        gtk_widget_grab_focus(widget[gtkStyleDummy]);

        positiveIcon=gtk_widget_render_icon(GetWidget(gtkStyleWindow),GTK_STOCK_APPLY,GTK_ICON_SIZE_BUTTON,NULL);
        negativeIcon=gtk_widget_render_icon(GetWidget(gtkStyleWindow),GTK_STOCK_CANCEL,GTK_ICON_SIZE_BUTTON,NULL);
        defaultIcon=gtk_widget_render_icon(GetWidget(gtkStyleWindow),GTK_STOCK_CLOSE,GTK_ICON_SIZE_BUTTON,NULL);
        helpIcon=gtk_widget_render_icon(GetWidget(gtkStyleWindow),GTK_STOCK_HELP,GTK_ICON_SIZE_BUTTON,NULL);
        closeIcon=gtk_widget_render_icon(GetWidget(gtkStyleWindow),GTK_STOCK_CLOSE,GTK_ICON_SIZE_MENU,NULL);
      }

      GtkWidgets::~GtkWidgets()
      {
        if (positiveIcon!=NULL) {
          g_object_unref(positiveIcon);
        }
        if (negativeIcon!=NULL) {
          g_object_unref(negativeIcon);
        }
        if (defaultIcon!=NULL) {
          g_object_unref(defaultIcon);
        }
        if (helpIcon!=NULL) {
          g_object_unref(helpIcon);
        }
        if (closeIcon!=NULL) {
          g_object_unref(closeIcon);
        }

        gtk_object_destroy(GTK_OBJECT(widget[gtkStyleWindow]));
        gtk_object_destroy(GTK_OBJECT(widget[gtkStyleMenu]));
      }

      GtkWidget* GtkWidgets::GetWidget(GtkStyles style) const
      {
        return widget[style];
      }

      GtkStyle* GtkWidgets::GetStyle(GtkStyles style) const
      {
        return gtk_widget_get_style(widget[style]);
      }

      GtkImage::GtkImage(Display::ImageIndex image)
        : image(image)
      {
        switch (image) {
        case Display::checkImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleCheck],
                                               "indicator-size");
          height=width;
          break;
        case Display::radioImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleRadio],
                                               "indicator-size");
          height=width;
          break;
        case Display::hknobImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                               "slider-width");
          height=width;
          break;
        case Display::vknobImageIndex:
          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                                "slider-width");
          width=height;
          break;
        case Display::arrowLeftImageIndex:
        case Display::arrowRightImageIndex:
        case Display::arrowUpImageIndex:
        case Display::arrowDownImageIndex:
          width=8;
          height=8;
          break;
        case Display::menuSubImageIndex:
          width=6;
          height=7;
          break;
        case Display::comboImageIndex:
          if (gtk_major_version>=2 && gtk_minor_version>=12) {
            width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleComboBox],
                                                 "arrow-size");
          }
          else {
            width=15;
          }
          height=width;
          break;
        case Display::comboEditButtonImageIndex:
          /* Trick */
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleCheck],
                                               "indicator-size");
          height=width;
          break;
        case Display::arrowScrollLeftImageIndex:
        case Display::arrowScrollRightImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                                "stepper-size");

          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                               "slider-width");
          break;
        case Display::arrowScrollUpImageIndex:
        case Display::arrowScrollDownImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                                "slider-width");
          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                                "stepper-size");;
          break;
        case Display::comboDividerImageIndex:
          width=2;
          height=2;
          break;
        case Display::treeExpanderImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleList],
                                               "expander-size");
          height=width;
          break;
        case Display::hscaleKnobImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                                               "slider-length");
          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                                               "slider-width");
          break;
        case Display::vscaleKnobImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                                               "slider-width");
          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                                                "slider-length");
          break;
        case Display::leftSliderImageIndex:
        case Display::rightSliderImageIndex:
          width=0;
          height=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                                               "slider-width");
          break;
        case Display::topSliderImageIndex:
        case Display::bottomSliderImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                                               "slider-width");
          height=0;
          break;
        case Display::menuDividerImageIndex:
          width=4;
          height=2; // TODO: Find out correct value
          break;
        case Display::positiveImageIndex:
          width=gdk_pixbuf_get_width(gtkWidgets->positiveIcon);
          height=gdk_pixbuf_get_height(gtkWidgets->positiveIcon);
          break;
        case Display::negativeImageIndex:
          width=gdk_pixbuf_get_width(gtkWidgets->negativeIcon);
          height=gdk_pixbuf_get_height(gtkWidgets->negativeIcon);
          break;
        case Display::defaultImageIndex:
          width=gdk_pixbuf_get_width(gtkWidgets->defaultIcon);
          height=gdk_pixbuf_get_height(gtkWidgets->defaultIcon);
          break;
        case Display::helpImageIndex:
          width=gdk_pixbuf_get_width(gtkWidgets->helpIcon);
          height=gdk_pixbuf_get_height(gtkWidgets->helpIcon);
          break;
        case Display::closeImageIndex:
          width=gdk_pixbuf_get_width(gtkWidgets->closeIcon);
          height=gdk_pixbuf_get_height(gtkWidgets->closeIcon);
          break;
        case Display::ledImageIndex:
          width=(size_t)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleRadio],
                                               "indicator-size");
          height=width;
          break;

        }
      }

      size_t GtkImage::GetWidth() const
      {
        return width;
      }

      size_t GtkImage::GetHeight() const
      {
        return height;
      }

      bool GtkImage::GetAlpha() const
      {
        return true;
      }

      unsigned long GtkImage::GetDrawCap() const
      {
        return drawsDisabled|drawsFocused|drawsActivated;
      }

      void GtkImage::Draw(OS::DrawInfo *draw, int x, int y, size_t w, size_t h)
      {
        ::Display     *display=dynamic_cast<Display*>(OS::display)->display;
        ::Pixmap      pixmap;
        GdkDrawable   *gtkDraw;
        GdkRectangle  rectangle;
        unsigned long mask;
        XGCValues     values;
        ::GC          gc;

        assert(w!=0 && h!=0);

        switch (image) {
        case Display::ledImageIndex:
          if (draw->selected) {
            draw->PushForeground(OS::Display::fillColor);
            draw->FillArc(x,y,width-1,height-1,0*64,360*64);
            draw->PopForeground();
          }
          else {
            draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shadowColor);
            draw->FillArc(x,y,width-1,height-1,0*64,360*64);
            draw->PopForeground();
          }

          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shineColor);
          draw->DrawArc(x+width / 4,y+height / 4,
                       width / 2,height / 2,60*64,90*64);
          draw->PopForeground();

          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shadowColor);
          draw->DrawArc(x,y,width-1,height-1,0*64,360*64);
          draw->PopForeground();
          return;
        default:
          break;
        }

        rectangle.x=0;
        rectangle.y=0;
        rectangle.width=w;
        rectangle.height=h;

        pixmap=::XCreatePixmap(display,
                               ::XDefaultRootWindow(display),
                               w,h,
                               OS::display->GetColorDepth());

        if (pixmap==0) {
          std::cerr << "Cannot create Pixmap" << std::endl;
          return;
        }

        mask=0;
        gc=::XCreateGC(display,pixmap,mask,&values);
        if (gc==0) {
          std::cerr << "Cannot create GC" << std::endl;
          XFreePixmap(display,pixmap);
          return;
        }

        gtkDraw=gdk_pixmap_foreign_new(pixmap);
        if (gtkDraw==NULL) {
          std::cerr << "Cannot create GdkDrawable" << std::endl;
          XFreeGC(display,gc);
          XFreePixmap(display,pixmap);
          return;
        }

        gdk_drawable_set_colormap(gtkDraw,gdk_colormap_get_system());

        ::Drawable drawable=0;
        ::GC       targetGC=0;

        if (dynamic_cast<DrawInfo*>(draw)!=NULL) {
          drawable=dynamic_cast<DrawInfo*>(draw)->drawable;
          targetGC=dynamic_cast<DrawInfo*>(draw)->gc;
        }
#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        else if (dynamic_cast<Cairo::X11DrawInfo*>(draw)!=NULL) {
          drawable=dynamic_cast<Cairo::X11DrawInfo*>(draw)->drawable;
        }
#endif
        else {
          assert(true);
        }

        ::XCopyArea(display,
                    drawable,
                    pixmap,
                    gc,
                    x,y,w,h,
                    0,0);

        switch (image) {
        case Display::checkImageIndex:
          gtk_paint_check(gtkWidgets->GetStyle(GtkWidgets::gtkStyleCheck),
                          gtkDraw,
                          draw->activated ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL,
                          draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleCheck],
                          "checkbutton",
                          0,0,w,h);
          break;
        case Display::radioImageIndex:
          gtk_paint_option(gtkWidgets->GetStyle(GtkWidgets::gtkStyleRadio),
                           gtkDraw,
                           draw->activated ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL,
                           draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                           &rectangle,
                           gtkWidgets->widget[draw->selected ? GtkWidgets::gtkStyleRadio : GtkWidgets::gtkStyleRadio2],
                           "radiobutton",
                           0,0,w,h);
          break;
        case Display::hknobImageIndex:
          gtk_paint_slider(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                           gtkDraw,
                           GetGtkRangeStateType(draw,gtkWidgets->widget[GtkWidgets::gtkStyleHKnob]),
                           GTK_SHADOW_OUT,
                           &rectangle,
                           gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                           "slider",
                           0,0,w,h,
                           GTK_ORIENTATION_HORIZONTAL);
          break;
        case Display::vknobImageIndex:
          gtk_paint_slider(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                           gtkDraw,
                           GetGtkRangeStateType(draw,gtkWidgets->widget[GtkWidgets::gtkStyleVKnob]),
                           GTK_SHADOW_OUT,
                           &rectangle,
                           gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                           "slider",
                           0,0,w,h,
                           GTK_ORIENTATION_VERTICAL);
          break;
        case Display::arrowLeftImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "arrow",
                          GTK_ARROW_LEFT,
                          false,
                          0,0,w,h);
          break;
        case Display::arrowRightImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "arrow",
                          GTK_ARROW_RIGHT,
                          true,
                          0,0,w,h);
          break;
        case Display::arrowUpImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "arrow",
                          GTK_ARROW_UP,
                          true,
                          0,0,w,h);
          break;
        case Display::arrowDownImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "arrow",
                          GTK_ARROW_DOWN,
                          true,
                          0,0,w,h);
          break;
        case Display::arrowScrollLeftImageIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                        GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                        0,0,w,h);

#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                          GTK_ARROW_LEFT,
                          true,
                          0,0,w,h);
#else
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                          GTK_ARROW_LEFT,
                          true,
                          (w-w/2)/2,(h-h/2)/2,w/2,h/2);
#endif
          break;
        case Display::arrowScrollRightImageIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                        GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                        0,0,w,h);

#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                          GTK_ARROW_RIGHT,
                          true,
                          0,0,w,h);
#else
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                          GTK_ARROW_RIGHT,
                          true,
                          (w-w/2)/2,(h-h/2)/2,w/2,h/2);
#endif
          break;
        case Display::arrowScrollUpImageIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                        GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                        0,0,w,h);

#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                          GTK_ARROW_UP,
                          true,
                          0,0,w,h);
#else
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                          GTK_ARROW_UP,
                          true,
                          (w-w/2)/2,(h-h/2)/2,w/2,h/2);
#endif
          break;
        case Display::arrowScrollDownImageIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                        GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                        0,0,w,h);

#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                          GTK_ARROW_DOWN,
                          true,
                          0,0,w,h);
#else
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                          GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob])->stepper_detail,
                          GTK_ARROW_DOWN,
                          true,
                          (w-w/2)/2,(h-h/2)/2,w/2,h/2);
#endif
          break;
        case Display::comboImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleComboBox),
                          gtkDraw,
                          GetGtkStateType(draw),
                          GTK_SHADOW_NONE,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleComboBox],
                          "arrow",
                          GTK_ARROW_DOWN,
                          false,
                          0,0,w,h);
          break;
        case Display::comboEditButtonImageIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleButton);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),
                        draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleButton],
                        buttonDetail,
                        0,0,w,h);

          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          NULL,
                          GTK_ARROW_DOWN,
                          true,
                          3,3,w-6,h-6);
          break;
        case Display::comboDividerImageIndex:
          gtk_paint_vline(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVLine),
                          gtkDraw,
                          GTK_STATE_NORMAL,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleVLine],
                          NULL,
                          0,h-1,0);
          break;
        case Display::treeExpanderImageIndex:
          gtk_paint_expander(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                             gtkDraw,
                             GTK_STATE_NORMAL,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleList],
                             "treeview",
                             width/2,height/2,
                             draw->selected ? GTK_EXPANDER_EXPANDED : GTK_EXPANDER_COLLAPSED);
          break;
        case Display::hscaleKnobImageIndex:
          gtk_paint_slider(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHScale),
                           gtkDraw,
                           GetGtkRangeStateType(draw,gtkWidgets->widget[GtkWidgets::gtkStyleHScale]),
                           draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                           &rectangle,
                           gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                           "hscale",
                           0,0,w,h,
                           GTK_ORIENTATION_HORIZONTAL);
          break;
        case Display::vscaleKnobImageIndex:
          gtk_paint_slider(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVScale),
                           gtkDraw,
                           GetGtkRangeStateType(draw,gtkWidgets->widget[GtkWidgets::gtkStyleVScale]),
                           draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                           &rectangle,
                           gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                           "vscale",
                           0,0,w,h,
                           GTK_ORIENTATION_VERTICAL);
          break;
        case Display::leftSliderImageIndex:
          gtk_adjustment_set_value(gtkWidgets->adjustment,100);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHScale),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::bottomSliderImageIndex:
          gtk_adjustment_set_value(gtkWidgets->adjustment,100);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVScale),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::rightSliderImageIndex:
          gtk_adjustment_set_value(gtkWidgets->adjustment,1);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHScale),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::topSliderImageIndex:
          gtk_adjustment_set_value(gtkWidgets->adjustment,1);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVScale),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::menuDividerImageIndex:
          gtk_paint_hline(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuItem),
                          gtkDraw,
                          GTK_STATE_NORMAL,
                          &rectangle,
                          NULL,
                          NULL,
                          0,w-1,0);
          break;
        case Display::menuSubImageIndex:
          gtk_paint_arrow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "menuitem",
                          GTK_ARROW_RIGHT,
                          false,
                          0,0,w,h);
          break;
        case Display::positiveImageIndex:
          gdk_pixbuf_xlib_render_to_drawable_alpha(gtkWidgets->positiveIcon,pixmap,
                                                   0,0,0,0,w,h,
                                                   GDK_PIXBUF_ALPHA_FULL,128,
                                                   XLIB_RGB_DITHER_NORMAL,0,0);
          break;
        case Display::negativeImageIndex:
          gdk_pixbuf_xlib_render_to_drawable_alpha(gtkWidgets->negativeIcon,pixmap,
                                                   0,0,0,0,w,h,
                                                   GDK_PIXBUF_ALPHA_FULL,128,
                                                   XLIB_RGB_DITHER_NORMAL,0,0);
          break;
        case Display::defaultImageIndex:
          gdk_pixbuf_xlib_render_to_drawable_alpha(gtkWidgets->defaultIcon,pixmap,
                                                   0,0,0,0,w,h,
                                                   GDK_PIXBUF_ALPHA_FULL,128,
                                                   XLIB_RGB_DITHER_NORMAL,0,0);
          break;
        case Display::helpImageIndex:
          gdk_pixbuf_xlib_render_to_drawable_alpha(gtkWidgets->helpIcon,pixmap,
                                                   0,0,0,0,w,h,
                                                   GDK_PIXBUF_ALPHA_FULL,128,
                                                   XLIB_RGB_DITHER_NORMAL,0,0);
          break;
        case Display::closeImageIndex:
          gdk_pixbuf_xlib_render_to_drawable_alpha(gtkWidgets->closeIcon,pixmap,
                                                   0,0,0,0,w,h,
                                                   GDK_PIXBUF_ALPHA_FULL,128,
                                                   XLIB_RGB_DITHER_NORMAL,0,0);
          break;
        }

        if (dynamic_cast<DrawInfo*>(draw)!=NULL) {
          ::XCopyArea(display,pixmap,drawable,targetGC,0,0,w,h,x,y);
        }
#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        else if (dynamic_cast<Cairo::DrawInfo*>(draw)!=NULL) {
          cairo_t *cairo;

          cairo=dynamic_cast<Cairo::DrawInfo*>(draw)->cairo;

          ::cairo_save(cairo);
          ::gdk_cairo_set_source_pixmap(cairo,gtkDraw,x,y);
          ::cairo_rectangle(cairo,x,y,w,h);
          ::cairo_fill(cairo);
          ::cairo_stroke(cairo);
          ::cairo_restore(cairo);
        }
#endif
        else {
          assert(true);
        }

          g_object_unref(gtkDraw);

        ::XFreeGC(display,gc);
        ::XFreePixmap(display,pixmap);
      }

      GtkFill::GtkFill(Display::FillIndex fill)
      : fill(fill)
      {
        // no code
      }

      void GtkFill::Draw(OS::DrawInfo* draw,
                         int xOff, int yOff, size_t width, size_t height,
                         int x, int y, size_t w, size_t h)
      {
        ::Display      *display=dynamic_cast<Display*>(OS::display)->display;
        ::GdkPixmap    *gtkDraw;
        ::GdkRectangle rectangle;
        size_t         xframe;
        size_t         yframe;

        if (w==0 || h==0) { // TODO: Make this an assertion?
          return;
        }

        // The rectangle defined by x,y,w,h must allways be completely within the rectangle
        // defined by xOff,yOff,width,height!
        assert(x>=xOff && y>=yOff && x+w<=xOff+width && y+h<=yOff+height);

        // Handle fills, that can directly draw to the destination surface...
        switch (fill) {
        case Display::helpBackgroundFillIndex:
          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->helpBackgroundColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          return;
        case Display::entryBackgroundFillIndex:
          if (draw->selected) {
            if (draw->focused) {
              draw->PushForeground(gtkWidgets->GetStyle(GtkWidgets::gtkStyleEdit)->base[GTK_STATE_SELECTED].pixel);
              draw->FillRectangle(x,y,w,h);
              draw->PopForeground();
            }
            else {
              draw->PushForeground(gtkWidgets->GetStyle(GtkWidgets::gtkStyleEdit)->base[GTK_STATE_ACTIVE].pixel);
              draw->FillRectangle(x,y,w,h);
              draw->PopForeground();
            }
            return;
          }
          break;
        case Display::graphBackgroundFillIndex:
          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->whiteColor);
          draw->FillRectangle(x,y,w,h);
          draw->PopForeground();
          return;
        default:
          break;
        }

        switch (fill) {
        case Display::buttonBackgroundFillIndex:
        case Display::positiveButtonBackgroundFillIndex:
        case Display::negativeButtonBackgroundFillIndex:
        case Display::defaultButtonBackgroundFillIndex:
        case Display::scrollButtonBackgroundFillIndex:
        case Display::comboBackgroundFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->xthickness;
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->ythickness;
          break;
        case Display::toolbarButtonBackgroundFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]->style->xthickness;
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]->style->ythickness;
          break;
        case Display::hprogressBackgroundFillIndex:
        case Display::vprogressBackgroundFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleProgress]->style->xthickness;
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleProgress]->style->ythickness;
          break;
        case Display::hknobBoxFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->xthickness;
          yframe=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                        "trough-border");
          break;
        case Display::vknobBoxFillIndex:
          xframe=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                        "trough-border");
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->ythickness;
          break;
        case Display::tabBackgroundFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->xthickness;
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->ythickness;
          break;
        case Display::tabRiderBackgroundFillIndex:
          xframe=2;
          yframe=2;
          break;
        case Display::columnBackgroundFillIndex:
          xframe=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->xthickness;
          yframe=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->ythickness;
          break;
        default:
          xframe=0;
          yframe=0;
          break;
        }

        // We cannot draw into the negative area...
        // TODO: Normally this should be fixed by the calling widgets, but we do it
        // here at a central place...

        if (xOff-(int)xframe<0) {
          width+=xOff-(int)xframe;
          xOff=xframe;
        }

        if (yOff-(int)yframe<0) {
          height+=yOff-(int)yframe;
          yOff=yframe;
        }

        if (x<0) {
          w+=x;
          x=0;
        }

        if (y<0) {
          h+=y;
          y=0;
        }

        rectangle.x=x-xOff+xframe;
        rectangle.y=y-yOff+yframe;
        rectangle.width=w;
        rectangle.height=h;

        gtkDraw=gdk_pixmap_new(NULL,
                               width+2*xframe,height+2*yframe,
                               OS::display->GetColorDepth());
        if (gtkDraw==NULL) {
          std::cerr << "Cannot create GdkDrawable" << std::endl;
          return;
        }

        switch (fill) {
        case Display::backgroundFillIndex:
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                             gtkDraw,
                             GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                             NULL,
                             0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::tableBackgroundFillIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleList);
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
           gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                             gtkDraw,
                             draw->selected ? GTK_STATE_SELECTED : GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleList],
                             "cell",
                             0,0,width+2*xframe,height+2*yframe);
#else	
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                             gtkDraw,
                             draw->selected ? GTK_STATE_SELECTED : GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleList],
                             "cell_odd",
                             0,0,width+2*xframe,height+2*yframe);
#endif	
          break;
        case Display::tableBackground2FillIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleList);
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
           gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                              gtkDraw,
                              draw->selected ? GTK_STATE_SELECTED : GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                              &rectangle,
                              gtkWidgets->widget[GtkWidgets::gtkStyleList],
                              "cell",
                              0,0,width+2*xframe,height+2*yframe);
#else	
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                             gtkDraw,
                             draw->selected ? GTK_STATE_SELECTED : GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleList],
                             "cell_even",
                             0,0,width+2*xframe,height+2*yframe);
#endif	
          break;
        case Display::buttonBackgroundFillIndex:
        case Display::positiveButtonBackgroundFillIndex:
        case Display::negativeButtonBackgroundFillIndex:
        case Display::defaultButtonBackgroundFillIndex:
        case Display::scrollButtonBackgroundFillIndex:
        case Display::comboBackgroundFillIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleButton);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),
                        draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleButton],
                        buttonDetail,
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::toolbarButtonBackgroundFillIndex:
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleToolButton),
                        gtkDraw,
                        GetGtkStateType(draw),
                        draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleToolButton],
                        buttonDetail,
                        0,0,width+2*xframe,height+2*yframe);
#else
          if (gtk_button_get_relief(GTK_BUTTON(gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]))!=GTK_RELIEF_NONE ||
              draw->activated) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleToolButton),
                          gtkDraw,
                          GetGtkStateType(draw),
                          draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleToolButton],
                          buttonDetail,
                          0,0,width+2*xframe,height+2*yframe);
          }
          else {
#if defined(HAVE_LIB_HILDON)
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                             gtkDraw,
                             GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                             NULL,
                             0,0,width+2*xframe,height+2*yframe);
#else
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleToolbar),
                          gtkDraw,
                          GTK_STATE_NORMAL,
                          (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleToolbar],
                                                                "shadow-type"),
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleToolbar],
                          "toolbar",
                          0,0,width+2*xframe,height+2*yframe);
#endif
          }
#endif
          break;
        case Display::entryBackgroundFillIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleEdit);
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleEdit),
                             gtkDraw,
                             GetGtkStateType(draw),GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleEdit],
                             "entry_bg",
                             0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::hprogressBackgroundFillIndex:
        case Display::vprogressBackgroundFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleProgress),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleProgress],
                        "trough",
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::hprogressFillIndex:
        case Display::vprogressFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleProgress),
                        gtkDraw,
                        GTK_STATE_PRELIGHT,GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleProgress],
                        "bar",
                        rectangle.x,rectangle.y,rectangle.width,rectangle.height);
          break;
        case Display::hknobBoxFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                        gtkDraw,
                        GTK_STATE_ACTIVE,GTK_SHADOW_IN,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                        "trough",
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::vknobBoxFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                        gtkDraw,
                        GTK_STATE_ACTIVE,GTK_SHADOW_IN,
                        NULL,//&rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                        "trough",
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::tabBackgroundFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleTab),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleTab],
                        "notebook",
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::tabRiderBackgroundFillIndex:
          gtk_paint_extension(gtkWidgets->GetStyle(GtkWidgets::gtkStyleTab),
                              gtkDraw,
                              draw->selected ? GTK_STATE_NORMAL : GTK_STATE_ACTIVE,GTK_SHADOW_OUT,
                              &rectangle,
                              gtkWidgets->widget[GtkWidgets::gtkStyleTab],
                              "tab",
                              0,0,width+2*xframe,height+2*yframe,
                              GTK_POS_BOTTOM);
          break;
        case Display::columnBackgroundFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_OUT,
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleColumn],
                        buttonDetail,
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::menuStripBackgroundFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuBar),
                        gtkDraw,
                        GTK_STATE_NORMAL,
                        (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleMenuBar],
                                                              "shadow-type"),
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleMenuBar],
                        "menubar",
                        0,0,width+2*xframe,height+2*yframe);
          break;
        case Display::menuPulldownBackgroundFillIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuBar),
                        gtkDraw,
                        GTK_STATE_NORMAL,
                        (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleMenuBar],
                                                              "shadow-type"),
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleMenuBar],
                        "menubar",
                        0,0,width+2*xframe,height+2*yframe);
          if (draw->selected) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuPullDown),
                          gtkDraw,
                          GTK_STATE_PRELIGHT,
                          (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleMenuPullDown],
                                                                "selected-shadow-type"),
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleMenuPullDown],
                          "menuitem",
                          0,0,width+2*xframe,height+2*yframe);
          }
          break;
        case Display::menuEntryBackgroundFillIndex:
            gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuItem),
                               gtkDraw,
                               GTK_STATE_NORMAL,
                               GTK_SHADOW_NONE,
                               &rectangle,
                               gtkWidgets->widget[GtkWidgets::gtkStyleMenuItem],
                               "menuitem",
                               0,0,width+2*xframe,height+2*yframe);
          if (draw->selected) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenuItem),
                          gtkDraw,
                          GTK_STATE_PRELIGHT,
                          (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleMenuItem],
                                                                "selected-shadow-type"),
                          &rectangle,
                          gtkWidgets->widget[GtkWidgets::gtkStyleMenuItem],
                          "menuitem",
                          0,0,width+2*xframe,height+2*yframe);
          }
          break;
        case Display::toolbarBackgroundFillIndex:
#if defined(HAVE_LIB_HILDON)
          gtk_paint_flat_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                             gtkDraw,
                             GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                             &rectangle,
                             gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                             NULL,
                             0,0,width+2*xframe,height+2*yframe);
#else
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleToolbar),
                        gtkDraw,
                        GTK_STATE_NORMAL,
                        (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleToolbar],
                                                              "shadow-type"),
                        &rectangle,
                        gtkWidgets->widget[GtkWidgets::gtkStyleToolbar],
                        "toolbar",
                        0,0,width+2*xframe,height+2*yframe);
#endif
          break;
        default:
          assert(false);
          break;
        }


        if (dynamic_cast<DrawInfo*>(draw)!=NULL) {
          ::Drawable drawable=dynamic_cast<DrawInfo*>(draw)->drawable;
          ::GC       targetGC=dynamic_cast<DrawInfo*>(draw)->gc;

          ::XCopyArea(display,gdk_x11_drawable_get_xid(gtkDraw),drawable,targetGC,x-xOff+xframe,y-yOff+yframe,w,h,x,y);
        }
#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        else if (dynamic_cast<Cairo::DrawInfo*>(draw)!=NULL) {
          ::cairo_t *cairo=dynamic_cast<Cairo::DrawInfo*>(draw)->cairo;

          ::cairo_save(cairo);
          ::gdk_cairo_set_source_pixmap(cairo,gtkDraw,xOff-xframe,yOff-yframe);
          ::cairo_rectangle(cairo,x,y,w,h);
          ::cairo_fill(cairo);
          ::cairo_stroke(cairo);
          ::cairo_restore(cairo);
        }
#endif
        else {
          assert(true);
        }

        g_object_unref(gtkDraw);
      }

      GtkFrame::GtkFrame(Display::FrameIndex frame)
      : frame(frame)
      {
        alpha=true;
        switch (frame) {
        case Display::buttonFrameIndex:
        case Display::positiveButtonFrameIndex:
        case Display::negativeButtonFrameIndex:
        case Display::defaultButtonFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::toolbarButtonFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::scrollButtonFrameIndex:
          leftBorder=0;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
          break;
        case Display::editFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleEdit]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleEdit]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::hprogressFrameIndex:
        case Display::vprogressFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleProgress]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleProgress]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::hknobBoxFrameIndex:
          if (GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],"stepper-spacing")==0) {
            leftBorder=0;
            rightBorder=0;
            topBorder=0;
            bottomBorder=0;
          }
          else {
            leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");
            rightBorder=leftBorder;
            topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");
            bottomBorder=topBorder;
          }
          break;
        case Display::vknobBoxFrameIndex:
          if (GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],"stepper-spacing")==0) {
            leftBorder=0;
            rightBorder=0;
            topBorder=0;
            bottomBorder=0;
          }
          else {
            leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                              "trough-border");
            rightBorder=leftBorder;
            topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");
            bottomBorder=bottomBorder;
          }
          break;
        case Display::hscrollFrameIndex:
          if (GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],"stepper-spacing")==0) {
            leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");;
            rightBorder=leftBorder;
            topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");
            bottomBorder=topBorder;
          }
          else {
            leftBorder=0;
            rightBorder=0;
            topBorder=0;
            bottomBorder=0;
          }
          break;
        case Display::vscrollFrameIndex:
          if (GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],"stepper-spacing")==0) {
            leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                                              "trough-border");
            rightBorder=leftBorder;
            topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                                             "trough-border");;
            bottomBorder=bottomBorder;
          }
          else {
            leftBorder=0;
            rightBorder=0;
            topBorder=0;
            bottomBorder=0;
          }
          break;
        case Display::tabFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::tabRiderFrameIndex:
          leftBorder=2;
          rightBorder=leftBorder;
          topBorder=2;
          bottomBorder=topBorder;
          break;
        case Display::listboxFrameIndex:
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          leftBorder=0;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
#else
          if (gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->xthickness!=0 ||
              gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->ythickness!=0) {
            leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->xthickness;
            rightBorder=leftBorder;
            topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->ythickness;
            bottomBorder=topBorder;
          }
          else {
            leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleList]->style->xthickness;
            rightBorder=leftBorder;
            topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleList]->style->ythickness;
            bottomBorder=topBorder;
          }
#endif
          break;
        case Display::scrolledFrameIndex:
          leftBorder=0;//gtkWidgets->widget[GtkWidgets::gtkStyleList]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=0;//gtkWidgets->widget[GtkWidgets::gtkStyleList]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::columnFrameIndex:
        case Display::columnLeftFrameIndex:
        case Display::columnMiddleFrameIndex:
        case Display::columnRightFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::mainWindowFrameIndex:
#if defined(HAVE_LIB_HILDON)
          {
            GtkBorder *border;

            gtk_widget_style_get(gtkWidgets->widget[GtkWidgets::gtkStyleWindow],"borders",&border,NULL);

            leftBorder=border->left;
            rightBorder=border->right;
            topBorder=border->top;
            bottomBorder=border->bottom;

            g_free(border);
          }
#else
          leftBorder=0;
          rightBorder=leftBorder;
          topBorder=0;
          bottomBorder=topBorder;
#endif
          break;
        case Display::menuWindowFrameIndex:
        case Display::popupWindowFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleColumn]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::hscaleFrameIndex:
          leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHScale],"trough-border");
          rightBorder=leftBorder;
          topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleHScale],"trough-border");
          bottomBorder=topBorder;
          break;
        case Display::vscaleFrameIndex:
          leftBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVScale],"trough-border");
          rightBorder=leftBorder;
          topBorder=GetWidgetStyleIntValue(gtkWidgets->widget[GtkWidgets::gtkStyleVScale],"trough-border");
          bottomBorder=topBorder;
          break;
        case Display::focusFrameIndex:
          leftBorder=1;
          rightBorder=1;
          topBorder=1;
          bottomBorder=1;
          break;
        case Display::statuscellFrameIndex:
          leftBorder=gtkWidgets->widget[GtkWidgets::gtkStyleStatus]->style->xthickness;
          rightBorder=leftBorder;
          topBorder=gtkWidgets->widget[GtkWidgets::gtkStyleStatus]->style->ythickness;
          bottomBorder=topBorder;
          break;
        case Display::plateFrameIndex:
          leftBorder=1;
          rightBorder=1;
          topBorder=1;
          bottomBorder=1;
          break;
        case Display::valuebarFrameIndex:
          leftBorder=1;
          rightBorder=1;
          topBorder=1;
          bottomBorder=1;
          break;
        case Display::groupFrameIndex:
          leftBorder=2;
          rightBorder=2;
          topBorder=2;
          bottomBorder=2;
          break;
        }

        minWidth=leftBorder+rightBorder;
        minHeight=topBorder+bottomBorder;

        if (minWidth==0 && minHeight==0) {
          alpha=false;
        }
      }

      void GtkFrame::Draw(OS::DrawInfo* draw, int x, int y, size_t w, size_t h)
      {
        ::Display         *display=dynamic_cast<Display*>(OS::display)->display;
        ::Pixmap          pixmap;
        GdkDrawable       *gtkDraw;
        unsigned long     mask;
        XGCValues         values;
        ::GC              gc;

        if (minWidth==0 && minHeight==0) {
          return;
        }

        if (x<0 && -x>w) {
          return;
        }

        if (y<0 && -y>h) {
          return;
        }

        if (x<0) {
          w+=x;
          x=0;
        }

        if (y<0) {
          h+=y;
          y=0;
        }

        switch (frame) {
        case Display::plateFrameIndex:
          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->blackColor);
          draw->DrawRectangle(x,y,w,h);
          draw->PopForeground();
          return;
        case Display::valuebarFrameIndex:
          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shineColor);
          draw->DrawLine(x,y+h-1,x,y+1);
          draw->DrawLine(x,y,x+w-1,y);
          draw->PopForeground();

          draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shadowColor);
          draw->DrawLine(x+w-1,y+1,x+w-1,y+h-1);
          draw->DrawLine(x+w-1,y+h-1,x,y+h-1);
          draw->PopForeground();
          return;
        case Display::groupFrameIndex:
          {
            int top;

            if (HasGap()) {
              top=y+gh / 2;
            }
            else {
              top=y;
            }
            draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shadowColor);
            draw->DrawLine(x,y+h-1,x,top);
            draw->DrawLine(x+w-2,top+1,x+w-2,y+h-2);
            draw->DrawLine(x+1,y+h-2,x+w-2,y+h-2);
            if (HasGap()) {
              draw->DrawLine(x,top,x+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)-1,top);
              draw->DrawLine(x+3*OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+gw+1,top,x+w-1,top);
              draw->DrawLine(x+3*OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+gw,top,
                             x+3*OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+gw,top+1);
            }
            else {
              draw->DrawLine(x,top,x+w-1,top);
            }
            draw->PopForeground();

            draw->PushForeground(dynamic_cast<GtkTheme*>(OS::display->GetTheme())->shineColor);
            draw->DrawLine(x+1,y+h-2,x+1,top+1);
            draw->DrawLine(x+w-1,top+1,x+w-1,y+h-1);
            draw->DrawLine(x+1,y+h-1,x+w-2,y+h-1);
            if (HasGap()) {
              draw->DrawLine(x+2,top+1,x+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)-1,top+1);
              draw->DrawLine(x+3*OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+gw+1,top+1,x+w-2,top+1);
              draw->DrawLine(x+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),top,x+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),top+1);
            }
            else {
              draw->DrawLine(x+2,top+1,x+w-2,top+1);
            }
            draw->PopForeground();
          }
          return;
        default:
          break;
        }

        pixmap=::XCreatePixmap(display,
                               ::XDefaultRootWindow(display),
                               w,h,
                               OS::display->GetColorDepth());

        if (pixmap==0) {
          std::cerr << "Cannot create Pixmap" << std::endl;
          return;
        }

        mask=0;
        gc=::XCreateGC(display,pixmap,mask,&values);
        if (gc==0) {
          std::cerr << "Cannot create GC" << std::endl;
          XFreePixmap(display,pixmap);
          return;
        }

        gtkDraw=gdk_pixmap_foreign_new(pixmap);
        if (gtkDraw==NULL) {
          std::cerr << "Cannot create GdkDrawable" << std::endl;
          XFreeGC(display,gc);
          XFreePixmap(display,pixmap);
          return;
        }

        gdk_drawable_set_colormap(gtkDraw,gdk_colormap_get_system());

        ::Drawable drawable=0;
        ::GC       targetGC=0;

        if (dynamic_cast<DrawInfo*>(draw)!=NULL) {
          drawable=dynamic_cast<DrawInfo*>(draw)->drawable;
          targetGC=dynamic_cast<DrawInfo*>(draw)->gc;
        }
#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        else if (dynamic_cast<Cairo::X11DrawInfo*>(draw)!=NULL) {
          drawable=dynamic_cast<Cairo::X11DrawInfo*>(draw)->drawable;
        }
#endif
        else {
          assert(true);
        }

        ::XCopyArea(display,
                    drawable,
                    pixmap,
                    gc,
                    x,y,w,h,
                    0,0);

        switch (frame) {
        case Display::buttonFrameIndex:
        case Display::positiveButtonFrameIndex:
        case Display::negativeButtonFrameIndex:
        case Display::defaultButtonFrameIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleButton);
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleButton],
                        buttonDetail,
                        0,0,w,h);
          break;
        case Display::toolbarButtonFrameIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleButton);
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleButton],
                        buttonDetail,
                        0,0,w,h);
#else	
          if (gtk_button_get_relief(GTK_BUTTON(gtkWidgets->widget[GtkWidgets::gtkStyleToolButton]))!=GTK_RELIEF_NONE ||
              draw->activated) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleToolButton),
                          gtkDraw,
                          GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                          NULL,
                          gtkWidgets->widget[GtkWidgets::gtkStyleToolButton],
                          buttonDetail,
                          0,0,w,h);
          }
#endif	
          break;
        case Display::scrollButtonFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                        gtkDraw,
                        GetGtkStateType(draw),draw->selected ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                        GTK_RANGE_GET_CLASS(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob])->stepper_detail,
                        0,0,w,h);
          break;
        case Display::editFrameIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleEdit);
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleEdit),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_NONE,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleEdit],
                        "entry_frame",
                        0,0,w,h);
#else
          gtk_paint_shadow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleEdit),
                           gtkDraw,
                           GTK_STATE_NORMAL,GTK_SHADOW_IN,
                           NULL,
                           gtkWidgets->widget[GtkWidgets::gtkStyleEdit],
                           "entry",
                           0,0,w,h);
#endif
          break;
        case Display::hprogressFrameIndex:
        case Display::vprogressFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleProgress),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_IN,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleProgress],
                        "trough",
                        0,0,w,h);
          break;
        case Display::hknobBoxFrameIndex:
        case Display::hscrollFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHKnob),
                        gtkDraw,
                        GTK_STATE_ACTIVE,GTK_SHADOW_IN,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],
                        "trough",
                        0,0,w,h);
          break;
        case Display::vknobBoxFrameIndex:
        case Display::vscrollFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVKnob),
                        gtkDraw,
                        GTK_STATE_ACTIVE,GTK_SHADOW_IN,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVKnob],
                        "trough",
                        0,0,w,h);
          break;
        case Display::tabFrameIndex:
          gtk_paint_box_gap(gtkWidgets->GetStyle(GtkWidgets::gtkStyleTab),
                            gtkDraw,
                            GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                            NULL,
                            gtkWidgets->widget[GtkWidgets::gtkStyleTab],
                            "notebook",
                            0,0,w,h,
                            GTK_POS_TOP,
                            gx,gw);
          break;
        case Display::tabRiderFrameIndex:
          gtk_paint_extension(gtkWidgets->GetStyle(GtkWidgets::gtkStyleTab),
                              gtkDraw,
                              draw->selected ? GTK_STATE_NORMAL : GTK_STATE_ACTIVE,
                              GTK_SHADOW_OUT,
                              NULL,
                              gtkWidgets->widget[GtkWidgets::gtkStyleTab],
                              "tab",
                              0,0,w,h,
                              GTK_POS_BOTTOM);
          break;
        case Display::listboxFrameIndex:
        case Display::scrolledFrameIndex:
          if (gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->xthickness!=0 ||
              gtkWidgets->widget[GtkWidgets::gtkStyleScrolled]->style->ythickness!=0) {
            GtkShadowType shadowType;

            g_object_get(gtkWidgets->widget[GtkWidgets::gtkStyleScrolled],
                         "shadow-type",&shadowType,NULL);
            gtk_paint_shadow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleScrolled),
                             gtkDraw,
                             GTK_STATE_NORMAL,shadowType,
                             NULL,
                             gtkWidgets->widget[GtkWidgets::gtkStyleScrolled],
                             "scrolled_window",
                             0,0,w,h);
          }
          else {
            gtk_paint_shadow(gtkWidgets->GetStyle(GtkWidgets::gtkStyleList),
                             gtkDraw,
                             GTK_STATE_NORMAL,GTK_SHADOW_IN,
                             NULL,
                             gtkWidgets->widget[GtkWidgets::gtkStyleList],
                             NULL,
                             0,0,w,h);
          }
          break;
        case Display::columnFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleColumn],
                        buttonDetail,
                        0,0,w,h);
          break;
        case Display::columnLeftFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleColumnLeft],
                        buttonDetail,
                        0,0,w,h);
          break;
        case Display::columnMiddleFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleColumnMiddle],
                        buttonDetail,
                        0,0,w,h);
          break;
        case Display::columnRightFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                        gtkDraw,
                        GetGtkStateType(draw),GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleColumnRight],
                        buttonDetail,
                        0,0,w,h);
          break;
#if defined(HAVE_LIB_HILDON)
        case Display::mainWindowFrameIndex:
          if (leftBorder>0) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                          NULL,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "left-border",
                          0,topBorder,leftBorder,h-topBorder-bottomBorder);
          }
          if (rightBorder>0) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                          NULL,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "right-border",
                          w-rightBorder,topBorder,rightBorder,h-topBorder-bottomBorder);
          }
          if (topBorder>0) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                          NULL,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "top-border",
                          0,0,w,topBorder);
          }
          if (bottomBorder>0) {
            gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleWindow),
                          gtkDraw,
                          GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                          NULL,
                          gtkWidgets->widget[GtkWidgets::gtkStyleWindow],
                          "bottom-border",
                          0,h-bottomBorder,w,bottomBorder);
          }
          break;
#endif
        case Display::menuWindowFrameIndex:
        case Display::popupWindowFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleMenu),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_OUT,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleMenu],
                        "menu",
                        0,0,w,h);
          break;
        case Display::hscaleFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleHScale),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_IN,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleHScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::vscaleFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleVScale),
                        gtkDraw,
                        GTK_STATE_NORMAL,GTK_SHADOW_IN,
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleVScale],
                        "trough",
                        0,0,w,h);
          break;
        case Display::focusFrameIndex:
          SetFocusIfRequested(draw,GtkWidgets::gtkStyleButton);
          gtk_paint_focus(gtkWidgets->GetStyle(GtkWidgets::gtkStyleButton),
                          gtkDraw,
                          GTK_STATE_NORMAL,
                          NULL,
                          gtkWidgets->GetWidget(GtkWidgets::gtkStyleButton),
                          "button",
                          0,0,w,h);
          break;
        case Display::statuscellFrameIndex:
          gtk_paint_box(gtkWidgets->GetStyle(GtkWidgets::gtkStyleStatus),
                        gtkDraw,
                        GTK_STATE_NORMAL,
                        (GtkShadowType)GetWidgetStyleIntValue(gtkWidgets->GetWidget(GtkWidgets::gtkStyleStatus),"shadow-type"),
                        NULL,
                        gtkWidgets->widget[GtkWidgets::gtkStyleStatus],
                        "statusbar",
                        0,0,w,h);
          break;
        default:
          assert(false);
          break;
        }

        draw->PushClipBegin(x,y,w,h);
        draw->SubClipRegion(x+leftBorder,y+topBorder,w-minWidth,h-minHeight);
        draw->PushClipEnd();

        if (dynamic_cast<DrawInfo*>(draw)!=NULL) {
          ::XCopyArea(display,pixmap,drawable,targetGC,0,0,w,h,x,y);
        }
#if defined(HAVE_LIB_CAIRO) && defined(CAIRO_HAS_XLIB_SURFACE)
        else if (dynamic_cast<Cairo::DrawInfo*>(draw)!=NULL) {
          ::cairo_t *cairo;

          cairo=dynamic_cast<Cairo::DrawInfo*>(draw)->cairo;

          ::cairo_save(cairo);
          ::gdk_cairo_set_source_pixmap(cairo,gtkDraw,x,y);
          ::cairo_rectangle(cairo,x,y,w,h);
          ::cairo_fill(cairo);
          ::cairo_stroke(cairo);
          ::cairo_restore(cairo);
        }
#endif
        else {
          assert(true);
        }

        draw->PopClip();

        g_object_unref(gtkDraw);

        ::XFreeGC(display,gc);
        ::XFreePixmap(display,pixmap);
      }

      int GetWidgetStyleIntValue(GtkWidget *widget, const char* name)
      {
        GValue gvalue;

        memset(&gvalue,0,sizeof(GValue));
        g_value_init(&gvalue,G_TYPE_INT);
        gtk_widget_style_get_property(widget,name,&gvalue);
        return g_value_get_int(&gvalue);
      }

      bool GetWidgetStyleBoolValue(GtkWidget *widget, const char* name)
      {
        GValue gvalue;

        memset(&gvalue,0,sizeof(GValue));
        g_value_init(&gvalue,G_TYPE_BOOLEAN);
        gtk_widget_style_get_property(widget,name,&gvalue);
        return g_value_get_boolean(&gvalue);
      }

      void GetWidgetStyleBorderValue(GtkWidget *widget, const char* name,
                                     int& left, int&right, int &top, int& bottom)
      {
        GtkBorder *border=NULL;

        gtk_widget_style_get(widget,name,&border,NULL);
        if (border!=NULL) {
          left=border->left;
          right=border->right;
          top=border->top;
          bottom=border->bottom;
          g_free(border);
        }
      }

      GtkTheme::GtkTheme(OS::Display *display)
      : OS::Base::Theme(display)
      {
        gtkWidgets=new GtkWidgets();

        color[Display::backgroundColor]=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->bg[GTK_STATE_NORMAL].pixel;
        color[Display::textColor]=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->text[GTK_STATE_NORMAL].pixel;
        color[Display::tableTextColor]=gtkWidgets->widget[GtkWidgets::gtkStyleEdit]->style->text[GTK_STATE_NORMAL].pixel;
        color[Display::textSelectColor]=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->text[GTK_STATE_SELECTED].pixel;
        color[Display::textDisabledColor]=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->text[GTK_STATE_INSENSITIVE].pixel;
        color[Display::fillColor]=gtkWidgets->widget[GtkWidgets::gtkStyleEdit]->style->base[GTK_STATE_SELECTED].pixel;
        color[Display::fillTextColor]=gtkWidgets->widget[GtkWidgets::gtkStyleEdit]->style->text[GTK_STATE_SELECTED].pixel;
        color[Display::graphScaleColor]=gtkWidgets->widget[GtkWidgets::gtkStyleButton]->style->text[GTK_STATE_INSENSITIVE].pixel;
        color[Display::blackColor]=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->black.pixel;
        color[Display::whiteColor]=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->white.pixel;
        color[Display::tabTextColor]=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->text[GTK_STATE_NORMAL].pixel;
        color[Display::tabTextSelectColor]=gtkWidgets->widget[GtkWidgets::gtkStyleTab]->style->text[GTK_STATE_NORMAL].pixel; // Common for all themes?

        display->AllocateNamedColor("light yellow",helpBackgroundColor);

        whiteColor=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->white.pixel;
        blackColor=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->black.pixel;
        shineColor=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->light[GTK_STATE_NORMAL].pixel;
        shadowColor=gtkWidgets->widget[GtkWidgets::gtkStyleWindow]->style->dark[GTK_STATE_NORMAL].pixel;

        fill[Display::backgroundFillIndex]=new GtkFill(Display::backgroundFillIndex);

        fill[Display::tableBackgroundFillIndex]=new GtkFill(Display::tableBackgroundFillIndex);
        fill[Display::tableBackground2FillIndex]=new GtkFill(Display::tableBackground2FillIndex);
        fill[Display::helpBackgroundFillIndex]=new GtkFill(Display::helpBackgroundFillIndex);

        fill[Display::buttonBackgroundFillIndex]=new GtkFill(Display::buttonBackgroundFillIndex);
        fill[Display::positiveButtonBackgroundFillIndex]=new GtkFill(Display::positiveButtonBackgroundFillIndex);
        fill[Display::negativeButtonBackgroundFillIndex]=new GtkFill(Display::negativeButtonBackgroundFillIndex);
        fill[Display::defaultButtonBackgroundFillIndex]=new GtkFill(Display::defaultButtonBackgroundFillIndex);
        fill[Display::toolbarButtonBackgroundFillIndex]=new GtkFill(Display::toolbarButtonBackgroundFillIndex);
        fill[Display::scrollButtonBackgroundFillIndex]=new GtkFill(Display::scrollButtonBackgroundFillIndex);
        fill[Display::comboBackgroundFillIndex]=new GtkFill(Display::comboBackgroundFillIndex);
        fill[Display::entryBackgroundFillIndex]=new GtkFill(Display::entryBackgroundFillIndex);
        fill[Display::hprogressBackgroundFillIndex]=new GtkFill(Display::hprogressBackgroundFillIndex);
        fill[Display::hprogressFillIndex]=new GtkFill(Display::hprogressFillIndex);
        fill[Display::vprogressBackgroundFillIndex]=new GtkFill(Display::vprogressBackgroundFillIndex);
        fill[Display::vprogressFillIndex]=new GtkFill(Display::vprogressFillIndex);
        fill[Display::hknobBoxFillIndex]=new GtkFill(Display::hknobBoxFillIndex);
        fill[Display::vknobBoxFillIndex]=new GtkFill(Display::vknobBoxFillIndex);
        fill[Display::tabBackgroundFillIndex]=new GtkFill(Display::tabBackgroundFillIndex);
        fill[Display::tabRiderBackgroundFillIndex]=new GtkFill(Display::tabRiderBackgroundFillIndex);
        fill[Display::columnBackgroundFillIndex]=new GtkFill(Display::columnBackgroundFillIndex);
        fill[Display::columnEndBackgroundFillIndex]=fill[Display::backgroundFillIndex];
#if defined(HAVE_LIB_HILDON)
        fill[Display::menuStripBackgroundFillIndex]=new GtkFill(Display::backgroundFillIndex);
  #if HILDON_VERSION == 0
        fill[Display::menuPulldownBackgroundFillIndex]=new GtkFill(Display::backgroundFillIndex);
  #else
        fill[Display::menuPulldownBackgroundFillIndex]=new GtkFill(Display::menuEntryBackgroundFillIndex);
  #endif
#else
        fill[Display::menuStripBackgroundFillIndex]=new GtkFill(Display::menuStripBackgroundFillIndex);
        fill[Display::menuPulldownBackgroundFillIndex]=new GtkFill(Display::menuPulldownBackgroundFillIndex);
#endif
        fill[Display::menuEntryBackgroundFillIndex]=new GtkFill(Display::menuEntryBackgroundFillIndex);
        fill[Display::graphBackgroundFillIndex]=new GtkFill(Display::graphBackgroundFillIndex);// TODO
        fill[Display::toolbarBackgroundFillIndex]=new GtkFill(Display::toolbarBackgroundFillIndex);// TODO

        frame[Display::buttonFrameIndex]=new GtkFrame(Display::buttonFrameIndex);
        frame[Display::positiveButtonFrameIndex]=new GtkFrame(Display::positiveButtonFrameIndex);
        frame[Display::negativeButtonFrameIndex]=new GtkFrame(Display::negativeButtonFrameIndex);
        frame[Display::defaultButtonFrameIndex]=new GtkFrame(Display::defaultButtonFrameIndex);
        frame[Display::toolbarButtonFrameIndex]=new GtkFrame(Display::toolbarButtonFrameIndex);
        frame[Display::scrollButtonFrameIndex]=new GtkFrame(Display::scrollButtonFrameIndex);
        frame[Display::editFrameIndex]=new GtkFrame(Display::editFrameIndex);
        frame[Display::hprogressFrameIndex]=new GtkFrame(Display::hprogressFrameIndex);
        frame[Display::vprogressFrameIndex]=new GtkFrame(Display::vprogressFrameIndex);
        frame[Display::hknobBoxFrameIndex]=new GtkFrame(Display::hknobBoxFrameIndex);
        frame[Display::vknobBoxFrameIndex]=new GtkFrame(Display::vknobBoxFrameIndex);
        frame[Display::hscrollFrameIndex]=new GtkFrame(Display::hscrollFrameIndex);
        frame[Display::vscrollFrameIndex]=new GtkFrame(Display::vscrollFrameIndex);
        frame[Display::tabFrameIndex]=new GtkFrame(Display::tabFrameIndex);
        frame[Display::tabRiderFrameIndex]=new GtkFrame(Display::tabRiderFrameIndex);
        frame[Display::listboxFrameIndex]=new GtkFrame(Display::listboxFrameIndex);
        frame[Display::scrolledFrameIndex]=emptyFrame;
        frame[Display::columnFrameIndex]=new GtkFrame(Display::columnFrameIndex);
        frame[Display::columnLeftFrameIndex]=new GtkFrame(Display::columnLeftFrameIndex);
        frame[Display::columnMiddleFrameIndex]=new GtkFrame(Display::columnMiddleFrameIndex);
        frame[Display::columnRightFrameIndex]=new GtkFrame(Display::columnRightFrameIndex);
        frame[Display::columnEndFrameIndex]=emptyFrame; // TODO
        frame[Display::comboFrameIndex]=frame[Display::buttonFrameIndex];
        frame[Display::editComboFrameIndex]=emptyFrame;
        frame[Display::mainWindowFrameIndex]=new GtkFrame(Display::mainWindowFrameIndex);
        frame[Display::menuWindowFrameIndex]=new GtkFrame(Display::menuWindowFrameIndex);
        frame[Display::popupWindowFrameIndex]=new GtkFrame(Display::popupWindowFrameIndex);
        frame[Display::tooltipWindowFrameIndex]=frame[Display::popupWindowFrameIndex];
        frame[Display::hscaleFrameIndex]=new GtkFrame(Display::hscaleFrameIndex);
        frame[Display::vscaleFrameIndex]=new GtkFrame(Display::vscaleFrameIndex);
        frame[Display::focusFrameIndex]=new GtkFrame(Display::focusFrameIndex);
        frame[Display::statuscellFrameIndex]=new GtkFrame(Display::statuscellFrameIndex);
        frame[Display::statusbarFrameIndex]=emptyFrame;
        frame[Display::plateFrameIndex]=new GtkFrame(Display::plateFrameIndex);
        frame[Display::valuebarFrameIndex]=new GtkFrame(Display::valuebarFrameIndex);
        frame[Display::groupFrameIndex]=new GtkFrame(Display::groupFrameIndex);
        frame[Display::menuStripFrameIndex]=emptyFrame;
        frame[Display::menuPulldownFrameIndex]=emptyFrame;

        image[Display::checkImageIndex]=new GtkImage(Display::checkImageIndex);
        image[Display::radioImageIndex]=new GtkImage(Display::radioImageIndex);
        image[Display::hknobImageIndex]=new GtkImage(Display::hknobImageIndex);
        image[Display::vknobImageIndex]=new GtkImage(Display::vknobImageIndex);
        image[Display::arrowLeftImageIndex]=new GtkImage(Display::arrowLeftImageIndex);
        image[Display::arrowRightImageIndex]=new GtkImage(Display::arrowRightImageIndex);
        image[Display::arrowUpImageIndex]=new GtkImage(Display::arrowUpImageIndex);
        image[Display::arrowDownImageIndex]=new GtkImage(Display::arrowDownImageIndex);
        image[Display::arrowScrollLeftImageIndex]=new GtkImage(Display::arrowScrollLeftImageIndex);
        image[Display::arrowScrollRightImageIndex]=new GtkImage(Display::arrowScrollRightImageIndex);
        image[Display::arrowScrollUpImageIndex]=new GtkImage(Display::arrowScrollUpImageIndex);
        image[Display::arrowScrollDownImageIndex]=new GtkImage(Display::arrowScrollDownImageIndex);
        image[Display::comboImageIndex]=new GtkImage(Display::comboImageIndex);
        image[Display::comboEditButtonImageIndex]=new GtkImage(Display::comboEditButtonImageIndex);
        image[Display::comboDividerImageIndex]=new GtkImage(Display::comboDividerImageIndex);
        image[Display::treeExpanderImageIndex]=new GtkImage(Display::treeExpanderImageIndex);
        image[Display::hscaleKnobImageIndex]=new GtkImage(Display::hscaleKnobImageIndex);
        image[Display::vscaleKnobImageIndex]=new GtkImage(Display::vscaleKnobImageIndex);
        image[Display::leftSliderImageIndex]=new GtkImage(Display::leftSliderImageIndex);
        image[Display::rightSliderImageIndex]=new GtkImage(Display::rightSliderImageIndex);
        image[Display::topSliderImageIndex]=new GtkImage(Display::topSliderImageIndex);
        image[Display::bottomSliderImageIndex]=new GtkImage(Display::bottomSliderImageIndex);
        image[Display::menuDividerImageIndex]=new GtkImage(Display::menuDividerImageIndex);
        image[Display::menuSubImageIndex]=new GtkImage(Display::menuSubImageIndex);
        image[Display::positiveImageIndex]=new GtkImage(Display::positiveImageIndex);
        image[Display::negativeImageIndex]=new GtkImage(Display::negativeImageIndex);
        image[Display::defaultImageIndex]=new GtkImage(Display::defaultImageIndex);
        image[Display::helpImageIndex]=new GtkImage(Display::helpImageIndex);
        image[Display::closeImageIndex]=new GtkImage(Display::closeImageIndex);
        image[Display::ledImageIndex]=new GtkImage(Display::ledImageIndex);

        scrollKnobMode=(GetWidgetStyleBoolValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],"has-secondary-backward-stepper") &&
                       GetWidgetStyleBoolValue(gtkWidgets->widget[GtkWidgets::gtkStyleHKnob],"has-secondary-forward-stepper")) ?
                       scrollKnobModeDouble : scrollKnobModeSingle;
      }

      GtkTheme::~GtkTheme()
      {
        delete gtkWidgets;

        display->FreeColor(helpBackgroundColor);

        for (size_t i=0; i<Display::colorCount; i++) {
          display->FreeColor(color[i]);
        }
      }

      bool GtkTheme::HasFontSettings() const
      {
        return false;
      }

#if defined(HAVE_LIB_GCONF)
      static std::wstring ExtractFontNameFromGConf(const char* path)
      {
        GConfClient *client;
        GError      *error=NULL;
        const char  *tmp;

        client=gconf_client_get_default();
        tmp=gconf_client_get_string(client,path,&error);

        if (tmp==NULL || tmp[0]=='\0') {
          return L"";
        }

        std::wstring name(Lum::Base::StringToWString(tmp));
        size_t       x=name.length()-1;

        while (x>0 && name[x]>=L'0' && name[x]<=L'9') {
          x--;
        }
        while (x>0 && name[x]==L' ') {
          x--;
        }

        return name.substr(0,x+1);
      }

      static size_t ExtractFontSizeFromGConf(const char* path)
      {
        GConfClient *client;
        GError      *error=NULL;
        const char  *tmp;

        client=gconf_client_get_default();
        tmp=gconf_client_get_string(client,path,&error);

        if (tmp==NULL || tmp[0]=='\0') {
          return 0;
        }

        std::wstring size(Lum::Base::StringToWString(tmp));
        size_t       x=size.length()-1;
        size_t       value;

        while (x>0 && size[x]>=L'0' && size[x]<=L'9') {
          x--;
        }

        assert(x<size.length());

        size=size.substr(x+1);

        if (Lum::Base::WStringToNumber(size,value)) {
          return value;
        }

        assert(false);
      }
#endif

      std::wstring GtkTheme::GetProportionalFontName() const
      {
        std::wstring result;

#if defined(HAVE_LIB_GCONF)
        result=ExtractFontNameFromGConf("/desktop/gnome/interface/font_name");
#endif

        if (result.empty()) {
          result=L"Helvetica";
        }

        return result;
      }

      size_t GtkTheme::GetProportionalFontSize() const
      {
        size_t result;

#if defined(HAVE_LIB_GCONF)
        result=ExtractFontSizeFromGConf("/desktop/gnome/interface/font_name");
#endif

        if (result==0) {
          result=9;
        }

        return round((result*100.0*100.0)/(72*display->GetDPI()));
      }

      std::wstring GtkTheme::GetFixedFontName() const
      {
        std::wstring result;

#if defined(HAVE_LIB_GCONF)
        result=ExtractFontNameFromGConf("/desktop/gnome/interface/monospace_font_name");
#endif

        if (result.empty()) {
          result=L"Fixed";
        }

        return result;
      }

      size_t GtkTheme::GetFixedFontSize() const
      {
        size_t result;

#if defined(HAVE_LIB_GCONF)
        result=ExtractFontSizeFromGConf("/desktop/gnome/interface/monospace_font_name");
#endif

        if (result==0) {
          result=9;
        }

        return round((result*100.0*100.0)/(72*display->GetDPI()));
      }

      OS::Theme::ScrollKnobMode GtkTheme::GetScrollKnobMode() const
      {
        return scrollKnobMode;
      }

      OS::Theme::ToolbarPosition GtkTheme::GetToolbarPosition() const
      {
#if defined(HAVE_LIB_HILDON)
       return OS::Theme::toolbarBottom;
#else
       return OS::Theme::toolbarTop;
#endif
      }

      size_t GtkTheme::GetMinimalButtonWidth() const
      {
        return Lum::Base::GetSize(Lum::Base::Size::stdCharWidth,8);
      }

      size_t GtkTheme::GetFirstTabOffset() const
      {
        return 0;
      }

      size_t GtkTheme::GetLastTabOffset() const
      {
        return OS::display->GetSpaceHorizontal(OS::Display::spaceInterObject);
      }

      size_t GtkTheme::GetEntryHorizTextOffset() const
      {
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
        return 2;
#else
        int left=2,right=2,top=2,bottom=2;

        GetWidgetStyleBorderValue(gtkWidgets->widget[GtkWidgets::gtkStyleEdit],"inner-border",
                                  left,right,top,bottom);

        return left;
#endif
     }

      size_t GtkTheme::GetEntryVertTextOffset() const
      {
#if defined(HAVE_LIB_HILDON) && HILDON_VERSION == 0
        return 2;
#else
        int left=2,right=2,top=2,bottom=2;

        GetWidgetStyleBorderValue(gtkWidgets->widget[GtkWidgets::gtkStyleEdit],"inner-border",
                                  left,right,top,bottom);

        return top;
#endif
      }
    }
  }
}
