/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Model/Dir.h>

#include <Lum/Private/Config.h>

#include <sys/stat.h>
#include <sys/types.h>

#if defined(__WIN32__) || defined(WIN32)
  #include <io.h>
  #include <direct.h>
  #include <Lum/OS/Win32/OSAPI.h>

#else
  #include <dirent.h>
  #include <unistd.h>
#endif

#include <algorithm>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

namespace Lum {
  namespace Model {

    bool lessEntry(const Dir::Entry& e1, const Dir::Entry& e2)
    {
      if (e1.type!=e2.type) {
        return e1.type==Dir::dirType;
      }
      else {
        return e1.name<e2.name;
      }
    }

    Dir::Filter::Filter(const std::wstring& name)
    : name(name)
    {
      // no code
    }

    Dir::Filter::~Filter()
    {
      // no code
    }

    std::wstring Dir::Filter::GetName() const
    {
      return name;
    }

    Dir::AllFilter::AllFilter(const std::wstring& name)
    : Filter(name)
    {
      // no code
    }

    bool Dir::AllFilter::Match(const std::wstring& directory, const Entry& entry) const
    {
      return true;
    }

    Dir::AllVisibleFilter::AllVisibleFilter(const std::wstring& name)
    : Filter(name)
    {
      // no code
    }

    bool Dir::AllVisibleFilter::Match(const std::wstring& directory, const Entry& entry) const
    {
      if (entry.name.length()>0) {
        return entry.name[0]!=L'.';
      }
      else {
        return true;
      }
    }

    Dir::PatternFilter::PatternFilter(const std::wstring& name)
    : Filter(name)
    {
      // no code
    }

    bool Dir::PatternFilter::Match(const std::wstring& directory, const Entry& entry) const
    {
      if (entry.type==dirType) {
        return true;
      }

      for (size_t x=0; x<patterns.size(); ++x) {
        if (Base::MatchWildcard(patterns[x],entry.name,false)) {
          return true;
        }
      }
      return false;
    }

    void Dir::PatternFilter::AddPattern(const std::wstring& pattern)
    {
      patterns.push_back(pattern);
    }

    Dir::Dir()
    : filter(NULL)
    {
      // no code
    }

    size_t Dir::GetRows() const
    {
      return content.size();
    }

    size_t Dir::GetColumns() const
    {
      return 3;
    }

    std::wstring Dir::GetString(size_t column, size_t row) const
    {
      switch (column) {
      case 1:
        return content[row-1].name;
      case 2:
        if (content[row-1].type==dirType) {
          return _l(L"MODEL_DIR_DIR",L"(dir)");
        }
        else {
          return Base::NumberToWString(content[row-1].size);
        }
      case 3:
        {
/*          std::wstring res(L"----------");
          if (content[row-1].user & rightExecute) {
            res[0]='x';
          }
          if (content[row-1].user & rightRead) {
            res[1]='r';
          }
          if (content[row-1].user & rightWrite) {
            res[2]='w';
          }

          if (content[row-1].group & rightExecute) {
            res[3]='x';
          }
          if (content[row-1].group & rightRead) {
            res[4]='r';
          }
          if (content[row-1].group & rightWrite) {
            res[5]='w';
          }

          if (content[row-1].other & rightExecute) {
            res[6]='x';
          }
          if (content[row-1].other & rightRead) {
            res[7]='r';
          }
          if (content[row-1].other & rightWrite) {
            res[8]='w';
          }

          return res;*/
          if (content[row-1].timeModification.IsToday()) {
            return content[row-1].timeModification.GetLocalLocaleTime();
          }
          else {
            return content[row-1].timeModification.GetLocalLocaleDate();
          }
        }
      default:
        return L"";
      }
    }

    Object* Dir::GetObject(size_t /*column*/, size_t /*row*/) const
    {
      return NULL;
    }

    void Dir::SetFilter(Filter* filter)
    {
      this->filter=filter;
    }

    void Dir::SetDirectory(const std::wstring& directory,
                           bool noDotFiles, bool onlyDirs)
    {
      std::string name;
      char        buffer[4096];

      name=Base::WStringToString(directory);

      content.clear();

      if (getcwd(buffer,sizeof(buffer))==NULL) {
        Notify();
        return;
      }

      if (chdir(name.c_str())!=0) {
        chdir(buffer);
        Notify();
        return;
      }

#if defined(HAVE_FINDFIRSTFILEW)
      WIN32_FIND_DATAW data;
      HANDLE           handle;
      std::wstring     tmp(directory);

      if (tmp.length()>0 && tmp[tmp.length()-1]!=L'\\'&& tmp[tmp.length()-1]!=L'/') {
        tmp.append(1,L'\\');
      }
      tmp.append(1,L'*');

      handle=FindFirstFileW(tmp.c_str(),&data);
      if (handle!=INVALID_HANDLE_VALUE) {
        bool finished=false;

        while (!finished) {

          if ((!noDotFiles || (noDotFiles && !(data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN))) &&
              (!onlyDirs || (onlyDirs && !(data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))) &&
              !(data.cFileName[0]==L'.' && data.cFileName[1]==L'\0') &&
              !(data.cFileName[0]==L'.' && data.cFileName[1]==L'.' && data.cFileName[2]==L'\0')) {

            Entry entry;

            entry.name=data.cFileName;
            entry.size=data.nFileSizeLow;

            if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
              entry.type=dirType;
            }
            else {
              entry.type=normalType;
            }

            entry.user=0;
            entry.user|=rightRead;
            if (!(data.dwFileAttributes & FILE_ATTRIBUTE_READONLY)) {
              entry.user|=rightWrite;
            }

            entry.group=0;
            entry.other=0;
            entry.link=false;

            entry.timeModification.SetTime(data.ftLastWriteTime);

            if (entry.type==dirType || filter==NULL || filter->Match(directory,entry)) {
              content.push_back(entry);
            }
          }

          if (!FindNextFileW(handle,&data)) {
            if (GetLastError()==ERROR_NO_MORE_FILES) {
              finished=true;
            }
          }
        }

        FindClose(handle);
      }

#else
      DIR           *dir;
      struct dirent *entr;
      struct stat   info;

      dir=opendir(name.c_str());
      if (dir!=NULL) {
        this->directory=directory;

        while ((entr=readdir(dir))!=NULL) {
          if (!((entr->d_name[0]=='.' && entr->d_name[1]=='\0') ||
                (entr->d_name[0]=='.' && entr->d_name[1]=='.' && entr->d_name[2]=='\0'))) {
            if (stat(entr->d_name,&info)==0) {

              if (noDotFiles && entr->d_name[0]=='.') {
                continue;
              }

              if (onlyDirs && !S_ISDIR(info.st_mode)) {
                continue;
              }

              Entry entry;

              if (S_ISDIR(info.st_mode)) {
                entry.type=dirType;
              }
              else {
                entry.type=normalType;
              }
              entry.name=Base::StringToWString(entr->d_name);
              entry.size=info.st_size;

              entry.user=0;
              if (info.st_mode & S_IRUSR) {
                entry.user|=rightRead;
              }
              if (info.st_mode & S_IWUSR) {
                entry.user|=rightWrite;
              }
              if (info.st_mode & S_IXUSR) {
                entry.user|=rightExecute;
              }

/*#if defined(__WIN32__) || defined(WIN32)
              entry.group=0;
              entry.other=0;
              entry.link=false;
#else*/
              entry.group=0;
              if (info.st_mode & S_IRGRP) {
                entry.group|=rightRead;
              }
              if (info.st_mode & S_IWGRP) {
                entry.group|=rightWrite;
              }
              if (info.st_mode & S_IXGRP) {
                entry.group|=rightExecute;
              }

              entry.other=0;
              if (info.st_mode & S_IROTH) {
                entry.other|=rightRead;
              }
              if (info.st_mode & S_IWOTH) {
                entry.other|=rightWrite;
              }
              if (info.st_mode & S_IXOTH) {
                entry.other|=rightExecute;
              }

              if (lstat(entr->d_name,&info)==0) {
                entry.link=S_ISLNK(info.st_mode);
              }
              else {
                entry.link=false;
              }
//#endif
              entry.timeModification.SetTime(info.st_mtime);

              if (filter==NULL || filter->Match(directory,entry)) {
                content.push_back(entry);
              }
            }
          }
        }

        closedir(dir);
      }
#endif

      chdir(buffer);

      std::sort(content.begin(),content.end(),lessEntry);

      Notify();
    }

    void Dir::GetEntry(size_t row, Entry& entry)
    {
      entry=content[row-1];
    }
  }
}
