/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/View.h>

#include <Lum/Scroller.h>

namespace Lum {

  static View::Prefs *prefs=new View::Prefs(L"View");

  View::Prefs::Prefs(const std::wstring& name)
    : Group::Prefs(name)
  {
    // no code
  }

  void View::Prefs::Initialize()
  {
    Group::Prefs::Initialize();

    frame=OS::display->GetFrame(OS::Display::scrolledFrameIndex);
  }

  View::View()
  : object(NULL),hScroller(NULL),vScroller(NULL),
    hVisible(false),vVisible(false)
  {
    SetPrefs(::Lum::prefs);
  }

  View::~View()
  {
    delete hScroller;
    delete vScroller;
    delete object;
  }

  bool View::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    if (hScroller!=NULL && (!onlyVisible || hVisible)) {
      if (!visitor.Visit(hScroller)) {
        return false;
      }
    }

    if (vScroller!=NULL && (!onlyVisible || vVisible)) {
      if (!visitor.Visit(vScroller)) {
        return false;
      }
    }

    if (object!=NULL) {  // if grid was not yet created (before calcSize)
      if (!visitor.Visit(object)) {
        return false;
      }
    }

    return true;
  }

  Scrollable* View::GetObject() const
  {
    return object;
  }

  void View::SetObject(Scrollable* object)
  {
    assert(object!=NULL);

    if (this->object!=NULL) {
      UnattachModel(object->GetHAdjustment());
      UnattachModel(object->GetVAdjustment());
      delete this->object;
    }

    this->object=object;
    object->SetParent(this);
    object->SetFlex(true,true);

    AttachModel(object->GetHAdjustment());
    AttachModel(object->GetVAdjustment());
  }

  bool View::SetModel(Base::Model* model)
  {
    return object->SetModel(model);
  }

  void View::CalcSize()
  {
    assert(object!=NULL);

    vScroller=new Scroller();
    vScroller->SetParent(this);
    vScroller->Set(true);
    vScroller->SetFlex(false,true);
    vScroller->SetModel(object->GetVAdjustment());

    hScroller=new Scroller();
    hScroller->SetParent(this);
    hScroller->Set(false);
    hScroller->SetFlex(true,false);
    hScroller->SetModel(object->GetHAdjustment());

    if (RequestedFocus()) {
      /* Delegate focusing to the real table */
      object->RequestFocus();
      UnrequestFocus();
    }

    object->CalcSize();

    minWidth=object->GetOMinWidth();
    width=object->GetOWidth();
    minHeight=object->GetOMinHeight();
    height=object->GetOHeight();

    hScroller->CalcSize();

    minWidth=std::max(hScroller->GetOWidth(),minWidth);
    width=std::max(hScroller->GetOWidth(),width);
    minHeight+=hScroller->GetOHeight();
    height+=hScroller->GetOHeight();

    vScroller->CalcSize();

    minWidth+=vScroller->GetOMinWidth();
    width+=vScroller->GetOWidth();
    minHeight=std::max(vScroller->GetOHeight(),minHeight);
    height=std::max(vScroller->GetOHeight(),height);

    Group::CalcSize();
  }

  void View::Layout()
  {
    object->CalculateDimension();

    size_t tWidth,tHeight;

    tWidth=width;
    tHeight=height;

    // Calculate object dimensions

    if (hVisible) {
      tHeight-=hScroller->GetOHeight();
    }

    if (vVisible) {
      tWidth-=vScroller->GetOWidth();
    }

    // Resize and position objects

    object->MoveResize(x,y,tWidth,tHeight);

    if (hVisible) {
      hScroller->MoveResize(x,y+tHeight,tWidth,hScroller->GetOHeight());
    }

    if (vVisible) {
      vScroller->MoveResize(x+tWidth,y,vScroller->GetOWidth(),tHeight);
    }

    Group::Layout();
  }

  void View::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (object!=NULL) {
      if (model==object->GetHAdjustment()) {
        if (hVisible && object->GetHAdjustment()->GetVisible()==object->GetHAdjustment()->GetTotal()) {
          hScroller->Hide();
          hVisible=false;
          SetRelayout();
          //Redraw();
          Draw(oX,oY,oWidth,oHeight);
        }
        else if (!hVisible && object->GetHAdjustment()->GetVisible()<object->GetHAdjustment()->GetTotal()) {
          hVisible=true;
          SetRelayout();
          //Redraw();
          Draw(oX,oY,oWidth,oHeight);
        }
        return;
      }
      else if (model==object->GetVAdjustment()) {
        if (vVisible && object->GetVAdjustment()->GetVisible()==object->GetVAdjustment()->GetTotal()) {
          vScroller->Hide();
          vVisible=false;
          SetRelayout();
          //Redraw();
          Draw(oX,oY,oWidth,oHeight);
        }
        else if (!vVisible && object->GetVAdjustment()->GetVisible()<object->GetVAdjustment()->GetTotal()) {
          vVisible=true;
          SetRelayout();
          //Redraw();
          Draw(oX,oY,oWidth,oHeight);
        }
        return;
      }
    }

    Group::Resync(model,msg);
  }
}

