/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Table.h>

namespace Lum {

  static Table::Prefs *prefs=new Table::Prefs();

  Table::Prefs::Prefs()
  : Group::Prefs(L"Table")
  {
    // no code
  }

  void Table::Prefs::Initialize()
  {
    Group::Prefs::Initialize();

    frame=OS::display->GetFrame(OS::Display::scrolledFrameIndex);
  }

  Table::Table()
  : object(NULL),hScroller(NULL),vScroller(NULL),
    hVisible(false),vVisible(false),
    table(new TableView()),
    header(NULL),headerModel(new Model::HeaderImpl()),showHeader(false)
  {
    SetPrefs(::Lum::prefs);

    SetCanFocus(true);

    headerModel->AddColumn(L"",Base::Size::modePixel,32000,true);
  }

  Table::~Table()
  {
    UnattachModel(table->GetHAdjustment());
    UnattachModel(table->GetVAdjustment());

    delete object;
    delete hScroller;
    delete vScroller;
  }

  bool Table::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    if (object!=NULL) {
      if (!visitor.Visit(object)) {
        return false;
      }
    }

    if (hScroller!=NULL && (!onlyVisible || hVisible)) {
      if (!visitor.Visit(hScroller)) {
        return false;
      }
    }

    if (vScroller!=NULL && (!onlyVisible || vVisible)) {
      if (!visitor.Visit(vScroller)) {
        return false;
      }
    }

    return true;
  }

  TableView* Table::GetTableView() const
  {
    return table;
  }

  /**
    Assign a custom instance for the internal table object.

    You must call SetModel() after you have called SetTableView() otherwise
    model assignment will not work.
  */
  void Table::SetTableView(TableView* tableView)
  {
    assert(tableView!=NULL);

    if (table!=NULL) {
      delete table;
    }

    table=tableView;
  }

  Model::Header* Table::GetHeaderModel() const
  {
    return headerModel.Get();
  }

  void Table::SetHeaderModel(Model::Header* headerModel)
  {
    assert(!IsVisible());

    this->headerModel=headerModel;
  }

  void Table::SetShowHeader(bool show)
  {
    showHeader=show;
  }

  bool Table::SetModel(Base::Model* model)
  {
    return table->SetModel(model);
  }

  void Table::SetSelection(Model::Selection* selection)
  {
    table->SetSelection(selection);
  }

  void Table::SetSelectionAction(Model::Action* action)
  {
    table->SetSelectionAction(action);
  }

  void Table::SetDoubleClickAction(Model::Action* action)
  {
    table->SetDoubleClickAction(action);
  }

  void Table::SetFormatProvider(TableView::FormatProvider* formatProvider)
  {
    table->SetFormatProvider(formatProvider);
  }

  void Table::CalcSize()
  {
    if (header==NULL) {
      header=new Header();
      header->SetFlex(true,false);
      header->SetModel(headerModel);
      header->SetAdjustment(table->GetHAdjustment());
    }

    if (table==NULL) {
      table=new TableView();
    }

    table->SetFlex(true,true);
    table->SetHeader(headerModel);

    if (object==NULL) {
      object=new VPanel();
      object->SetParent(this);
      object->SetFlex(true,true);
      object->SetFrame(OS::display->GetFrame(OS::Display::listboxFrameIndex));

      if (showHeader && header!=NULL) {
        dynamic_cast<VPanel*>(object)->Add(header);
      }

      dynamic_cast<VPanel*>(object)->Add(table);
    }

    AttachModel(table->GetHAdjustment());
    AttachModel(table->GetVAdjustment());

    vScroller=new Scroller();
    vScroller->SetParent(this);
    vScroller->Set(true);
    vScroller->SetFlex(false,true);
    vScroller->SetModel(table->GetVAdjustment());
    vScroller->CalcSize();

    hScroller=new Scroller();
    hScroller->SetParent(this);
    hScroller->Set(false);
    hScroller->SetFlex(true,false);
    hScroller->SetModel(table->GetHAdjustment());
    hScroller->CalcSize();

    if (RequestedFocus()) {
      /* Delegate focusing to the real table */
      table->RequestFocus();
      UnrequestFocus();
    }

    object->CalcSize();

    // Initial dimension equal to table dimension
    minWidth=object->GetOMinWidth();
    width=object->GetOWidth();
    minHeight=object->GetOMinHeight();
    height=object->GetOHeight();

    // Add Horizontal scroller
    minWidth=std::max(hScroller->GetOWidth(),minWidth);
    width=std::max(hScroller->GetOWidth(),width);
    minHeight+=hScroller->GetOHeight();

    // Add Vertical scroller
    minWidth+=vScroller->GetOMinWidth();
    width+=vScroller->GetOWidth();
    minHeight=std::max(vScroller->GetOHeight(),minHeight);
    height=std::max(vScroller->GetOHeight(),height);

    Group::CalcSize();
  }

  void Table::Layout()
  {
    table->CalculateDimension();

    size_t tWidth,tHeight;

    tWidth=width;
    tHeight=height;

    // Calculate object dimensions

    if (hVisible) {
      tHeight-=hScroller->GetOHeight();
    }

    if (vVisible) {
      tWidth-=vScroller->GetOWidth();
    }

    // Resize and position objects

    object->MoveResize(x,y,tWidth,tHeight);

    if (hVisible) {
      hScroller->MoveResize(x,y+tHeight,tWidth,hScroller->GetOHeight());
    }

    if (vVisible) {
      vScroller->MoveResize(x+tWidth,y,vScroller->GetOWidth(),tHeight);
    }

    Group::Layout();
  }

  size_t Table::GetHorizontalScrollerHeight() const
  {
    return hScroller->GetOHeight();
  }

  size_t Table::GetVerticalScrollerWidth() const
  {
    return vScroller->GetOWidth();
  }

  void Table::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (table!=NULL) {
      if (model==table->GetHAdjustment() && table->GetHAdjustment()->IsValid()) {
        if (hVisible && table->GetHAdjustment()->GetVisible()==table->GetHAdjustment()->GetTotal()) {
          hScroller->Hide();
          hVisible=false;
          SetRelayout();
          if (visible) {
            Draw(oX,oY,oWidth,oHeight);
          }
        }
        else if (!hVisible && table->GetHAdjustment()->GetVisible()<table->GetHAdjustment()->GetTotal()) {
          hVisible=true;
          SetRelayout();
          if (visible) {
            Draw(oX,oY,oWidth,oHeight);
          }
        }
        return;
      }
      else if (model==table->GetVAdjustment()  && table->GetVAdjustment()->IsValid()) {
        if (vVisible && table->GetVAdjustment()->GetVisible()==table->GetVAdjustment()->GetTotal()) {
          vScroller->Hide();
          vVisible=false;
          SetRelayout();
          if (visible) {
            Draw(oX,oY,oWidth,oHeight);
          }
        }
        else if (!vVisible && table->GetVAdjustment()->GetVisible()<table->GetVAdjustment()->GetTotal()) {
          vVisible=true;
          SetRelayout();
          if (visible) {
            Draw(oX,oY,oWidth,oHeight);
          }
        }
        return;
      }
    }

    Group::Resync(model,msg);
  }
}
