#include <Lum/Prefs/Base.h>

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <iostream>
#include <list>

#include <Lum/Base/String.h>

namespace Lum {
  namespace Prefs {

    static std::list<Prefs*> *list=NULL;
    static Config::Node      *config=NULL;
    static Config::Node      *theme=NULL;

    Prefs::Prefs(const std::wstring& name)
    : name(name)
    {
      if (list==NULL) {
        list=new std::list<Prefs*>;
      }

      list->push_back(this);
    }

    Prefs::~Prefs()
    {
      // no code
    }

    /**
      The return the (class/object) name of the Prefs object.

      This name is also used to identify object in the configuration and
      theme files.
    */
    std::wstring Prefs::GetName() const
    {
      return name;
    }

    /**
      Method that will be called when the Prefs object should read itself
      from configration file.

      'top' points to the top it the objects section in the config file.

      The config file should hold theme independend information.
    */
    void Prefs::ReadConfig(::Lum::Config::Node* /*top*/)
    {
      // no code
    }

    /**
      Method that will be called when the Prefs object should read itself
      from theme file.

      'top' points to the top it the objects section in the theme file.

      The theme file should hold theming information.
    */
    void Prefs::ReadTheme(::Lum::Config::Node* /*top*/)
    {
      // no code
    }

    /**
      A callback method in which the Prefs object should initialize itself to
      its default values. Initialisation is delayed from the constructor to
      this method to asure that the low level display object is initialized.
    */
    void InitializePrefs(Prefs* dont)
    {
      std::list<Prefs*>::iterator iter;

      iter=list->begin();
      while (iter!=list->end()) {

        if (*iter!=dont) {
          (*iter)->Initialize();
        }

        ++iter;
      }
    }

    bool LoadConfig(const std::wstring& name)
    {
      if (!::Lum::Config::ConfigAccessible(name)) {
        return false;
      }

     Config::ErrorList errorList;

      config=::Lum::Config::LoadConfigFromXMLFile(name,errorList);

      for (::Lum::Config::ErrorList::iterator iter=errorList.begin(); iter!=errorList.end(); ++iter) {
        std::cerr <<Base::WStringToString((*iter).GetError()) << std::endl;
      }

      return config!=NULL;
    }

    bool LoadTheme(const std::wstring& name)
    {
      if (!::Lum::Config::ConfigAccessible(name)) {
        return false;
      }

     Config::ErrorList errorList;

      theme=::Lum::Config::LoadConfigFromXMLFile(name,errorList);

      for (::Lum::Config::ErrorList::iterator iter=errorList.begin(); iter!=errorList.end(); ++iter) {
        std::cerr <<Base::WStringToString((*iter).GetError()) << std::endl;
      }

      return theme!=NULL;
    }

    void ReadDisplayConfig()
    {
      std::list<Prefs*>::iterator iter;
      Config::Node                *top;

      if (config==NULL) {
        return;
      }

      top=config->GetChild(L"Display");
      if (top==NULL) {
        return;
      }

      iter=list->begin();
      while (iter!=list->end()) {
        if ((*iter)->GetName()==L"Display") {
          (*iter)->ReadConfig(top);
          return;
        }

        ++iter;
      }
    }
    void ReadOthersConfig()
    {
      std::list<Prefs*>::iterator iter;

      if (config==NULL) {
        return;
      }

      iter=list->begin();
      while (iter!=list->end()) {
        if ((*iter)->GetName()!=L"Display") {
         Config::Node *top=config->GetChild((*iter)->GetName());
          if (top!=NULL) {
            (*iter)->ReadConfig(top);
          }
        }

        ++iter;
      }
    }

    void ReadDisplayTheme()
    {
      std::list<Prefs*>::iterator iter;
      Config::Node                *top;

      if (theme==NULL) {
        return;
      }

      top=theme->GetChild(L"Display");
      if (top==NULL) {
        return;
      }

      iter=list->begin();
      while (iter!=list->end()) {
        if ((*iter)->GetName()==L"Display") {
          (*iter)->ReadTheme(top);
          return;
        }

        ++iter;
      }
    }
    void ReadOthersTheme()
    {
      std::list<Prefs*>::iterator iter;

      if (theme==NULL) {
        return;
      }

      iter=list->begin();
      while (iter!=list->end()) {
        if ((*iter)->GetName()!=L"Display") {
         Config::Node *top=theme->GetChild((*iter)->GetName());
          if (top!=NULL) {
            (*iter)->ReadTheme(top);
          }
        }

        ++iter;
      }
    }

    void FreeData()
    {
      delete config;
      delete theme;
    }

    void FreePrefs()
    {
      std::list<Prefs*>::iterator iter;

      iter=list->begin();
      while (iter!=list->end()) {
       delete *iter;

        ++iter;
      }

      delete list;
    }
  }
}
