/*
  This source is part of the Illumination library
  Copyright (C) 2005  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/OS/Base/Display.h>

#include <iostream>

#include <Lum/Private/Config.h>
#include <wctype.h>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>
#include <Lum/Base/Util.h>

namespace Lum {
  namespace OS {
    namespace Base {

      static bool IsLowerByteSet(unsigned char *bytes)
      {
        return bytes[0]!=0;
      }

      Display::Display()
      : scrWidth(0),scrHeight(0),workAreaWidth(0),workAreaHeight(0),
        multiClickTime(200),
        appName(L""),prefs(NULL),
        theme(NULL),
        keyboardHandler(NULL)
      {
        int i=1;

        if (IsLowerByteSet((unsigned char*)&i)) {
          systemByteOrder=littleEndian;
        }
        else {
          systemByteOrder=bigEndian;
        }
      }

      void Display::EvaluateDisplaySize()
      {
        switch (type) {
        case typeGraphical:
          if (scrWidth<=640) {
            size=sizeTiny;
          }
          else if (scrWidth<1024) {
            size=sizeSmall;
          }
          else if (scrWidth<1280) {
            size=sizeNormal;
          }
          else {
            size=sizeHuge;
          }
          break;
        case typeTextual:
          size=sizeTiny;
          break;
        }
      }

      void Display::Close()
      {
        for (std::map<size_t,FontRef>::iterator iter=propFonts.begin(); iter!=propFonts.end(); ++iter) {
          iter->second=NULL;
        }

        for (std::map<size_t,FontRef>::iterator iter=fixedFonts.begin(); iter!=fixedFonts.end(); ++iter) {
          iter->second=NULL;
        }

        propFont=NULL;
        fixedFont=NULL;

        delete keyboardHandler;

        delete theme;
      }

      std::wstring Display::GetAppName() const
      {
        return appName;
      }

      void Display::SetAppName(const std::wstring& name)
      {
        appName=name;
      }

      std::wstring Display::GetProgramName() const
      {
        return programName;
      }

      void Display::SetProgramName(const std::wstring& name)
      {
        programName=name;
      }

      void Display::SetArgs(const std::vector<std::wstring>& args)
      {
        this->args=args;
      }

      size_t Display::GetArgCount() const
      {
        return args.size();
      }

      std::wstring Display::GetArg(size_t pos) const
      {
        assert(pos<args.size());

        return args[pos];
      }

      std::wstring Display::GetPrefsPath() const
      {
        ::Lum::Base::Path path;

        path.SetNativeDir(::Lum::Base::Path::GetHome());
        path.AppendDir(L".illumination");

        return path.GetPath();
      }

      std::wstring Display::GetPrefsName() const
      {
        ::Lum::Base::Path path;

        path.SetNativeDir(GetPrefsPath());
        path.SetBaseName(driver->GetDriverName()+L".xml");

        return path.GetPath();
      }

      std::wstring Display::GetThemePath() const
      {
        ::Lum::Base::Path path;

        path.SetNativeDir(GetPrefsPath());
        path.AppendDir(L"themes");
        path.AppendDir(prefs->theme);

        return path.GetPath();
      }

      std::wstring Display::GetThemeName() const
      {
        ::Lum::Base::Path path;

        path.SetNativeDir(GetThemePath());
        path.SetBaseName(L"theme.xml");

        return path.GetPath();
      }

      Theme* Display::GetTheme() const
      {
        return theme;
      }

      ::Lum::OS::Display::Type Display::GetType() const
      {
        return type;
      }

      ::Lum::OS::Display::Size Display::GetSize() const
      {
        return size;
      }

      ::Lum::OS::Display::ColorMode Display::GetColorMode() const
      {
        return colorMode;
      }

      size_t Display::GetColorDepth() const
      {
        return colorDepth;
      }

      size_t Display::GetScreenWidth() const
      {
        return scrWidth;

      }
      size_t Display::GetScreenHeight() const
      {
        return scrHeight;
      }

      size_t Display::GetWorkAreaWidth() const
      {
        if (workAreaWidth==0) {
          return GetScreenWidth();
        }
        else {
          return workAreaWidth;
        }
      }
      size_t Display::GetWorkAreaHeight() const
      {
        if (workAreaHeight==0) {
          return GetScreenHeight();
        }
        else {
          return workAreaHeight;
        }
      }

      unsigned long Display::GetMultiClickTime() const
      {
        return multiClickTime;
      }

      void Display::SetColorDepth(size_t depth)
      {
        colorDepth=depth;
      }

      void Display::SetMultiClickTime(unsigned long time)
      {
        multiClickTime=time;
      }

      Color Display::GetColor(ColorIndex color) const
      {
        return this->color[color];
      }

      Fill* Display::GetFill(FillIndex fill) const
      {
        return theme->GetFill(fill);
      }

      Frame* Display::GetFrame(FrameIndex frame) const
      {
        return theme->GetFrame(frame);
      }

      Image* Display::GetImage(ImageIndex image) const
      {
        return theme->GetImage(image);
      }

      Color Display::GetColorByName(const std::string& name)
      {
        int index;

        index=GetColorIndexByName(name);

        assert(index>=0);

        return GetColor((ColorIndex)index);
      }

      void Display::SignalEventLoop()
      {
        std::cerr << "WARNING: Display::SignalEventLoop() is not implemented by this driver!" << std::endl;
      }

      void Display::QueueActionForAsyncNotification(Model::Action *action)
      {
        Guard<TryMutex> guard(asyncMutex);

        asyncActions.push_back(action);

        SignalEventLoop();
      }

      void Display::HandleAsyncActions()
      {
        if (asyncMutex.TryLock()) {
          while (asyncActions.begin()!=asyncActions.end()) {
            Model::ActionRef action(*asyncActions.begin());

            asyncActions.erase(asyncActions.begin());
            action->Trigger();
          }
          asyncMutex.Unlock();
        }
      }

      void Display::QueueActionForEventLoop(Model::Action *action)
      {
        eventLoopActions.push_back(action);
      }

      void Display::HandleEventLoopActions()
      {
        while (eventLoopActions.begin()!=eventLoopActions.end()) {
          Model::ActionRef action(*eventLoopActions.begin());

          eventLoopActions.erase(eventLoopActions.begin());
          action->Trigger();
        }
      }

      void Display::AddTimer(long seconds, long microseconds,
                             Model::Action* action)
      {
        assert(action!=NULL);

        ::Lum::Base::SystemTime time(seconds,microseconds);
        ::Lum::Base::SystemTime now;
        Timer                   timer;

        time.Add(now);

        timer.time=time;
        timer.action=action;

        timerList.push_back(timer);
      }

      void Display::RemoveTimer(Model::Action* action)
      {
        std::list<Timer>::iterator iter;

        iter=timerList.begin();
        while (iter!=timerList.end()) {
          if (iter->action.Get()==action) {
            iter=timerList.erase(iter);
          }
          else {
            ++iter;
          }
        }
      }

      void Display::CheckTimers()
      {
        std::list<Timer>::iterator iter;
        ::Lum::Base::SystemTime    now;

        iter=timerList.begin();
        while (iter!=timerList.end()) {
          if (iter->time<=now) {
            Model::ActionRef action;

            action=iter->action;

            timerList.erase(iter);

            action->Trigger();

            // A Timer might trigger a new dialog. This dialog may have its own event loop
            // which might call CheckTimers again, resulting in recursive calls to CheckTimers
            // changing timerList.
            // To make this work, we restart the loop over timerList every time we trigger
            // an action.
            iter=timerList.begin();
          }
          else {
            ++iter;
          }
        }
      }

      /**
        Return the next pending future time event. If there is no event pending it
        will return a event 10 seconds in the future.
      */
      bool Display::GetNextTimer(::Lum::Base::SystemTime& interval)
      {
        if (!timerList.empty()) {
          ::Lum::Base::SystemTime now;
          ::Lum::Base::SystemTime time;

          for (std::list<Timer>::iterator iter=timerList.begin(); iter!=timerList.end(); ++iter) {
            if (iter==timerList.begin()) {
              time=iter->time;
            }
            else if (iter->time<time) {
              time=iter->time;
            }
          }

          interval=time;
          interval.Sub(now);

          return true;
        }
        else {
          interval=::Lum::Base::SystemTime(1,0);

          return false;
        }
      }

      bool Display::SetClipboard(const std::wstring& /*content*/)
      {
        return false;
      }

      std::wstring Display::GetClipboard() const
      {
        return L"";
      }

      void Display::ClearClipboard()
      {
        // no code
      }

     void Display::KeyToKeyDescription(unsigned long qualifier,
                                       const std::wstring& key,
                                       std::wstring& description)
     {
       description=L"";
       if (qualifier!=0) {
         if (qualifier & (qualifierShiftLeft|qualifierShiftRight)) {
           description+=L"S";
         }
         if (qualifier & (qualifierControlLeft|qualifierControlRight)) {
           description+=L"C";
         }
         if (qualifier & (qualifierAltLeft|qualifierAltRight)) {
           description+=L"A";
         }
         if (qualifier & (qualifierMetaLeft|qualifierMetaRight)) {
           description+=L"M";
         }
         if (qualifier & (qualifierSuperLeft|qualifierSuperRight)) {
           description+=L"X";
         }
         if (qualifier & (qualifierHyperLeft|qualifierHyperRight)) {
           description+=L"H";
         }
         if (qualifier & qualifierButton1) {
           description+=L"1";
         }
         if (qualifier & qualifierButton2) {
           description+=L"2";
         }
         if (qualifier & qualifierButton3) {
           description+=L"3";
         }
         if (qualifier & qualifierButton4) {
           description+=L"4";
         }
         if (qualifier & qualifierButton5) {
           description+=L"5";
         }
         description+=L"+";
       }
       description+=key;
     }

      void Display::KeyToDisplayKeyDescription(unsigned long qualifier,
                                               const std::wstring& key,
                                               std::wstring& description)
      {

        description=L"";
        if (qualifier!=0) {
          if (qualifier & qualifierShift) {
            if (description.length()>0) {
              description+=L"+";
            }
            description+=L"Shift";
          }
          if (qualifier & qualifierControl) {
            if (description.length()>0) {
              description+=L"+";
            }
            description+=L"Ctrl";
          }
          if (qualifier & qualifierAlt) {
           if (description.length()>0) {
             description+=L"+";
            }
            description+=L"Alt";
          }
          if (qualifier & qualifierMeta) {
            if (description.length()>0) {
              description+=L"+";
            }
            description+=L"Meta";
          }
          if (qualifier & qualifierSuper) {
            if (description.length()>0) {
             description+=L"+";
            }
            description+=L"Super";
          }
          if (qualifier & qualifierHyper) {
            if (description.length()>0) {
             description+=L"+";
            }
            description+=L"Hyper";
          }
         if (qualifier & qualifierButton1) {
           if (description.length()>0) {
             description+=L"+";
            }
           description+=L"MouseButton1";
         }
         if (qualifier & qualifierButton2) {
           if (description.length()>0) {
             description+=L"+";
            }
           description+=L"MouseButton2";
         }
         if (qualifier & qualifierButton3) {
           if (description.length()>0) {
             description+=L"+";
            }
           description+=L"MouseButton3";
         }
         if (qualifier & qualifierButton4) {
           if (description.length()>0) {
             description+=L"+";
            }
           description+=L"MouseButton4";
         }
         if (qualifier & qualifierButton5) {
           if (description.length()>0) {
             description+=L"+";
            }
           description+=L"MouseButton5";
         }
          description+=L"+";
        }

        if (key.length()==1) {
          // TODO: Better use local and std::toupper(char,locale) for this
          description.append(1,(wchar_t)towupper(key.at(0)));
        }
        else {
          description+=key;
        }
      }

      bool Display::KeyDescriptionToKey(const std::wstring& description,
                                        unsigned long& qualifier,
                                        std::wstring& key)
      {
        size_t pos;

        qualifier=0;
        key=L"";
        pos=0;
        while ((pos<description.length()) && (description[pos]!='+')) {
          pos++;
        }

        if (pos<description.length()) {
          for (size_t x=0; x<pos; x++) {
            switch (description[x]) {
            case 'S':
              qualifier|=qualifierShift;
              break;
            case 'C':
              qualifier|=qualifierControl;
              break;
            case 'A':
              qualifier|=qualifierAlt;
              break;
            case 'M':
              qualifier|=qualifierMeta;
              break;
            case 'X':
              qualifier|=qualifierSuper;
              break;
            case 'H':
              qualifier|=qualifierHyper;
              break;
            case '1':
              qualifier|=qualifierButton1;
              break;
            case '2':
              qualifier|=qualifierButton2;
              break;
            case '3':
              qualifier|=qualifierButton3;
              break;
            case '4':
              qualifier|=qualifierButton4;
              break;
            case '5':
              qualifier|=qualifierButton5;
              break;
            default:
              return false;
              break;
            }
          }

          ++pos;
        }
        else {
          pos=0;
        }

        if (pos>description.length()) {
          return false;
        }

        key=description.substr(pos);

        return true;
      }

      size_t Display::GetSpaceHorizontal(Space space) const
      {
        switch (type) {
          case typeTextual:
            switch (space) {
            case spaceWindowBorder:
              return 0;
            case spaceInterGroup:
              return 1;
            case spaceGroupIndent:
              return 1;
            case spaceInterObject:
              return 0;
            case spaceObjectBorder:
              return 0;
            case spaceIntraObject:
              return 1;
            case spaceLabelObject:
              return 1;
            case spaceObjectSensibility:
              return 0;
            case spaceObjectMouseMove:
              return 1;
            case spaceObjectDetail:
              return 1;
            }
          break;
          case  typeGraphical:
            switch (space) {
            case spaceWindowBorder:
              return 12;
            case spaceInterGroup:
              return 18;
            case spaceGroupIndent:
              return 12;
            case spaceInterObject:
              return 6;
            case spaceObjectBorder:
              return 6;
            case spaceIntraObject:
              return 6;
            case spaceLabelObject:
              return 12;
            case spaceObjectSensibility:
              return 6;
            case spaceObjectMouseMove:
              return 3;
            case spaceObjectDetail:
              return 16;
            }
          break;
        }

        return 1;
      }

      size_t Display::GetSpaceVertical(Space space) const
      {
        switch (type) {
          case typeTextual:
            switch (space) {
            case spaceWindowBorder:
              return 0;
            case spaceInterGroup:
              return 0;
            case spaceGroupIndent:
              return 0;
            case spaceInterObject:
              return 0;
            case spaceObjectBorder:
              return 0;
            case spaceIntraObject:
              return 0;
            case spaceLabelObject:
              return 0;
            case spaceObjectSensibility:
              return 0;
            case spaceObjectMouseMove:
              return 1;
            case spaceObjectDetail:
              return 0;
            }
          break;
          case  typeGraphical:
            switch (space) {
            case spaceWindowBorder:
              return 12;
            case spaceInterGroup:
              return 18;
            case spaceGroupIndent:
              return 12;
            case spaceInterObject:
              return 6;
            case spaceObjectBorder:
              return 3;
            case spaceIntraObject:
              return 6;
            case spaceLabelObject:
              return 6;
            case spaceObjectSensibility:
              return 6;
            case spaceObjectMouseMove:
              return 3;
            case spaceObjectDetail:
              return 16;
            }
          break;
        }

        return 1;
      }

      Display::TextDirection Display::GetDefaultTextDirection() const
      {
        return textDirectionLeftToRight;
      }

      FontRef Display::GetFontInternal(FontType /*type*/, size_t /*size*/) const
      {
        FontRef font;

        return font;
      }

      bool Display::GetFontFromCache(FontType type, size_t size, FontRef& font) const
      {
        if (type==fontTypeProportional) {
          std::map<size_t,FontRef>::const_iterator iter=propFonts.find(size);

          if (iter!=propFonts.end()) {
            font=iter->second;

            return true;
          }
        }
        else if (type==fontTypeFixed) {
          std::map<size_t,FontRef>::const_iterator iter=fixedFonts.find(size);

          if (iter!=fixedFonts.end()) {
            font=iter->second;

            return true;
          }
        }
        else {
          assert(false);
        }

        return false;
      }

      OS::Font* Display::GetFont(FontType type, size_t normal, size_t size,
                                 std::map<size_t,FontRef>& cache, const FontRef &standard) const
      {
        assert(standard.Valid());

        if (size>normal) {
          while (size>normal) {
            FontRef font;

            if (!GetFontFromCache(type,size,font)) {
              font=GetFontInternal(type,size);
              cache[size]=font;
            }

            if (font.Valid()) {
              return font.Get();
            }

            --size;
          }
        }
        else if (size<normal) {
          while (size<normal) {
            FontRef font;

            if (!GetFontFromCache(type,size,font)) {
              font=GetFontInternal(type,size);
              cache[size]=font;
            }

            if (font.Valid()) {
              return font.Get();
            }

            ++size;
          }
        }

        return standard.Get();
      }

      OS::Font* Display::GetFont(FontType type, size_t scale) const
      {
        switch (type) {
        case fontTypeProportional:
          return GetFont(type,propFontSize,propFontSize*scale/100,propFonts,propFont);
        case fontTypeFixed:
          return GetFont(type,fixedFontSize,fixedFontSize*scale/100,fixedFonts,fixedFont);
        }

        return NULL;
      }

      OS::Font* Display::GetFontByPixel(FontType type, size_t size) const
      {
        FontRef font;

        if (type==fontTypeProportional) {
          if (!GetFontFromCache(type,size,font)) {
            font=GetFontInternal(type,size);
            propFonts[size]=font;
          }
        }
        else if (type==fontTypeFixed) {
          if (!GetFontFromCache(type,size,font)) {
            font=GetFontInternal(type,size);
            fixedFonts[size]=font;
          }
        }
        else {
          assert(false);
        }

        if (font.Valid()) {
          return font.Get();
        }
        else {
          return NULL;
        }
      }

      OS::Font* Display::GetFontByMaxPixel(FontType type, size_t size) const
      {
        switch (type) {
        case fontTypeProportional:
          assert(size>=propFontSize);
        case fontTypeFixed:
          assert(size>=fixedFontSize);
        }

        switch (type) {
        case fontTypeProportional:
          return GetFont(type,propFontSize,size,propFonts,propFont);
        case fontTypeFixed:
          return GetFont(type,fixedFontSize,size,fixedFonts,fixedFont);
        }

        return NULL;
      }

      KeyboardHandler* Display::GetKeyboardHandler() const
      {
        return keyboardHandler;
      }

      OS::Display::ByteOrder Display::GetSystemByteOrder() const
      {
        return systemByteOrder;
      }

      Driver::Driver(const std::wstring name, int priority)
      : OS::Driver(name,priority)
      {
        // no code
      }

    }
  }
}

