/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>     // for isnan()

struct icon_data {
  GdkPixbuf **data;
  int count;
} icons[] = {
  { NULL, CACHE_TYPE_MAX+5 },             /* ICON_CACHE_TYPE      */
  { NULL, CACHE_TYPE_MAX+5 },             /* ICON_CACHE_TYPE_SEMI */
  { NULL, CACHE_TYPE_MAX+5 },             /* ICON_CACHE_TYPE_1_5X */
  { NULL, CACHE_TYPE_MAX+5 },             /* ICON_CACHE_TYPE_2X   */
  { NULL, CACHE_CONT_MAX+1 },             /* ICON_CACHE_SIZE      */
  { NULL, 9 },                            /* ICON_STARS           */
  { NULL, LOG_TYPE_MAX+1 },               /* ICON_LOG             */
  { NULL, 9 },                            /* ICON_HEADING         */
  { NULL, WPT_SYM_MAX+1 },                /* ICON_WPT             */
  { NULL, 2 },                            /* ICON_TB              */
  { NULL, 6 },                            /* ICON_MISC            */
  { NULL, WPT_SYM_MAX+CACHE_TYPE_MAX+10 },/* ICON_POS             */
  { NULL, 3 },                            /* ICON_FILE            */
  { NULL, -1 }
};

/* ICON_CACHE_TYPE / ICON_CACHE_TYPE_SEMI / ICON_CACHE_TYPE_2X */
const char *cache_type_icon_name[] = {
  "traditional", "multi",      "mystery", "virtual", "webcam", "event",
  "letterbox",   "earthcache", "wherigo", "megaevent", "cito",

  /* special overlays */
  "ovl_override", "ovl_found", "ovl_note", "ovl_mine",

  NULL
};

/* ICON_CACHE_SIZE */
const char *cache_size_icon_name[] = {
  "regular", "small",      "micro", "other", "not_chosen", "large",
  "virtual", NULL
};

/* ICON_STARS      */
const char *stars_icon_name[] = {
  "1", "1_5", "2", "2_5", "3", "3_5", "4", "4_5", "5", NULL
};

/* ICON_LOG        */
const char *log_icon_name[] = {
  "smile", "sad", "maint", "note", "big_smile", "enabled",
  "greenlight", "rsvp", "attended", "camera", "disabled",
  "needsmaint", "coord_update",
  "traffic_cone",
  "traffic_cone", /* LOG_TYPE_NEEDS_ARCHIVED */
  "traffic_cone", 
  NULL
};  

/* ICON_HEADING    */
const char *heading_icon_name[] = {
  "n", "ne", "e", "se", "s", "sw", "w", "nw", "none", NULL
};

/* ICON_WPT        */
const char *wpt_sym_icon_name[] = {
  "multistage", "parking", "final", "question",
  "trailhead", "refpoint", NULL
};

/* ICON_TB         */
const char *tb_icon_name[] = {
  "tb", "coin", NULL
};

/* ICON_MISC         */
const char *misc_icon_name[] = {
  "locked", "unlocked", "delete", 
#ifdef FREMANTLE
  "64x64/paypal", 
#else
  "32x32/paypal", 
#endif
  "32x32/pos_geomath", 
#ifdef FREMANTLE
  "64x64/gpxview",
#else
  "32x32/gpxview",
#endif
  NULL
};

/* ICON_POS        */
const char *pos_icon_name[] = {
  "wpt_multistage",         "wpt_parking",           "wpt_final",
  "wpt_question",           "wpt_trailhead",         "wpt_refpoint", 

  "cache_type_traditional", "cache_type_multi",      "cache_type_mystery", 
  "cache_type_virtual",     "cache_type_webcam",     "cache_type_event",
  "cache_type_letterbox",   "cache_type_earthcache", "cache_type_wherigo", 
  "cache_type_megaevent",   "cache_type_cito",

  "pos_wpt", "pos_gps", "pos_geomath", "pos_map", "pos_home",

  "pos_get", "pos_set", "maemo-mapper",

  NULL
};

/* ICON_FILE         */
const char *file_icon_name[] = {
  "gc", "folder", "zip", NULL
};

static void icons_load(int type, char *format, const char *names[]) {
  int i;

  if(!icons[type].count) {
    icons[type].data = NULL;
    return;
  }

  icons[type].data = malloc(sizeof(GdkPixbuf *) * icons[type].count);
  
  for(i=0;i<icons[type].count;i++) {
    g_assert(names[i]);

    GError *error = NULL;
    char filename[128];
    strcpy(filename, ICONPATH);
    snprintf(filename+strlen(filename),
	     sizeof(filename)-strlen(filename), format, names[i], "png");
    icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);
    
    if(error) {
      error = NULL;
      /* try again in local dir */
      strcpy(filename, "./data/icons/");
      snprintf(filename+strlen(filename), 
	       sizeof(filename)-strlen(filename), format, names[i], "png");
      icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);
	
      if(error) {
	icons[type].data[i] = NULL;
	g_warning("Could not load cache type icon %s: %s\n", 
		      names[i], error->message);
	g_error_free(error);
	error = NULL;
      }   
    }
  }
  
  g_assert(!names[i]);
}

void icons_init(void) {

  /* load cache type icons */
  icons_load(ICON_CACHE_TYPE, "32x32/cache_type_%s.%s", cache_type_icon_name);

  /* load semitransparent cache type icons */
  icons_load(ICON_CACHE_TYPE_SEMI, "32x32/cache_type_%s_semi.%s", cache_type_icon_name);

  /* load 150% sized cache type icons */
  icons_load(ICON_CACHE_TYPE_1_5X, "48x48/cache_type_%s.%s", cache_type_icon_name);

  /* load double sized cache type icons */
  icons_load(ICON_CACHE_TYPE_2X, "64x64/cache_type_%s.%s", cache_type_icon_name);

  /* load cache container/size icons */
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  icons_load(ICON_CACHE_SIZE, "45x12/cache_size_%s.%s", cache_size_icon_name);
#else
  icons_load(ICON_CACHE_SIZE, "90x24/cache_size_%s.%s", cache_size_icon_name);
#endif

  /* load cache difficulty/terrain/quality icons */
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  icons_load(ICON_STARS, "12x12/stars%s.%s", stars_icon_name);
#else
  icons_load(ICON_STARS, "16x16/stars%s.%s", stars_icon_name);
#endif

  /* load cache log icons */
  icons_load(ICON_LOG, "32x32/log_icon_%s.%s", log_icon_name);

  /* load icons to visualize heading */
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  icons_load(ICON_HEADING, "32x32/heading_%s.%s", heading_icon_name);
#else
  icons_load(ICON_HEADING, "48x48/heading_%s.%s", heading_icon_name);
#endif

  /* load waypoint icons */
  icons_load(ICON_WPT, "32x32/wpt_%s.%s", wpt_sym_icon_name);

  /* load travelbug/coin icons */
  icons_load(ICON_TB, "32x32/%s.%s", tb_icon_name);

  /* load misc icons */
  icons_load(ICON_MISC, "%s.%s", misc_icon_name);

  /* load pos icons */
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  icons_load(ICON_POS, "16x16/%s.%s", pos_icon_name);
#else
  icons_load(ICON_POS, "32x32/%s.%s", pos_icon_name);
#endif

  /* load file icons */
  icons_load(ICON_FILE, "file_%s.%s", file_icon_name);
}

void icons_free(void) {
  int i;
  struct icon_data *icon = icons;

  while(icon->count >= 0) {
    if(icon->count) {
      for(i=0;i<icon->count;i++)
	if(icon->data[i])
	  gdk_pixbuf_unref(icon->data[i]);

      free(icon->data);
    }

    icon++;
  }
}

GdkPixbuf *icon_get(int type, int num) {
  if(num < 0)                  return NULL;
  if(num >= icons[type].count) return NULL;

  return icons[type].data[num];
}

GtkWidget *icon_get_widget(int type, int num) {
  GdkPixbuf *pbuf = icon_get(type, num);
  if(!pbuf) return NULL;

  return gtk_image_new_from_pixbuf(pbuf);
}

GdkPixbuf *icon_bearing(pos_t from, pos_t to) {
  if(isnan(from.lat) || isnan(from.lon) || 
     isnan(to.lat) || isnan(to.lon))
    return icon_get(ICON_HEADING, 8);

  int idx = (gpx_pos_get_bearing(from, to)+22.5)/45.0;

  /* make sure we stay in icon bounds */
  if(idx < 0) idx += 8;
  if(idx > 7) idx -= 8;
  return icon_get(ICON_HEADING, idx);
}

