/*
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <glib/gstdio.h>

#include <fcntl.h>

#ifndef LIBXML_TREE_ENABLED
#error "Tree not enabled in libxml"
#endif

#include "gpxview.h"

#define ID_PATTERN  "guid="

static size_t mem_write(void *ptr, size_t size, size_t nmemb,
                        void *stream) {
  curl_mem_t *p = (curl_mem_t*)stream;

  p->ptr = g_realloc(p->ptr, p->len + size*nmemb + 1);
  if(p->ptr) {
    memcpy(p->ptr+p->len, ptr, size*nmemb);
    p->len += size*nmemb;
    p->ptr[p->len] = 0;
  }
  return nmemb;
}

/* return list of all votes in this file */
static gcvote_t *votes_parse(xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;

  gcvote_t *votes = NULL, **cur = &votes;

  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {

      if(strcasecmp((char*)cur_node->name, "vote") == 0) {
	*cur = g_new0(gcvote_t, 1);
	(*cur)->quality = GCVOTE_NONE;

	/* unhandled attributes: */
	/* userName, voteAvg, voteUser, waypoint, vote1-5 */

	/* parse votes attributes */
	char *prop_str = (char*)xmlGetProp(cur_node, BAD_CAST "voteMedian");
	if(prop_str) { 
	  (*cur)->quality = atoi(prop_str);
	  xmlFree(prop_str);
	}

	prop_str = (char*)xmlGetProp(cur_node, BAD_CAST "voteCnt");
	if(prop_str) {
	  (*cur)->votes = atoi(prop_str);
	  xmlFree(prop_str);
	}

	prop_str = (char*)xmlGetProp(cur_node, BAD_CAST "cacheId");
	if(prop_str) {
	  (*cur)->id = g_strdup(prop_str);
	  xmlFree(prop_str);
	}

	cur = &(*cur)->next;
      } else if(strcasecmp((char*)cur_node->name, "errorstring") == 0) {
	/* ignore for now ... */
      } else 
	printf("found unhandled votes/%s\n", cur_node->name);
    }
  } 
  return votes;
}

/* parse root element */
static gcvote_t *parse_root(xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;
  gcvote_t *votes = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {

      if(strcasecmp((char*)cur_node->name, "votes") == 0) {
	if(!votes)
	  votes = votes_parse(doc, cur_node);
	else
	  printf("gcvote: ignoring multiple votes lists\n");
      } else 
	printf("found unhandled %s\n", cur_node->name);
    }
  }
  return votes;
}

static gcvote_t *parse_doc(xmlDocPtr doc) {
  /* Get the root element node */
  xmlNode *root_element = xmlDocGetRootElement(doc);
  gcvote_t *votes = parse_root(doc, root_element);  

  /*free the document */
  xmlFreeDoc(doc);

  /*
   * Free the global variables that may
   * have been allocated by the parser.
   */
  xmlCleanupParser();

  return votes;
}

static void curl_set_proxy(CURL *curl, proxy_t *proxy) {
  if(proxy && proxy->host) {
    if(proxy->ignore_hosts)
      printf("WARNING: Proxy \"ignore_hosts\" unsupported!\n");

    printf("gcvote: using proxy %s:%d\n", proxy->host, proxy->port);

    curl_easy_setopt(curl, CURLOPT_PROXY, proxy->host);
    curl_easy_setopt(curl, CURLOPT_PROXYPORT, proxy->port);

    if(proxy->use_authentication && 
       proxy->authentication_user && proxy->authentication_password) {
      printf("gcvote:   use auth for user %s\n", proxy->authentication_user);

      char *cred = g_strdup_printf("%s:%s",
                                   proxy->authentication_user,
                                   proxy->authentication_password);

      curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD, cred);
      g_free(cred);
    }
  } else
    printf("gcvote: no proxy configured\n");
}

void gcvote_request_free(gcvote_request_t *request) {
  /* decrease refcount and only free structure if no references are left */
  request->refcount--;
  if(request->refcount > 0) {
    printf("gcvote: still %d references, keeping request\n", 
	   request->refcount);
    return;
  }

  printf("gcvote: no references left, freeing request\n");
  if(request->url)  g_free(request->url);

  if(request->mem.ptr)
    g_free(request->mem.ptr);

  g_free(request);
}

/* gcvote net result handler */
static gboolean gcvote_result_handler(gpointer data) {
  gcvote_request_t *request = (gcvote_request_t*)data;

  printf("gcvote: result handler\n");

  if(request->refcount < 2) {
    printf("gcvote: main app isn't listening anymore\n");
    gcvote_request_free(request);
    return FALSE;
  }

  if(request->res) {
    printf("gcvote: curl failed\n");
    request->cb(NULL, request->userdata);
    gcvote_request_free(request);
    return FALSE;
  }

  /* parse the reply */
  /* nothing could be parsed, just give up */
  if(!request->mem.ptr || !request->mem.len) { 
    printf("gcvote: ignoring zero length reply\n");
    request->cb(NULL, request->userdata);
    gcvote_request_free(request);
    return FALSE;
  }

  /* start XML parser */
  LIBXML_TEST_VERSION;

  /* parse the file and get the DOM */
  xmlDoc *doc = xmlReadMemory(request->mem.ptr, request->mem.len, 
			      NULL, NULL, 0);

  /* nothing could be parsed, just give up */
  if(!doc) { 
    request->cb(NULL, request->userdata);
    gcvote_request_free(request);
    return FALSE;
  }

  vote_t *vote = NULL;
  gcvote_t *votes = parse_doc(doc);
  
  /* search for requested cache in reply and free chain */
  while(votes) {
    gcvote_t *next = votes->next;

    if(!vote && !strcmp(votes->id, request->id) && (votes->quality > 0)) {

      vote = g_new0(vote_t, 1);
      vote->quality = votes->quality;
      vote->votes = votes->votes;
    }

    if(votes->id) g_free(votes->id);
    g_free(votes);
    votes = next;
  }

  if(vote) {
    printf("gcvote: worker found vote %d/%d\n", vote->quality, vote->votes);
    request->cb(vote, request->userdata);
  } else
    printf("gcvote: no vote found\n");

  gcvote_request_free(request);
  return FALSE;
}

static void gcvotes_free(gcvote_t *votes) {
  while(votes) {
    gcvote_t *next = votes->next;
    if(votes->id) g_free(votes->id);
    g_free(votes);
    votes = next;
  }
}

void gcvote_save(appdata_t *appdata, cache_t *cache, curl_mem_t *mem) {
  if(!mem->len || !mem->ptr) return;

  /* save data to disk */
  char *filename = g_strdup_printf("%s%s/gcvote.xml",
				   appdata->image_path,
				   cache->id);
  if(checkdir(filename) != 0) 
    printf("gcvote: unable to create file path\n");
  else {
    printf("gcvote: write %d bytes to %s\n", mem->len, filename);

    int handle = g_open(filename, O_WRONLY | O_CREAT, 0644);
    if(handle >= 0) {
      int len = write(handle, mem->ptr, mem->len);
      close(handle);
      
      /* if write failed, then remove the file */
      if(len != mem->len) 
	g_remove(filename);
    }
  }

  free(filename);
}

static void *worker_thread(void *ptr) {
  gcvote_request_t *request = (gcvote_request_t*)ptr;
  struct curl_httppost *formpost=NULL;
  struct curl_httppost *lastptr=NULL;
  
  printf("gcvote: worker thread running\n");

  request->mem.ptr = NULL;
  request->mem.len = 0;

  printf("gcvote: request to get votes for id %s\n", request->id);

  curl_formadd(&formpost, &lastptr,
	       CURLFORM_COPYNAME, "version",
	       CURLFORM_COPYCONTENTS, APP VERSION,
	       CURLFORM_END);
  
  curl_formadd(&formpost, &lastptr,
	       CURLFORM_COPYNAME, "cacheIds",
	       CURLFORM_COPYCONTENTS, request->id,
	       CURLFORM_END);
  
  /* try to init curl */
  CURL *curl = curl_easy_init();
  if(!curl) {
    curl_formfree(formpost);

    /* callback anyway, so main loop can also clean up */
    g_idle_add(gcvote_result_handler, request); 
  
    return NULL;
  }
  
  curl_set_proxy(curl, request->proxy);
  
  /* play nice and report some user agent */
  curl_easy_setopt(curl, CURLOPT_USERAGENT, APP " " VERSION);
  
  /* download to memory */
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, &request->mem);
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, mem_write);
  
  /* what URL that receives this POST */ 
  curl_easy_setopt(curl, CURLOPT_URL, 
		   "http://dosensuche.de/GCVote/getVotes.php");
  curl_easy_setopt(curl, CURLOPT_HTTPPOST, formpost);
  
  request->res = curl_easy_perform(curl);
  printf("gcvote: curl perform returned with %d\n", request->res);
  
  //  printf("received %d bytes\n", mem.len);
  //  printf("data: %s\n", mem.ptr);
  
  /* always cleanup */ 
  curl_easy_cleanup(curl);

  printf("gcvote: 1\n");

  /* then cleanup the formpost chain */ 
  curl_formfree(formpost);

  printf("gcvote: 2\n");

  /* cause gtk main loop to handle result only if main loop */
  /* is still interested. Don't free request then, since the */
  /* gcvote_result_handler will do this */
  if(request->refcount > 1) 
    g_idle_add(gcvote_result_handler, request); 
  else
    gcvote_request_free(request);
  
  printf("gcvote: worker thread done\n");

  return NULL;
}

gcvote_request_t *gcvote_request(appdata_t *appdata, gcvote_cb cb, 
			       char *url, gpointer data) {
  if(!url) return NULL;
  
  /* ------ prepare request for worker thread --------- */
  gcvote_request_t *request = g_new0(gcvote_request_t, 1);
  request->url = g_strdup(url);
  request->id = strstr(request->url, ID_PATTERN);
  if(!request->id) {
    printf("gcvote: unable to find id\n");
    gcvote_request_free(request);
    return NULL;
  }
  
  request->proxy = appdata->proxy;
  request->id += strlen(ID_PATTERN);
  request->refcount = 2;   // master and worker hold a reference
  request->cb = cb;
  request->userdata = data;

  if(!g_thread_create(&worker_thread, request, FALSE, NULL) != 0) {
    g_warning("gcvote: failed to create the worker thread");

    /* free request and return error */
    request->refcount--;    /* decrease by one for dead worker thread */
    gcvote_request_free(request);
    return NULL;
  }

  return request;
}

vote_t *gcvote_restore(appdata_t *appdata, cache_t *cache) {
  /* load data from disk */
  char *filename = g_strdup_printf("%s%s/gcvote.xml",
				   appdata->image_path,
				   cache->id);

  printf("gcvote: trying to restore from %s\n", filename);

  /* no such file? */
  if(!g_file_test(filename, G_FILE_TEST_EXISTS)) {
    printf("gcvote: no such file\n");
    free(filename);
    return NULL;
  }

  LIBXML_TEST_VERSION;
  xmlDoc *doc = xmlReadFile(filename, NULL, 0);

  if(doc == NULL) {
    printf("gcvote: error, could not parse file %s\n", filename);
    free(filename);
    return NULL;
  }
 
  free(filename);

  /* in this case this will sure only return one result */
  gcvote_t *votes = parse_doc(doc);
  
  if(!votes) {
    printf("gcvote: error, no vote found\n");
    free(filename);
    return NULL;
  }

  vote_t *vote = g_new0(vote_t, 1);
  vote->quality = votes->quality;
  vote->votes = votes->votes;

  printf("gcvote: found vote %d/%d\n", vote->quality, vote->votes);

  gcvotes_free(votes);

  return vote;
}
