/*
 * This file is part of gnokii-gconf
 *
 * Copyright (C) 2008 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>

#include <config.h>

/* strlen needed from string.h */
#include <string.h>

#if HILDON == 1
#include <hildon/hildon.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>
#else
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-font-selection-dialog.h>
#endif

#include <gnokii.h>
#include <gnokii-gconf-private.h>

#include <libosso.h>

#include <libintl.h>
#include <locale.h>

#define _(String) gettext(String)

#define MODEL_SIZE 6

const char *
gn_error_to_string (gn_error error)
{
	switch (error) {
	/* General codes */
	case GN_ERR_NONE:
		return NULL;
	case GN_ERR_FAILED:
		return "Command failed";
	case GN_ERR_UNKNOWNMODEL:
		return "Model specified isn't known/supported.";
	case GN_ERR_INVALIDSECURITYCODE:
		return "Invalid Security code.";
	case GN_ERR_INTERNALERROR:
		return "Problem occured internal to model specific code.";
	case GN_ERR_NOTIMPLEMENTED:
		return "Command called isn't implemented in model.";
	case GN_ERR_NOTSUPPORTED:
		return "Function not supported by the phone";
	case GN_ERR_USERCANCELED:
		return "User aborted the action.";
	case GN_ERR_UNKNOWN:
		return "Unknown error - well better than nothing";
	case GN_ERR_MEMORYFULL:
		return "The specified memory is full.";
	/* Statemachine */
	case GN_ERR_NOLINK:
		return "Couldn't establish link with phone.";
	case GN_ERR_TIMEOUT:
		return "Command timed out.";
	case GN_ERR_TRYAGAIN:
		return "Try again.";
	case GN_ERR_WAITING:
		return "Waiting for the next part of the message.";
	case GN_ERR_NOTREADY:
		return "Device not ready.";
	case GN_ERR_BUSY:
		return "Command is still being executed.";
	default:
		return "XXX Other error, FIXME";
	}
}


void getPhone_cb(GtkButton *button, GtkEntry *btEntry) {
	hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("Fonction not yet implemented : You'll have to get your Phone BT MAC Address yourself !"));
}

void identifyPhone_cb(GtkButton *button, GtkEntry *btEntry) {
	gchar *btMac;
	gchar *gnokiiCfg[6] = {"[global]", "port = ", "model = AT", "connection = bluetooth", "rfcomm_channel = 2", NULL};
	gn_data data;
	struct gn_statemachine state;
	gn_error err;
	char model[MODEL_SIZE];
	
	memset (model, 0, MODEL_SIZE);
	
	btMac = gtk_entry_get_text(GTK_ENTRY(btEntry));
	if (!btMac) {
		hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("Cannot identify a phone without his BT Mac Address !"));
		return;
	}
	
	gnokiiCfg[1] = g_strconcat(gnokiiCfg[1], btMac, NULL);
	if (gn_cfg_memory_read ((const char **)gnokiiCfg) < 0) {
		hildon_banner_show_informationf(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("Error reading gnokii configuration\n%s\n%s\n%s\n%s"), gnokiiCfg[0], gnokiiCfg[1], gnokiiCfg[2], gnokiiCfg[3]);
/*		g_free (gnokiiCfg[1]); */
		return;
	}
	
	memset (&state, 0, sizeof(struct gn_statemachine));
	
	if (gn_cfg_phone_load("", &state) < 0) {
		hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("gn_cfg_phone_load"));
		return;
	}

	err = gn_gsm_initialise(&state);
	if (err != GN_ERR_NONE) {
		hildon_banner_show_informationf(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("gn_gsm_initialise: %d"), err);
		return;
	}

	gn_data_clear(&data);
	data.model = model;
	
	err = gn_sm_functions(GN_OP_Identify, &data, &state);
	if (err != GN_ERR_NONE) {
		hildon_banner_show_informationf(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("gnokii identify : %s"), gn_error_to_string(err));
		return;
	}
	
	if (model[0] == '\0') {
		hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("gnokii cannot identify your phone : use AT"));
	} else {
		hildon_banner_show_informationf(GTK_WIDGET(gtk_widget_get_toplevel(button)), GTK_STOCK_DIALOG_INFO, _("gnokii identify your phone as : %s"), model);
	}
	
	gn_sm_functions (GN_OP_Terminate, NULL, &state);
	gn_data_clear (&data);
}

osso_return_t execute(osso_context_t *osso, gpointer data, gboolean user_activated)
{
	GtkWidget *dialog;
	GConfClient *gc_client;
	gint response;
	gchar *btMac;
	gchar *model;
	gchar *rfcomm;
	GtkWidget *modelCombo;
	GtkWidget *btEntry;
	GtkWidget *rfcommEntry;
	GtkWidget *button;
	GtkWidget *label;
	GtkWidget *table;
	gint num;
	gchar *supportedModel;
	
	/* Init type system */
	g_type_init();

	/* Load default GConf path */
	gc_client = gconf_client_get_default();

	if (gc_client == NULL) {
	    return FALSE;
	}

	/* Get string variable */
	btMac = gconf_client_get_string(gc_client, GNOKII_GCONF_BT_KEY, NULL);
	model = gconf_client_get_string(gc_client, GNOKII_GCONF_MODEL_KEY, NULL);
	rfcomm = gconf_client_get_string(gc_client, GNOKII_GCONF_RFCOMM_KEY, NULL);

	if (!btMac) {
		/* Get preferred phone */
		btMac = gconf_client_get_string(gc_client, "/system/osso/connectivity/BT/preferred", NULL);
	}
	if (!model) {
		/* base model */
		model = g_strdup("AT");
	}
	if (!rfcomm) {
		/* rfcomm : 2 */
		rfcomm = g_strdup("2");
	}
	
	/* Create dialog with OK and Cancel buttons. Leave the separator out, as we do not have any content. */
	dialog = gtk_dialog_new_with_buttons(
		_("Gnokii DBus control panel"),
		GTK_WINDOW(data),
		GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_OK,
		GTK_RESPONSE_OK,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_CANCEL,
		NULL);

	/* ... add something to the dialog ... */
	table = gtk_table_new(3, 3, TRUE);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), table);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	
	/* Add a label, a field & 1 button */
	label = gtk_label_new(_("phone BT Mac"));
	gtk_table_attach_defaults( GTK_TABLE(table), label, 0, 1, 0, 1);

	btEntry = gtk_entry_new();
	if (btMac) {
		gtk_entry_set_text(GTK_ENTRY(btEntry), btMac);
	}
	gtk_table_attach_defaults( GTK_TABLE(table), btEntry, 1, 2, 0, 1);

	button = gtk_button_new_with_label ("Get Phone");
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	g_signal_connect (G_OBJECT (button), "clicked",
	                  G_CALLBACK (getPhone_cb), (gpointer) btEntry);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 0, 1);
	gtk_widget_grab_default(button);
	
	
	label = gtk_label_new(_("phone Model"));
	gtk_table_attach_defaults( GTK_TABLE(table), label, 0, 1, 1, 2);

	modelCombo = gtk_combo_box_entry_new_text();
	supportedModel = g_strdup("AT");
	gtk_combo_box_append_text(GTK_COMBO_BOX(modelCombo), supportedModel);
	if (model && strcmp(supportedModel, model) == 0) {
		gtk_combo_box_set_active(GTK_COMBO_BOX(modelCombo), 0);
	}
	g_free(supportedModel);
	num = 0;
	while (gn_lib_get_supported_phone_model(num) != NULL) {
		supportedModel = g_strdup(gn_lib_get_supported_phone_model(num));
		gtk_combo_box_append_text(GTK_COMBO_BOX(modelCombo), supportedModel);
		if (model && strcmp(supportedModel, model) == 0) {
			gtk_combo_box_set_active(GTK_COMBO_BOX(modelCombo), num+1);
		}
		num++;
		g_free(supportedModel);
	}
	gtk_table_attach_defaults( GTK_TABLE(table), modelCombo, 1, 2, 1, 2);
	
	button = gtk_button_new_with_label ("Identify Phone");
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	g_signal_connect (G_OBJECT (button), "clicked",
	                  G_CALLBACK (identifyPhone_cb), (gpointer) btEntry);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 1, 2);
	
	label = gtk_label_new(_("rfcomm"));
	gtk_table_attach_defaults( GTK_TABLE(table), label, 0, 1, 2, 3);

	rfcommEntry =  gtk_spin_button_new_with_range( 1, 10, 1);
	if (rfcomm) {
		gtk_entry_set_text(GTK_ENTRY(rfcommEntry), rfcomm);
	}
	gtk_table_attach_defaults( GTK_TABLE(table), rfcommEntry, 1, 2, 2, 3);

	

	gtk_window_set_default_size(GTK_WINDOW(dialog), 500, 300);
	gtk_widget_show_all (dialog);
	
#if HILDON == 1
	if (!user_activated)
	{



	}
#endif
	/* Wait until user finishes the dialog. */
	response = gtk_dialog_run(GTK_DIALOG(dialog));

	if (response == GTK_RESPONSE_OK)
	{
		btMac = gtk_entry_get_text(GTK_ENTRY(btEntry));
		if (btMac) {
			/* Set string variable */
			gconf_client_set_string(gc_client, GNOKII_GCONF_BT_KEY, btMac, NULL);
		}
		model = gtk_combo_box_get_active_text(GTK_COMBO_BOX(modelCombo));
		if (model) {
			/* Set string variable */
			gconf_client_set_string(gc_client, GNOKII_GCONF_MODEL_KEY, model, NULL);
		}
		rfcomm = gtk_entry_get_text(GTK_ENTRY(rfcommEntry));
		if (rfcomm) {
			/* Set string variable */
			gconf_client_set_string(gc_client, GNOKII_GCONF_RFCOMM_KEY, rfcomm, NULL);
		}
	}

	/* Free the dialog (and it's children) */
	gtk_widget_destroy(GTK_WIDGET(dialog));

	/* Free gconf settings object */
	g_object_unref(gc_client);

	return OSSO_OK;
}             
     
osso_return_t save_state(osso_context_t *osso, gpointer data)
{
        /* ... save state ... */

        return OSSO_OK;
}
