/*
  This file is part of "GiveMeFive" - A simple "five in a row" game.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Control.h"
#include <iostream>
Control::Control()
 : backgroundColor("white",Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   lineColor("black",Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   player1Color("red",Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   player2Color("black",Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   markColor("yellow",Lum::OS::display->GetColor(Lum::OS::Display::fillColor))
{
  SetCanFocus(true);
  RequestFocus();
}

bool Control::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<GameArea*>(model);

  Lum::Control::SetModel(this->model);

  return this->model.Valid();
}

void Control::CalcSize()
{
  if (Lum::OS::display->GetSize()<=Lum::OS::Display::sizeSmall) {
    cellSize=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional)->height/2;
  }
  else {
    cellSize=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional)->height;
  }

  minWidth=(GameArea::dimension+1)*1+GameArea::dimension*cellSize;
  minHeight=minWidth;

  width=minWidth;
  height=minHeight;

  Lum::Control::CalcSize();
}

void Control::Layout()
{
  cellSize=(std::min(width,height)-(GameArea::dimension+1)*1)/GameArea::dimension;

  Lum::Control::Layout();
}

void Control::Draw(int x, int y, size_t w, size_t h)
{
  Lum::Control::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();
  int               offset;

  draw->PushForeground(lineColor);

  if (width>cellSize*GameArea::dimension+GameArea::dimension+1) {
    DrawParentBackground(this->x+cellSize*GameArea::dimension+GameArea::dimension+1,this->y,
                         width-(cellSize*GameArea::dimension+GameArea::dimension+1),height);
  }

  if (height>cellSize*GameArea::dimension+GameArea::dimension+1) {
    DrawParentBackground(this->x,this->y+cellSize*GameArea::dimension+GameArea::dimension+1,
                         width,height-(cellSize*GameArea::dimension+GameArea::dimension+1));
  }

  // Vertical lines
  offset=0;
  for (size_t i=0; i<=GameArea::dimension; i++) {
    draw->DrawLine(this->x+offset,this->y,
                   this->x+offset,this->y+(cellSize*GameArea::dimension+GameArea::dimension+1)+-1);
    offset+=1+cellSize;
  }

  // Horizontal lines
  offset=0;
  for (size_t i=0; i<=GameArea::dimension; i++) {
    draw->DrawLine(this->x,this->y+offset,
                   this->x+(cellSize*GameArea::dimension+GameArea::dimension+1)-1,this->y+offset);
    offset+=1+cellSize;
  }

  draw->PopForeground();

  // cells
  for (size_t x=0; x<GameArea::dimension; x++) {
    for (size_t y=0; y<GameArea::dimension; y++) {
      if (model.Valid() && model->IsMarked(x,y)){
        draw->PushForeground(markColor);
        draw->FillRectangle(this->x+1+x*(cellSize+1),
                            this->y+1+y*(cellSize+1),
                            cellSize,cellSize);
        draw->PopForeground();
      }
      else {
        draw->PushForeground(backgroundColor);
        draw->FillRectangle(this->x+1+x*(cellSize+1),
                            this->y+1+y*(cellSize+1),
                            cellSize,cellSize);
        draw->PopForeground();
      }
    }
  }

  if (!model.Valid()) {
    return;
  }

  // Stones
  for (size_t x=0; x<GameArea::dimension; x++) {
    for (size_t y=0; y<GameArea::dimension; y++) {
      switch (model->Get(x,y)) {
      case GameArea::player1:
        draw->PushForeground(player1Color);
        draw->FillArc(this->x+1+x*(cellSize+1),
                      this->y+1+y*(cellSize+1),
                      cellSize,
                      cellSize,
                      0,
                      64*360);
        draw->PopForeground();

        draw->PushForeground(lineColor);
        draw->DrawArc(this->x+1+x*(cellSize+1),
                      this->y+1+y*(cellSize+1),
                      cellSize,
                      cellSize,
                      0,
                      64*360);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::whiteColor);
        draw->DrawArc(this->x+1+x*(cellSize+1)+cellSize/4,
                      this->y+1+y*(cellSize+1)+cellSize/4,
                      cellSize/2,
                      cellSize/2,
                      60*64,90*64);
        draw->PopForeground();

        break;
      case GameArea::player2:
        draw->PushForeground(player2Color);
        draw->FillArc(this->x+1+x*(cellSize+1),
                      this->y+1+y*(cellSize+1),
                      cellSize,
                      cellSize,
                      0,
                      64*360);
        draw->PopForeground();

        draw->PushForeground(lineColor);
        draw->DrawArc(this->x+1+x*(cellSize+1),
                      this->y+1+y*(cellSize+1),
                      cellSize,
                      cellSize,
                      0,
                      64*360);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::whiteColor);
        draw->DrawArc(this->x+1+x*(cellSize+1)+cellSize/4,
                      this->y+1+y*(cellSize+1)+cellSize/4,
                      cellSize/2,
                      cellSize/2,
                      60*64,90*64);
        draw->PopForeground();
        break;
      case GameArea::empty:
        break;
      }
    }
  }
}

bool Control::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (model->GetState()!=GameArea::stateHuman) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down &&
      PointIsIn(event) &&
      event.button==Lum::OS::MouseEvent::button1) {

    for (size_t x=0; x<GameArea::dimension; x++) {
      for (size_t y=0; y<GameArea::dimension; y++) {
        int cx,cy;

        cx=this->x+1+x*(cellSize+1);
        cy=this->y+1+y*(cellSize+1);

        if (event.x>=cx && event.x<(int)(cx+cellSize) &&
            event.y>=cy && event.y<(int)(cy+cellSize)) {
          if (!model->IsSet(x,y)) {
            model->Set(x,y,model->GetCurrentPlayer());
            model->PlayerFinished();
            break;
          }
        }
      }
    }

    return true;
  }

  return false;
}

bool Control::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{

  if (event.type==Lum::OS::KeyEvent::down) {
    switch (event.key) {
    case Lum::OS::keyLeft:
      return true;
    case Lum::OS::keyRight:
      return true;
    case Lum::OS::keyUp:
      return true;
    case Lum::OS::keyDown:
      return true;
    case Lum::OS::keySpace:
      return true;
    default:
      break;
    }
  }

  return false;
}

void Control::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==this->model && visible) {
    Redraw();
  }
}

