/*
 *  Copyright (c) 2002-2007 Jiri Benc <jbenc@upir.cz>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * save.c
 *
 * Vsechno k ukladani a nahravani pozice (vcetne GUI).
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "SDL.h"
#include "level.h"
#include "readdata.h"
#include "boxes.h"
#include "params.h"
#include "keys.h"
#include "graphics.h"
#include "common.h"
#include "save.h"


/*! vyhodit */
typedef struct _save_struct {
  int actuallevel, actuallevelset;
  int phasefred[2];
  int phaserun, phasebagg;
  int fredup[2];
  int phaseactbagg;
  int gamespeed;
  int thingcnt;
  uchar *levelbagger;
  thing_stat *thingstat;
} save_struct;

/*
  Struktura save souboru:

 +0     21 bytu         "Fred Save File 2.0\13\10\26" (neukonceno '\0'!)
 +21    16 bytu string  Nazev 1. slotu (Pascalovsky string)
 +37     2 byty word    Zacatek 1. slotu (0=neni)
 +39    16 bytu string  Nazev 2. slotu
 +55     2 byty word    Zacatek 2. slotu
                :
 +147   16 bytu string  Nazev 8. slotu
 +163    2 byty word    Zacatek 8. slotu
 +165   Jednotlive sloty

  Struktura jednoho slotu:

 +0      2 byty word  Delka slotu v bytech (nepouzito)
 +2      1 byte       actuallevel xor 0x6a
 +3      1 byte       bity 0..6 = actuallevelset
                      bit 7 = phaseactbagg (stavova promenna)
 +4      2 byty       Stavove promenne:
                      1. byte: bit 0 = phasefred[0]
                               bit 1 = phasefred[1]
                               bit 2 = fredup[0]
                               bit 3 = fredup[1]
                               bity 4..6 = phaserun
                               bit 7 = phasebagg
                      2. byte: bity 0..3 = gamespeed
                               bity 4..7 = gamemenuitem (nepouzito)
 +6     48 bytu       Pole levelbagger
 +54     2 byty word  Promenna thingcnt
 +56     n bytu       Prvnich thingcnt polozek pole thingstat
                      - kazda polozka je ulozena jako trojbyte:
                      1. byte: bity 0..7 = thingstat->x (spodnich 8 bitu)
                      2. byte: bit 0     = thingstat->x (9. bit)
                               bity 1..7 = levelstat[y, x] (spodnich 7 bitu)
                      3. byte            = thingstat->y
 +x      1 byte       kontrolni soucet slotu
 Jeden slot: maximalne 1377 bytu
*/
#define CNT_SLOT                8               /* pocet slotu v savu */

#define SAVEID_LEN              21              /* delka idu */

#define SAVE_NAME               SAVEID_LEN      /* poloha jmena slotu v hlavicce */
#define SAVE_NAME_SIZE          16              /* velikost jednoho jmena */
#define SAVE_POSPOS             (SAVE_NAME + SAVE_NAME_SIZE)
                                /* pozice polozky se zacatkem slotu */
#define SLOTHEAD_SIZE           (SAVE_NAME_SIZE + 2)
                                /* velikost casti hlavicky prislusne jednomu slotu */

#define SAVEHEAD_SIZE           (SAVEID_LEN + CNT_SLOT * SLOTHEAD_SIZE)
                                /* velikost cele hlavicky */

#define SLOT_FIXSIZE            56              /* velikost pevne casti slotu */
#define SLOT_MAXSIZE            (MAXSIZE_THINGSTAT * 3 + 57)
                                /* maximalni velikost jednoho slotu */

/* ofsety promennych ve slotu: */
#define SAVE_ACTUALLEVEL        2
#define SAVE_ACTUALLEVELSET     3
#define SAVE_PHASEACTBAGG       SAVE_ACTUALLEVELSET
#define SAVE_STATUS1            4
#define SAVE_STATUS2            5
#define SAVE_LEVELBAGGER        6
#define SAVE_THINGCNT           54
#define SAVE_THINGSTAT          56

#define SAVE_XOR                0x6a


/* pozice okna s nahledem */
#define PREVIEW_X               400
#define PREVIEW_Y               154


char savefilename[] =                   /* jmeno savu */
#ifdef win32
    "fred.sav";
#else
    ".fredsave";
#endif
static char filename[80];               /* plne jmeno savu */

static char saveid[SAVEID_LEN + 1] = "Fred Save File 2.0\x0d\x0a\x1a";
                                        /* identifikator zacatku savu */

static int savechanged;                 /* priznak, zda byl zmenen obsah savesouboru */
static int emptyslot[CNT_SLOT];         /* priznaky, zda jsou sloty prazdne */
static char slotnames[CNT_SLOT][SAVE_NAME_SIZE];
                                        /* nazvy slotu */

static FILE *fsave;                     /* soubor se savem */
static uchar *buffer;                   /* buffer */

int savemenuitem;                       /* aktualne vybrana polozka savu */
                                           

/* Uzitecna makra pro praci s bufferem. */
#define save_getword(pos)           ((int)buffer[(pos)] + buffer[(pos) + 1] * 0x100)
#define save_putword(pos, w)        buffer[(pos)] = (w) & 0xff; buffer[(pos) + 1] = ((w) >> 8) & 0xff;
#define slot_getword(slot, pos)     ((int)buffer[SAVEHEAD_SIZE + (slot) * SLOT_MAXSIZE + (pos)]\
    + buffer[SAVEHEAD_SIZE + (slot) * SLOT_MAXSIZE + (pos)] * 0x100)
#define slot_putword(slot, pos, w)  buffer[SAVEHEAD_SIZE + (slot) * SLOT_MAXSIZE + (pos)] = (w) & 0xff;\
    buffer[SAVEHEAD_SIZE + (slot) * SLOT_MAXSIZE + (pos) + 1] = ((w) >> 8) & 0xff;

#define buf_getword(buf, pos)       ((int)(buf)[(pos)] + (buf)[(pos) + 1] * 0x100)
#define buf_putword(buf, pos, w)    (buf)[(pos)] = (w) & 0xff; (buf)[(pos) + 1] = ((w) >> 8) & 0xff;

/* Spocita kontrolni soucet a vrati ho jako navratovou hodnotu. */
uchar checksum(uchar *p, int size)
{
  uchar sum;

  for (sum = 0; size > 0; size--) sum += *p++;
  return sum ^ SAVE_XOR;
}

    
/* Vynuluje struktury a pripravi tak pudu pro novy save soubor. */
void new_savefile()
{
  /* zkopirovani idu, ale bez koncove nuly */
  memcpy(buffer, saveid, SAVEID_LEN);
  /* vynulovani hlavicky savu */
  memset(buffer + SAVEID_LEN, 0, SLOTHEAD_SIZE * CNT_SLOT);
  if (fsave) {
    fclose(fsave);
    fsave = NULL;
  }
}

/* Otevre soubor se savem. */
void open_savefile()
{
  int i1;
  int pos, w1;
  uchar *p;

  /* inicializace struktur */
  savechanged = 0;
  for (i1 = 0; i1 < CNT_SLOT; i1++) emptyslot[i1] = 1;
  memset(slotnames, 0, sizeof(slotnames));
  /* otevreni souboru */
  fsave = fopen(filename, "rb");
  if (!fsave) {
    new_savefile();
    return;
  }
  /* nacteme hlavicku */
  if (fread(buffer, 1, SAVEHEAD_SIZE, fsave) != SAVEHEAD_SIZE) {
    /* soubor je prilis kratky? */
    new_savefile();
    return;
  }
  /* zkontrolujeme id */
  for (i1 = 0; i1 < SAVEID_LEN; i1++)
    if (buffer[i1] != saveid[i1]) {
      /* spatny id */
      new_savefile();
      return;
    }

  /* postupne nacteme jednotlive sloty */
  for (i1 = 0; i1 < CNT_SLOT; i1++) {
    /* precteme pozici slotu */
    pos = save_getword(SAVE_POSPOS + (i1 * SLOTHEAD_SIZE));
    if (!pos) continue;                 /* slot je prazdny => jdeme dal */
    fseek(fsave, pos, SEEK_SET);
    /* nacteme pevnou cast slotu */
    p = buffer + SAVEHEAD_SIZE + i1 * SLOT_MAXSIZE;
    fread(p, 1, SLOT_FIXSIZE, fsave);
    w1 = buf_getword(p, SAVE_THINGCNT);
    if (w1 > MAXSIZE_THINGSTAT) {
      w1 = 0;
      slot_putword(i1, SAVE_THINGCNT, w1);
    }
    fread(p + SLOT_FIXSIZE, 1, w1 * 3 + 1, fsave);
    /* zkontrolujeme kontrolni soucet */
    if (checksum(p, SLOT_FIXSIZE + w1 * 3) == p[SLOT_FIXSIZE + w1 * 3]) {
      emptyslot[i1] = 0;
      w1 = buffer[SAVE_NAME + (i1 * SLOTHEAD_SIZE)];
      if (w1 >= SAVE_NAME_SIZE) w1 = SAVE_NAME_SIZE - 1;
      memcpy(slotnames[i1], buffer + SAVE_NAME + (i1 * SLOTHEAD_SIZE) + 1, w1);
    }
    else {
      save_putword(SAVE_POSPOS + (i1 * SLOTHEAD_SIZE), 0);
    }
    p[SAVE_ACTUALLEVEL] ^= SAVE_XOR;
  }
  fclose(fsave);
  fsave = NULL;
}

/* Funkce pro zavreni save souboru s pripadnym ulozenim zmen. Vraci 0, pokud
 * se ukladani nepovedlo; jinak 1. */
int close_savefile()
{
  int lens[CNT_SLOT];
  int i1, ofs, w1;
  uchar *p;

  if (!savechanged) return 1;
  /* otevreme soubor */
  fsave = fopen(filename, "wb");
  if (!fsave) return 0;
  ofs = SAVEHEAD_SIZE;                  /* v promenne ofs udrzujeme ofset slotu */
  for (i1 = 0; i1 < CNT_SLOT; i1++) {
    if (emptyslot[i1]) {
      /* prazdny slot */
      lens[i1] = 0;
      memset(buffer + SAVE_NAME + i1 * SLOTHEAD_SIZE, 0, SLOTHEAD_SIZE);
      continue;
    }
    p = buffer + SAVEHEAD_SIZE + i1 * SLOT_MAXSIZE;
    /* zaxorujeme cislo kola */
    p[SAVE_ACTUALLEVEL] ^= SAVE_XOR;
    /* zjistime delku ukladanych dat */
    w1 = buf_getword(p, SAVE_THINGCNT);
    lens[i1] = w1 * 3 + SLOT_FIXSIZE + 1;
    /* a tuto delku a ofset poukladame */
    buf_putword(p, 0, lens[i1]);
    save_putword(SAVE_POSPOS + i1 * SLOTHEAD_SIZE, ofs);
    /* spocitame kontrolni soucet a ulozime ho */
    p[lens[i1] - 1] = checksum(p, lens[i1] - 1);
    ofs += lens[i1];
  }
  /* zapiseme hlavicku */
  if (fwrite(buffer, 1, SAVEHEAD_SIZE, fsave) != SAVEHEAD_SIZE) {
    fclose(fsave);
    return 0;
  }
  /* postupne zapiseme jednotlive sloty */
  for (i1 = 0; i1 < CNT_SLOT; i1++) {
    if (emptyslot[i1]) continue;
    if (fwrite(buffer + SAVEHEAD_SIZE + i1 * SLOT_MAXSIZE, 1, lens[i1], fsave) != lens[i1]) {
      fclose(fsave);
      return 0;
    }
  }
  /* zavreme soubor */
  fclose(fsave);
  return 1;
}

/* Funkce pro zavreni save souboru (s pripadnym ulozenim zmen), osetrujici
 * chyby. */
void process_close_savefile()
{
  if (!close_savefile())
    msg_box(ls(STR_SAVE_WRITE_ERROR));
}


/* Funkce, ktera smaze dany slot. */
void delete_slot(int no)
{
  if (emptyslot[no]) return;
  savechanged = 1;
  memset(buffer + SAVE_NAME + SLOTHEAD_SIZE * no, 0, SLOTHEAD_SIZE);
  emptyslot[no] = 1;
}


/* Funkce, ktera vykresli nahled. */
void draw_preview(int no)
{
  uchar *ldef, *buf, def;
  int x, y, i1, cnt;

  /* opet nemusime volat drawf_*(), ale staci nam draw_*() - o prekresleni
   * se postara obdelnik, vykresleny pred vykreslenim nahledu */
  ldef = malloc(XSIZE_LEVELDEF * YSIZE_LEVELDEF);
  if (!ldef) return;
  buf = buffer + SAVEHEAD_SIZE + no * SLOT_MAXSIZE;
  load_level(buf[SAVE_ACTUALLEVELSET] & 0x7f, buf[SAVE_ACTUALLEVEL], ldef, NULL);

  /* vykreslime pevne objekty */
  for (y = 0; y < YSIZE_LEVELDEF; y++)
    for (x = 0; x < XSIZE_LEVELDEF; x++) {
      def = ldef[y * XSIZE_LEVELDEF + x];
      switch (def & 7) {
        case OBJ_BRICK:
          draw_img(imgmbrick[(def >> 3) & 15], PREVIEW_X + x * 4, PREVIEW_Y + y * 4);
          break;
        case OBJ_UP:
          draw_img(imgmup, PREVIEW_X + x * 4, PREVIEW_Y + y * 4);
          break;
        case OBJ_RUN:
          draw_img(imgmrun, PREVIEW_X + x * 4, PREVIEW_Y + y * 4);
          break;
        default:
          break;
        }
      }
  /* vykreslime baggery */
  for (i1 = 0; i1 < CNT_BAGGER - 1; i1++) {
    x = buf_getword(buf, SAVE_THINGSTAT + 3 * i1) & 0x1ff;
    if (x != 0x1ff) {
      /* kreslime jen ty, kteri jsou vypusteni */
      y = buf[SAVE_THINGSTAT + 3 * i1 + 2];
      draw_img(buf[SAVE_LEVELBAGGER + (i1 + 1) * SIZEDEF_BAGGER] & 64 ? imgmbagg : imgmbaggr,
          PREVIEW_X + x / 2, PREVIEW_Y + y / 2);
    }
  }
  /* a pak zbytek */
  cnt = buf_getword(buf, SAVE_THINGCNT);
  for (i1 = CNT_BAGGER - 1; i1 < cnt; i1++)
  {
    x = buf_getword(buf, SAVE_THINGSTAT + 3 * i1);
    def = x >> 9;
    x &= 0x1ff;
    y = buf[SAVE_THINGSTAT + 3 * i1 + 2];
    switch (def & 7) {
      case OBJ_BOULD:
        draw_img(imgmbould[(def >> 3) & 15], PREVIEW_X + x / 2, PREVIEW_Y + y / 2);
        break;
      case OBJ_FRED1:
        draw_img((buf[SAVE_LEVELBAGGER + 0] & 64) ? imgmfred[0] : imgmfredr[0],
            PREVIEW_X + x / 2, PREVIEW_Y + y / 2);
        break;
      case OBJ_FRED2:
        draw_img((buf[SAVE_LEVELBAGGER + 1] & 64) ? imgmfred[1] : imgmfredr[1],
            PREVIEW_X + x / 2, PREVIEW_Y + y / 2);
        break;
      default:
        break;
    }
  }
  draw_rect(PREVIEW_X - 1, PREVIEW_Y - 1, XSIZE_LEVELDEF * 4 + 2, YSIZE_LEVELDEF * 4 + 2,
      0xaa, 0xaa, 0xaa);
  free(ldef);
}


/* Funkce, ktera zobrazi nabidku se savy. load urcuje, zda se nahrava nebo
 * uklada. Vraci cislo savu, ktere bylo vybrano, nebo -1 (Escape). */
int display_saves(int load)
{
  SDLKey key;
  int forceredraw;
  int i1;
  
  /* obrazovku ukladat nemusime, protoze jsme vzdy volani z menu, a to potom
   * obrazovku obnovi za nas */
  open_savefile();
  forceredraw = 1;
  while (1) {
    if (forceredraw) {
      draw_frame(0, 0, 640, 400);
      draw_frame(512, 0, 128, 64);
      if (load) draw_text(540, 16, "Load");
      else draw_text(540, 16, "Save");
      for (i1 = 0; i1 < CNT_SLOT; i1++) {
        if (emptyslot[i1]) strcpy(slotnames[i1], "      ---");
        draw_text(32, 16 + i1 * 48, slotnames[i1]);
      }
    }
    drawf_rect(24, 12 + savemenuitem * 48, 286, 40, 0xff, 0x55, 0x55);
    /* zobrazeni nahledu */
    drawf_blackbar(PREVIEW_X - 1, PREVIEW_Y - 1,
        XSIZE_LEVELDEF * 4 + 2, YSIZE_LEVELDEF * 4 + 2);
    if (!emptyslot[savemenuitem]) draw_preview(savemenuitem);
    if (forceredraw) {
      redraw();
      forceredraw = 0;
    }
    else redrawf();
    drawf_rect(24, 12 + savemenuitem * 48, 286, 40, 0, 0, 0);

    key = wait_for_key();
    switch (key) {
      case SDLK_UP:
        if (--savemenuitem < 0) savemenuitem = CNT_SLOT - 1;
        break;
      case SDLK_DOWN:
        if (++savemenuitem >= CNT_SLOT) savemenuitem = 0;
        break;
      case SDLK_RETURN:
        if (!load || !emptyslot[savemenuitem]) return savemenuitem;
        break;
      case SDLK_ESCAPE:
        return -1;
      case SDLK_DELETE: case SDLK_BACKSPACE: case SDLK_CLEAR:
        if (lastmod & KMOD_CTRL) {
          delete_slot(savemenuitem);
          forceredraw = 1;
        }
        break;
      default:
        break;
    }
  }
}


/* Funkce pro inicializaci. */
void init_save()
{
  /* vytvoreni nazvu save souboru */
  strcpy(filename, homedir);
  strcat(filename, savefilename);
  buffer = emalloc(SAVEHEAD_SIZE + CNT_SLOT * SLOT_MAXSIZE);
  savemenuitem = 0;
}


/* Funkce pro nahrani pozice. Vrati 1, pokud bylo nahrano nove kolo. */
int game_load()
{
  int slot;
  uchar *p;
  int i1;

  /* zobrazeni vybirace ulozenych pozic */
  slot = display_saves(1);
  if (slot < 0) {
    process_close_savefile();
    return 0;
  }
  /* natazeni ulozeneho kola */
  p = buffer + SAVEHEAD_SIZE + slot * SLOT_MAXSIZE;
  actuallevel = p[SAVE_ACTUALLEVEL];
  actuallevelset = buf_getword(p, SAVE_ACTUALLEVELSET);
  phaseactbagg = actuallevelset >> 7;
  actuallevelset &= 0x7f;
  phasebagg = p[SAVE_STATUS1];
  phasefred[0] = phasebagg & 1;
  phasefred[1] = (phasebagg >> 1) & 1;
  fredup[0] = (phasebagg >> 2) & 1;
  fredup[1] = (phasebagg >> 3) & 1;
  phaserun = (phasebagg >> 4) & 7;
  phasebagg >>= 7;
  gamespeed = p[SAVE_STATUS2] & 15;
  memcpy(levelbagger, p + SAVE_LEVELBAGGER, CNT_BAGGER * SIZEDEF_BAGGER);
  thingcnt = buf_getword(p, SAVE_THINGCNT);
  p += SAVE_THINGSTAT;
  for (i1 = 0; i1 < thingcnt; i1++) {
    thingstat[i1].x = buf_getword(p, 0) & 0x1ff;
    if (thingstat[i1].x == 0x1ff) thingstat[i1].x = -1;
    p++;
    thingstat[i1].c = (*p++ >> 1) | 0x80;
    thingstat[i1].y = *p++;
  }
  /* zavreni save souboru */
  process_close_savefile();
  /* nacteni definice kola a zobrazeni */
  load_level(actuallevelset, actuallevel, leveldef, NULL);
  draw_level(1);
  return 1;
}


/* Funkce pro ulozeni pozice. */
void game_save()
{
  int slot, tmp, tmp2;
  int i1;
  char name[SAVE_NAME_SIZE];
  uchar *p;

  /* zobrazeni vybirace ulozenych pozic */
  slot = display_saves(0);
  if (slot < 0) {
    process_close_savefile();
    return;
  }
#ifdef maemo
  {
    time_t t;
    time(&t);
    strftime(name, SAVE_NAME_SIZE, ls(STR_SAVE_DATE_FORMAT), gmtime(&t));
  }
#else
  /* zadani jmena slotu */
  if (emptyslot[slot]) *name = '\0';
  else strcpy(name, slotnames[slot]);
  if (!input_text(32, 16 + slot * 48, SAVE_NAME_SIZE - 1, ' ', name, 0, 1)) {
    process_close_savefile();
    return;
  }
#endif
  savechanged = 1;
  /* ulozime zadane jmeno do bufferu */
  tmp = strlen(name);
  memset(buffer + SAVE_NAME + slot * SLOTHEAD_SIZE, 0, SAVE_NAME_SIZE);
  memcpy(buffer + SAVE_NAME + slot * SLOTHEAD_SIZE + 1, name, tmp);
  buffer[SAVE_NAME + slot * SLOTHEAD_SIZE] = tmp;
  /* a ulozime pozici */
  emptyslot[slot] = 0;
  p = buffer + SAVEHEAD_SIZE + slot * SLOT_MAXSIZE;
  p[SAVE_ACTUALLEVEL] = actuallevel;
  tmp = (actuallevelset & 0x7f) | ((phaseactbagg != 0) << 7);
  buf_putword(p, SAVE_ACTUALLEVELSET, tmp);
  tmp = (phasefred[0] != 0) | ((phasefred[1] != 0) << 1)
      | ((fredup[0] != 0) << 2) | ((fredup[1] != 0) << 3)
      | ((phaserun & 7) << 4) | ((phasebagg != 0) << 7);
  p[SAVE_STATUS1] = tmp;
  p[SAVE_STATUS2] = gamespeed & 15;
  memcpy(p + SAVE_LEVELBAGGER, levelbagger, CNT_BAGGER * SIZEDEF_BAGGER);
  buf_putword(p, SAVE_THINGCNT, thingcnt);
  p += SAVE_THINGSTAT;
  for (i1 = 0; i1 < thingcnt; i1++) {
    tmp = thingstat[i1].x & 0x1ff;
    tmp2 = thingstat[i1].y;
    tmp |= ((levelstat[tmp2 * XSIZE_LEVELSTAT + tmp] & 0x7f) << 9);
    buf_putword(p, 0, tmp);
    p += 2;
    *p++ = tmp2;
  }
  process_close_savefile();
}

