/*
 *  Copyright (c) 2002-2007 Jiri Benc <jbenc@upir.cz>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * level.c
 *
 * Funkce pro praci s koly.
 *
 */

#include "SDL.h"
#include <string.h>
#include "common.h"
#include "graphics.h"
#include "keys.h"
#include "params.h"
#include "level.h"
#include "demo.h"
#include "readdata.h"

#ifdef DEMOSAVE
#include "demosave.h"
#endif


uchar *leveldef;                        /* definice kola */
uchar *levelbagger;                     /* definice baggeru v kole */
                                        /* (nulta polozka urcuje Fredy) */
uchar *levelstat;                       /* aktualni stav kola */
thing_stat *thingstat;                  /* stavy pohyblivych veci v kole */
int thingcnt;                           /* aktualni velikost pole thingstat */

int actuallevelset, actuallevel;        /* aktualni (prave probihajici) sada a kolo */
int actualmaxlevel;                     /* pocet kol v aktualni sade */

int demoactive;                         /* priznak, ze je aktivni demo */
int levelfinished;                      /* priznak, ze je konec kola */
int levelrestart;                       /* priznak, ze se ma hrat kolo zase od zacatku */
int gamerestart;                        /* priznak, ze se ma vratit do hlavniho menu */

unsigned thcounter;                     /* citac zpracovani veci */

/* faze animace objektu: */
int phaserun;
int phasefred[2];
  /* animacni faze baggera a faze aktivity/neaktivity baggera (bagger se
   * pohybuje 2x pomaleji nez hrac) */
int phasebagg, phaseactbagg;

int fredup[2];                          /* priznak, ze Fred stoupa */

/* rychlost hry: */
int gamespeed;                          /* stupen rychlosti hry */

#define STEP_SPEED      20              /* pocet milisekund na jeden stupen rychlosti */


/* Inicializace datovych struktur pro kola. */
void init_levels()
{
  leveldef = emalloc(XSIZE_LEVELDEF * YSIZE_LEVELDEF);
  levelbagger = emalloc(CNT_BAGGER * SIZEDEF_BAGGER);
  levelstat = emalloc(XSIZE_LEVELSTAT * YSIZE_LEVELSTAT);
  thingstat = emalloc(sizeof(thing_stat) * MAXSIZE_THINGSTAT);
}


/* Nektera uzitecna makra pro praci s hernimi strukturami */
#define levelstat_yx(y, x)      levelstat[(y) * XSIZE_LEVELSTAT + x]
#define levelbagger_n(no, n)    levelbagger[(no) * SIZEDEF_BAGGER + (n)]
#define levelbagger_xfrom(no)   ((levelbagger[(no) * SIZEDEF_BAGGER] & 63) * 8)
#define levelbagger_yfrom(no)   ((levelbagger[(no) * SIZEDEF_BAGGER + 1] & 63) * 8)
#define levelbagger_xto(no)     ((levelbagger[(no) * SIZEDEF_BAGGER + 2] & 63) * 8)


/* Inicializace a vykresleni kola. Je-li afterload != 0, znamena to, ze kolo
 * bylo prave loadnuto a bude inicializovano trochu jinak. */
void draw_level(int afterload)
{
  uchar *posdef, def;
  int x, y, i1, i2, ii;
  char text[32];

  /* inicializace */
  memset(levelstat, 0, XSIZE_LEVELSTAT * YSIZE_LEVELSTAT);
  if (!afterload) {
    memset(thingstat, 0, sizeof(thing_stat) * MAXSIZE_THINGSTAT);
    for (i1 = 0; i1 < CNT_BAGGER - 1; i1++) thingstat[i1].x = -1;
    thingcnt = CNT_BAGGER - 1;
    phaserun = 0;
    phasefred[0] = phasefred[1] = 0;
    phasebagg = phaseactbagg = 0;
    fredup[0] = fredup[1] = 0;
  }
  thcounter = 0;
  levelfinished = levelrestart = 0;

  /* vykresleni */
  if (!demoactive)
    draw_bar(0, 0, 640, 400, 0, 0, 0);
  else draw_bar(0, 48 * 2, 224 * 2, 128 * 2, 0, 0, 0);
  if (!afterload) {
    /* v pripade noveho kola bereme veci z leveldefu */
    posdef = leveldef;
    for (y = 0; y < YSIZE_LEVELDEF; y++)
      for (x = 0; x < XSIZE_LEVELDEF; x++) {
        def = *posdef++;
        switch (def & 7) {
          case OBJ_BRICK:
            draw_img(imgbrick[(def >> 3) & 15], x * 16, y * 16);
            break;
          case OBJ_UP:
            draw_img(imgup[0], x * 16, y * 16);
            break;
          case OBJ_RUN:
            draw_img(imgrun[0], x * 16, y * 16);
            break;
          case OBJ_BOULD:
            draw_img(imgbould[(def >> 3) & 15], x * 16, y * 16);
            break;
          case OBJ_FRED1:
            draw_img((levelbagger[0] & 64) ? imgfred[0] : imgfredr[0], x * 16, y * 16);
            break;
          case OBJ_FRED2:
            draw_img((levelbagger[1] & 64) ? imgfred[2] : imgfredr[2], x * 16, y * 16);
            break;
        }
        switch (def & 7) {
          case OBJ_BRICK:
            for (i2 = 0; i2 < 8; i2++)
              for (i1 = 0; i1 < 8; i1++)
                levelstat_yx(y * 8 + i2, x * 8 + i1) = def;
            break;
          case OBJ_UP: case OBJ_RUN:
            for (i2 = 0; i2 < 8; i2++)
              for (i1 = 0; i1 < 16; i1++)
                levelstat_yx(y * 8 + i2, x * 8 + i1) = def & 127;
            levelstat_yx(y * 8, x * 8) = def;
            thingstat[thingcnt].x = x * 8;
            thingstat[thingcnt].y = y * 8;
            thingstat[thingcnt].c = 0;
            thingcnt++;
            break;
          case OBJ_BOULD: case OBJ_FRED1: case OBJ_FRED2:
            for (i2 = 0; i2 < 16; i2++)
              for (i1 = 0; i1 < 16; i1++)
                levelstat_yx(y * 8 + i2, x * 8 + i1) = def & 127;
            levelstat_yx(y * 8, x * 8) = def;
            thingstat[thingcnt].x = x * 8;
            thingstat[thingcnt].y = y * 8;
            thingstat[thingcnt].c = 0;
            thingcnt++;
            break;
        }
      }
  }
  else {
    /* v pripade loadu bereme z leveldefu jen zdi, vsechno ostatni jde z
     * thingstatu (v thingstat->c je byte, ktery patri do levelstatu) */
    posdef = leveldef;
    for (y = 0; y < YSIZE_LEVELDEF; y++)
      for (x = 0; x < XSIZE_LEVELDEF; x++) {
        def = *posdef++;
        if ((def & 7) == OBJ_BRICK) {
          draw_img(imgbrick[(def >> 3) & 15], x * 16, y * 16);
          for (i2 = 0; i2 < 8; i2++)
            for (i1 = 0; i1 < 8; i1++)
              levelstat_yx(y * 8 + i2, x * 8 + i1) = def;
        }
      }
    /* nejdrive vykreslime baggery */
    for (ii = 0; ii < CNT_BAGGER - 1; ii++)
      if (thingstat[ii].x >= 0) {
        /* kreslime jen ty, kteri jsou vypusteni */
        x = thingstat[ii].x;
        y = thingstat[ii].y;
        def = thingstat[ii].c;
        /* spocitame si, jaka byla jejich posledni faze (ciste kvuli
         * estetice - u ostatnich veci je tento problem samozrejme taky (ve
         * skutecnosti je prekreslujeme se spatnou fazi), ale tam to neni
         * tak videt) */
        i1 = phasebagg;
        if (!phaseactbagg) if (--i1 < 0) i1 = 1;
        draw_img((levelbagger_n(ii + 1, 0) & 64) ? imgbagg[i1] : imgbaggr[i1],
            x * 2, y * 2);
        for (i2 = 0; i2 < 16; i2++)
          for (i1 = 0; i1 < 16; i1++)
            levelstat_yx(y + i2, x + i1) = def & 127;
        levelstat_yx(y, x) = def;
      }
    /* a pak zbytek */
    for (ii = CNT_BAGGER - 1; ii < thingcnt; ii++)
    {
      x = thingstat[ii].x;
      y = thingstat[ii].y;
      def = thingstat[ii].c;
      switch (def & 7) {
        case OBJ_UP:
          draw_img(imgup[(phaserun >= 1 && phaserun <= 3) ? 0 : 1], x * 2, y * 2);
          break;
        case OBJ_RUN:
          draw_img(imgrun[(phaserun >=2 && phaserun <=4) ? 1 : 0], x * 2, y * 2);
          break;
        case OBJ_BOULD:
          draw_img(imgbould[(def >> 3) & 15], x * 2, y * 2);
          break;
        case OBJ_FRED1:
          draw_img((levelbagger[0] & 64) ? imgfred[phasefred[0]] : imgfredr[phasefred[0]],
              x * 2, y * 2);
          break;
        case OBJ_FRED2:
          draw_img((levelbagger[1] & 64) ? imgfred[2 + phasefred[1]] : imgfredr[2 + phasefred[1]],
              x * 2, y * 2);
          break;
      }
      switch (def & 7) {
        case OBJ_UP: case OBJ_RUN:
          for (i2 = 0; i2 < 8; i2++)
            for (i1 = 0; i1 < 16; i1++)
              levelstat_yx(y + i2, x + i1) = def & 127;
          levelstat_yx(y, x) = def;
          break;
        case OBJ_BOULD: case OBJ_FRED1: case OBJ_FRED2:
          for (i2 = 0; i2 < 16; i2++)
            for (i1 = 0; i1 < 16; i1++)
              levelstat_yx(y + i2, x + i1) = def & 127;
          levelstat_yx(y, x) = def;
          break;
      }
    }
    /* nakonec vynulujeme citace zpracovani */
    for (i1 = 0; i1 < thingcnt; i1++) thingstat[i1].c = 0;
  }
  if (!demoactive) {
    /* napiseme text - cislo kola, atd. */
    if (!param_userlevel) {
      sprintf(text, ls(STR_LEVEL_LEVEL), actuallevel);
      draw_text(0, 180 * 2, text);
      strcpy(text, ls(STR_LEVEL_PASSWORD));
      get_level_password(actuallevelset, actuallevel, text + 7);
      draw_text(298, 180 * 2, text);
    }
    else draw_text(0, 180 * 2, ls(STR_LEVEL_USER_LEVEL));
  }
  redraw();
}


/* Nalezeni veci v poli thingstat, ktera ma dane souradnice (krome baggeru). */
int find_thing(int x, int y)
{
  int i1;

  for (i1 = CNT_BAGGER - 1; i1 < thingcnt; i1++)
    if (thingstat[i1].x == x && thingstat[i1].y == y) return i1;
  error(ls(STR_LEVEL_ERR_INCONSIST));
  return -1;                            /* nikdy nenastane */
}


/* Posunuti kamene - funkce vola sama sebe rekurzivne, je-li treba posunout
 * i dalsi kameny.
 * Vyznam parametru:
 *   x, y  - souradnice balvanu (levy horni roh), ktery se ma posunout
 *   flags - priznaky:
 *           bit 0: left (je-li nastaven, posunuje se doleva)
 *           bit 1: first (dotycny balvan je koren prohledavaciho stromu -
 *                  tj. jde o prvni volani teto funkce)
 *           bit 2: move (je-li nastaven, je balvan skutecne posunut; jinak
 *                  je pouze zjisteno, zda posunout lze)
 * Vraci:
 *   0, pokud balvan nelze posunout
 *   1, pokud lze
 *
 * Poznamka: Pri prvnim volani je vyuzita finta, kdy se jako balvan tvari
 *           Fred ci bagger a funkce se vola s nastavenym priznakem "first"
 *           a nenastavenym "move".
 */
int boulder_move(int x, int y, int flags)
{
  /* Algoritmus:
   * Zjistime, ktere balvany s nami sousedi vpravo, a pro ty se rekurzivne
   * zavolame. To vse nejdrive v "overovaci" fazi (nenastaveny priznak
   * "move"), kdy se jen testuje moznost posunu. Jestlize se vratime v
   * poradku az do korene (ten sam sebe pozna podle nastaveneho priznaku
   * "first") a vsechno projde, pak teprve jedeme "naostro" - tj. projdeme
   * vsechno znovu (tentokrat s nastavenym "move") a posouvame.
   */
  int i1, lstat;

  if (!(flags & 1)) {
    /* posunujeme doprava */
    /* pokud jsme uplne vpravo, tak koncime neuspesne */
    if (x + 16 >= XSIZE_LEVELSTAT) return 0;
    /* prohledame misto vpravo od balvanu (odspodu) */
    for (i1 = 15; i1 >= 0; i1--) {
      lstat = levelstat_yx(y + i1, x + 16);
      if ((lstat & 7) == OBJ_BOULD) {
        /* narazili jsme na balvan */
        if (lstat & 128) {
          /* a je to levy horni roh balvanu - zkusime ho posunout */
          /* pri volani z priznaku zachovame jen "move", ostatni vynulujeme */
          if (!boulder_move(x + 16, y + i1, flags & 4)) return 0;
        }
      }
      else if (lstat & 7) return 0;     /* cokoliv jineho znamena neuspech */
    }
    /* jeste se podivame, jestli nejaky balvan nezasahuje zvrchu (tj.
     * nezacina nekde drive) */
    lstat = levelstat_yx(y, x + 16);
    if ((lstat & 7) == OBJ_BOULD && !(lstat & 128)) {
      /* zasahuje */
      for (i1 = y - 1; !(levelstat_yx(i1, x + 16) & 128); i1--) ;
      if (!boulder_move(x + 16, i1, flags & 4)) return 0;
    }

    if ((flags & 4) && !(flags & 2)) {
      /* balvan se ma posunout */
      levelstat_yx(y, x + 1) = levelstat_yx(y, x);
      for (i1 = 0; i1 < 16; i1++) {
        levelstat_yx(y + i1, x) = 0;
        levelstat_yx(y + i1, x + 16) = levelstat_yx(y + i1, x + 15);
      }
      redrawfhoriz_img(imgbould[(levelstat_yx(y, x + 1) >> 3) & 15],
          (x + 1) * 2, y * 2, x * 2);
      thingstat[find_thing(x, y)].x = x + 1;
    }
  }
  else {
    /* posunujeme doleva */
    /* pokud jsme uplne vlevo, tak koncime neuspesne */
    if (x <= 0) return 0;
    /* prohledame misto vlevo od balvanu (odspodu) */
    for (i1 = 15; i1 >= 0; i1--) {
      lstat = levelstat_yx(y + i1, x - 1);
      if ((lstat & 7) == OBJ_BOULD) {
        /* narazili jsme na balvan */
        if (levelstat_yx(y + i1, x - 16) & 128) {
          /* a odpovida levemu hornimu rohu balvanu - zkusime ho posunout */
          /* pri volani z priznaku zachovame jen "move" a nastavime "left" */
          if (!boulder_move(x - 16, y + i1, (flags & 4) | 1)) return 0;
        }
      }
      else if (lstat & 7) return 0;     /* cokoliv jineho znamena neuspech */
    }
    /* jeste se podivame, jestli nejaky balvan nezasahuje zvrchu (tj.
     * nezacina nekde drive) */
    if ((levelstat_yx(y, x - 1) & 7) == OBJ_BOULD && !(levelstat_yx(y, x - 16) & 128)) {
      /* zasahuje */
      for (i1 = y - 1; !(levelstat_yx(i1, x - 16) & 128); i1--) ;
      if (!boulder_move(x - 16, i1, (flags & 4) | 1)) return 0;
    }

    if ((flags & 4) && !(flags & 2)) {
      /* balvan se ma posunout */
      for (i1 = 0; i1 < 16; i1++) {
        levelstat_yx(y + i1, x - 1) = levelstat_yx(y + i1, x);
        levelstat_yx(y + i1, x + 15) = 0;
      }
      levelstat_yx(y, x) = levelstat_yx(y, x + 1);
      redrawfhoriz_img(imgbould[(levelstat_yx(y, x) >> 3) & 15],
          (x - 1) * 2, y * 2, x * 2);
      thingstat[find_thing(x, y)].x = x - 1;
    }
  }
  if ((flags & 2) && !(flags & 4)) {
    /* nastaveny priznak "first" a nenastaveny "move" */
    boulder_move(x, y, (flags & 1) | 6);
    /* Zde se vyuziva toho, ze pri nastavenem priznaku "first" se dotycny
     * balvan neposune, ani kdyz je nastaveny "move". Proto se muzeme klidne
     * zavolat znovu (potreti se uz nezavolame, nebot bude nastaveny
     * "move").
     */
  }
  return 1;
}


/* Vypusteni baggera. */
void bagger_start(int number)
{
  int x, y;
  int i1, i2;
  uchar lstat;

  /* oznac prislusneho baggera jako zpracovaneho */
  levelbagger_n(number, 1) |= 128;
  /* ulozime si, odkud ma bagger vyjit */
  x = levelbagger_xfrom(number);
  y = levelbagger_yfrom(number);
  /* zjistime, jestli je tam misto */
  for (i1 = 0; i1 < 16; i1++) {
    for (i2 = 0; i2 < 16; i2++)
      if (levelstat_yx(y + i1, x + i2) & 7) break;
    if (i2 != 16) break;
  }
  if (i1 == 16) {
    /* misto je, takze zobrazime baggera */
    lstat = (number << 3) | OBJ_BAGG;
    for (i1 = 0; i1 < 16; i1++)
      for (i2 = 0; i2 < 16; i2++)
        levelstat_yx(y + i1, x + i2) = lstat;
    levelstat_yx(y, x) = lstat | 128;
    /* oznacime ho jako zobrazeneho */
    levelbagger_n(number, 0) |= 128;
    /* zjistime, na kterou stranu ma byt otoceny (podle mista, kam jde) */
    if (x < levelbagger_xto(number)) {
      /* vpravo */
      levelbagger_n(number, 0) |= 64;
      drawf_img(imgbagg[phasebagg], x * 2, y * 2);
    }
    else {
      /* vlevo */
      levelbagger_n(number, 0) &= ~64;
      drawf_img(imgbaggr[phasebagg], x * 2, y * 2);
    }
    /* nastavime pozici baggera v poli veci */
    thingstat[number - 1].x = x;
    thingstat[number - 1].y = y;
  }
}


/* Posun baggera. */
void bagger_move(int number, int x, int y)
{
  int i1, i2;

  /* oznac prislusneho baggera jako zpracovaneho */
  levelbagger_n(number, 1) |= 128;

  if (x == levelbagger_xto(number)) {
    /* jsme uz na konci */
    for (i1 = 0; i1 < 16; i1++)
      for (i2 = 0; i2 < 16; i2++)
        levelstat_yx(y + i1, x + i2) = 0;
    drawf_blackbar(x * 2, y * 2, 16 * 2, 16 * 2);
    levelbagger_n(number, 0) &= ~128;
    thingstat[number - 1].x = -1;
  }
  else if (levelbagger_n(number, 0) & 64) {
    /* jsme obraceni doprava */

    /* nejdrive zjistime, zda vubec doprava muzeme */
    i2 = 0;
    for (i1 = 0; i1 < 16; i1++) {
      if ((levelstat_yx(y + i1, x + 16) & 7) == OBJ_BOULD)
        i2++;
      else if (levelstat_yx(y + i1, x + 16) & 7) break;
    }
    if (i2 && i1 == 16) {
      /* cesta je volna, az na balvany - zkusime je posunout */
      if (!boulder_move(x, y, 2)) i1 = 0;
    }
    if (i1 == 16) {
      /* cesta je volna */
      redrawfhoriz_img(imgbagg[phasebagg], (x + 1) * 2, y * 2, x * 2);
      levelstat_yx(y, x + 1) = levelstat_yx(y, x);
      for (i1 = 0; i1 < 16; i1++) {
        levelstat_yx(y + i1, x) = 0;
        levelstat_yx(y + i1, x + 16) = levelstat_yx(y + i1, x + 15);
      }
      thingstat[number - 1].x = x + 1;
      thingstat[number - 1].y = y;
    }
    else drawf_img(imgbagg[phasebagg], x * 2, y * 2);
  }
  else {
    /* jsme obraceni doleva */

    /* nejdrive zjistime, zda vubec doleva muzeme */
    i2 = 0;
    for (i1 = 0; i1 < 16; i1++) {
      if ((levelstat_yx(y + i1, x - 1) & 7) == OBJ_BOULD)
        i2++;
      else if (levelstat_yx(y + i1, x - 1) & 7) break;
    }
    if (i2 && i1 == 16) {
      /* cesta je volna, az na balvany - zkusime je posunout */
      if (!boulder_move(x, y, 3)) i1 = 0;
    }
    if (i1 == 16) {
      /* cesta je volna */
      redrawfhoriz_img(imgbaggr[phasebagg], (x - 1) * 2, y * 2, x * 2);
      for (i1 = 0; i1 < 16; i1++) {
        levelstat_yx(y + i1, x - 1) = levelstat_yx(y + i1, x);
        levelstat_yx(y + i1, x + 15) = 0;
      }
      levelstat_yx(y, x) = levelstat_yx(y, x + 1);
      thingstat[number - 1].x = x - 1;
      thingstat[number - 1].y = y;
    }
    else drawf_img(imgbaggr[phasebagg], x * 2, y * 2);
  }
}


/* Posun Freda nahoru. */
void fred_move_up(int fredno, int x, int *y, thing_stat *th)
{
  int i1;

  /* jestlize jsme uplne nahore, zacneme zase padat */
  if (*y < 1) {
    fredup[fredno] = 0;
    return;
  }

  /* podivame se, jestli nad sebou nemame zed */
  for (i1 = 0; i1 < 16; i1++)
    if (levelstat_yx(*y - 1, x + i1) & 7) break;
  if (i1 == 16) {
    /* nemame, muzeme se posunout nahoru */
    if (levelbagger_n(0, fredno) & 64)
      redrawfvert_img(imgfred[fredno * 2], x * 2, (*y - 1) * 2, *y * 2);
    else redrawfvert_img(imgfredr[fredno * 2], x * 2, (*y - 1) * 2, *y * 2);
    for (i1 = 0; i1 < 16; i1++) {
      levelstat_yx(*y - 1, x + i1) = levelstat_yx(*y, x + i1);
      levelstat_yx(*y + 15, x + i1) = 0;
    }
    levelstat_yx(*y, x) = levelstat_yx(*y + 1, x);
    th->y = *y - 1;
    (*y)--;
  }
  else fredup[fredno] = 0;
}


/* Posun Freda do boku, kdyz stoupa (tj. "vystoupeni"). */
void fred_move_up_off(int fredno, int x, int y, thing_stat *th, int dir)
{
  int i1;

  if (levelbagger_n(0, fredno) & 64) {
    /* Fred je otoceny doprava */
    if (dir == 2) {
      /* ale klavesa byla doleva, takze Freda otocime vlevo */
      levelbagger_n(0, fredno) &= ~64;
      drawf_img(imgfredr[fredno * 2], x * 2, y * 2);
      phasefred[fredno] = 0;
    }
    else {
      /* byla klavesa doprava */
      if (x + 16 >= XSIZE_LEVELSTAT ||
          !(levelstat_yx(y + 16, x + 16) & 7))
        /* jestlize jsme uplne vpravo nebo nemame na co vystoupit, vratime
         * se */
        return;
      /* podivame se, jestli mame vedle sebe volno */
      for (i1 = 0; i1 < 16; i1++)
        if (levelstat_yx(y + i1, x + 16) & 7) break;
      if (i1 == 16) {
        /* mame */
        redrawfhoriz_img(imgfred[fredno * 2], (x + 1) * 2, y * 2, x * 2);
        levelstat_yx(y, x + 1) = levelstat_yx(y, x);
        for (i1 = 0; i1 < 16; i1++) {
          levelstat_yx(y + i1, x) = 0;
          levelstat_yx(y + i1, x + 16) = levelstat_yx(y + i1, x + 15);
        }
        th->x = x + 1;
        fredup[fredno] = 0;
      }
    }
  }
  else {
    /* Fred je otoceny doleva */
    if (dir == 1) {
      /* ale klavesa byla doprava, takze Freda otocime vprava */
      levelbagger_n(0, fredno) |= 64;
      drawf_img(imgfred[fredno * 2], x * 2, y * 2);
      phasefred[fredno] = 0;
    }
    else {
      /* byla klavesa doleva */
      if (x <= 0 ||
          !(levelstat_yx(y + 16, x - 1) & 7))
        /* jestlize jsme uplne vlevo nebo nemame na co vystoupit, vratime
         * se */
        return;
      /* podivame se, jestli mame vedle sebe volno */
      for (i1 = 0; i1 < 16; i1++)
        if (levelstat_yx(y + i1, x - 1) & 7) break;
      if (i1 == 16) {
        /* mame */
        redrawfhoriz_img(imgfredr[fredno * 2], (x - 1) * 2, y * 2, x * 2);
        for (i1 = 0; i1 < 16; i1++) {
          levelstat_yx(y + i1, x - 1) = levelstat_yx(y + i1, x);
          levelstat_yx(y + i1, x + 15) = 0;
        }
        levelstat_yx(y, x) = levelstat_yx(y + 1, x);
        th->x = x - 1;
        fredup[fredno] = 0;
      }
    }
  }
}


/* Provedeni jednoho kroku hry. */
void step_level()
{
  Uint32 timer, now;
  int dir[2];
  int i_things, i1, y, fredno, bouldcnt;
  thing_stat *th, *found;
  int maxy, minx;
  uchar lstat;

  SDL_PumpEvents();                     /* zpracovani udalosti SDL */
  timer = SDL_GetTicks();               /* ulozeni pocatecniho poctu tiku */
  dir[0] = dir[1] = 0;
  if (!demoactive) {                    /* zjisteni smeru posunu Fredu */
    if (keyboard[keyscfg[2]]) dir[0] = 3;
    if (keyboard[keyscfg[3]]) dir[0] = 4;
    if (keyboard[keyscfg[1]]) dir[0] = 1;
    if (keyboard[keyscfg[0]]) dir[0] = 2;

    if (keyboard[keyscfg[6]]) dir[1] = 3;
    if (keyboard[keyscfg[7]]) dir[1] = 4;
    if (keyboard[keyscfg[5]]) dir[1] = 1;
    if (keyboard[keyscfg[4]]) dir[1] = 2;
  }
  else {
    /* nacteme klavesu z demosouboru */
    dir[0] = demo_getkey();
    if (dir[0] < 0) {
      /* demo skoncilo */
      levelfinished = 1;
      return;
    }
  }
#ifdef DEMOSAVE
  if (!dir[0]) return;
  store_demosave(dir[0]);
#endif
  thcounter++;
  fredno = 0;
  /* Algoritmus:
   * Predmety se zpracovavaji tak, jak jsou na obrazovce (tj. ne v poli
   * thingstat) odzdola nahoru a zleva doprava. Predmetu, ktery byl uz
   * zpracovan, se zvetsi hodnota promenne thingstat->c - tim ho pozname v
   * pristim pruchodu. Pole prochazime tak dlouho, dokud je jeste co
   * zpracovavat.
   */
  found = NULL;                 /* aby nervalo GCC */
  for (i_things = 0; i_things < thingcnt; i_things++) {
    /* nalezneme prvek nejvice vlevo dole, ktery jsme jeste nezpracovali */
    maxy = 0;
    minx = 1000;
    for (i1 = 0, th = thingstat; i1 < thingcnt; i1++, th++) {
      if (th->c == thcounter) continue;
      if (th->x < 0) {
        /* pokud je to nezobrazeny bagger, ukoncime tento pruchod */
        th->c++;                /* uz je zpracovany */
        i1 = -1;
        break;
      }
      if (th->y > maxy || (th->y == maxy && th->x < minx)) {
        minx = th->x;
        maxy = th->y;
        found = th;
      }
    }
    if (i1 < 0) continue;       /* nalezen nezobrazeny bagger -> dalsi iterace */
    found->c++;                 /* oznacime objekt jako zpracovany */
    lstat = levelstat_yx(maxy, minx);
    switch (lstat & 7) {
      case OBJ_UP:
        /* prekresleni vytahu, pokud se zmenila faze */
        if (phaserun == 4) drawf_img(imgup[1], minx * 2, maxy * 2);
        else if (phaserun == 1) drawf_img(imgup[0], minx * 2, maxy * 2);
        break;
      case OBJ_RUN:
        /* zjistime, jestli na nas nekdo (nebo neco) nestoji */
        for (i1 = 0; i1 < 16; i1++)
          if (levelstat_yx(maxy - 1, minx + i1) & 7)
            /* nekdo na nas stoji */
            break;
        if (i1 == 16) {
          /* nikdo na nas nestoji */
          if (phaserun >= 2 && phaserun <= 4)
            drawf_img(imgrun[1], minx * 2, maxy * 2);
          else drawf_img(imgrun[0], minx * 2, maxy * 2);
        }
        else {
          /* nekdo na nas stoji */
          if (phaserun >= 2 && phaserun <= 4)
            drawf_img(imgrun[3], minx * 2, maxy * 2);
          else drawf_img(imgrun[2], minx * 2, maxy * 2);
          y = (lstat >> 3) & 15;        /* cislo baggera, ktereho vypoustime */
          /* pokud uz bagger neni vypusten (a jeste nebyl obslouzen),
           * vypustime ho */
          if (!(levelbagger_n(y, 0) & 128) && !(levelbagger_n(y, 1) & 128))
            bagger_start(y);
        }
        break;
      case OBJ_BOULD:
        /* balvanem se nema smysl zabyvat, je-li uplne dole (pak uz nema kam
         * padat) */
        if (maxy + 16 < YSIZE_LEVELSTAT) {
          /* zjisti, zda mame pod sebou misto */
          for (i1 = 0; i1 < 16; i1++)
            if (levelstat_yx(maxy + 16, minx + i1) & 7)
              /* nemame */
              break;
          if (i1 == 16) {
            /* mame pod sebou misto, takze se posuneme o jedno dolu */
            redrawfvert_img(imgbould[(lstat >> 3) & 15], minx * 2, (maxy + 1) * 2,
              maxy * 2);
            levelstat[(maxy + 1) * XSIZE_LEVELSTAT + minx] = lstat;
            for (i1 = 0; i1 < 16; i1++) {
              levelstat_yx(maxy, minx + i1) = 0;
              levelstat_yx(maxy + 16, minx + i1) = levelstat_yx(maxy + 15, minx + i1);
            }
            found->y++;
          }
        }
        break;
      case OBJ_BAGG:
        if (phaseactbagg) {
          /* bagger se pohybuje jen jednou za dve kola */
          y = (lstat >> 3) & 15;
          if (!(levelbagger_n(y, 1) & 128))
            bagger_move(y, minx, maxy);
        }
        break;
      case OBJ_FRED2:
        fredno = 1;
        /* break tu neni schvalne */
      case OBJ_FRED1:
        if (dir[fredno] == 4)           /* klavesa "dolu" */
          fredup[fredno] = 0;
        y = maxy;
        if (fredup[fredno])             /* Fred prave stoupa */
          fred_move_up(fredno, minx, &y, found);

        if (maxy + 16 < YSIZE_LEVELSTAT && !fredup[fredno]) {
          /* podivame se, jestli nejsme nad prazdnem */
          for (i1 = 0; i1 < 16; i1++)
            if (levelstat_yx(maxy + 16, minx + i1) & 7) break;
          if (i1 == 16) {
            /* jsme nad prazdnem, tedy budeme padat */
            if (levelbagger_n(0, fredno) & 64)
              redrawfvert_img(imgfred[fredno * 2], minx * 2, (maxy + 1) * 2,
                  maxy * 2);
            else redrawfvert_img(imgfredr[fredno * 2], minx * 2, (maxy + 1) * 2,
                maxy * 2);
            if (!dir[fredno] || dir[fredno] > 2)
              phasefred[fredno] = 0;
            levelstat_yx(maxy + 1, minx) = lstat;
            y++;
            for (i1 = 0; i1 < 16; i1++) {
              levelstat_yx(maxy, minx + i1) = 0;
              levelstat_yx(maxy + 16, minx + i1) = levelstat_yx(maxy + 15, minx + i1);
            }
            found->y++;
          }
        }

        if (dir[fredno]) {
          if (fredup[fredno]) {
            if (dir[fredno] != 3)
              /* Fred stoupa a byla stisknuta klavesa "vlevo" nebo "vpravo"
               * ("dolu" to byt nemohlo, protoze uz drive jsme v takovem
               * pripade nastavili fredup[fredno] = 0) */
              fred_move_up_off(fredno, minx, y, found, dir[fredno]);
          }
          else {
            /* Fred nestoupa a byla stisknuta klavesa */
            i1 = 0;
            if (y != maxy && y + 16 < YSIZE_LEVELSTAT) {
              /* jestlize jsme klesli, podivame se, jestli uz mame pod sebou
               * pevnou plochu */
              for (; i1 < 16; i1++)
                if (levelstat_yx(y + 16, minx + i1) & 7) break;
            }
            /* jestlize jsme neklesli, tak uz vime, ze tam tu pevnou plochu
             * mame */
            if (i1 < 16)                /* mame pod sebou pevnou plochu */
              switch (dir[fredno]) {
                case 1: /* vpravo */
                  if (!(levelbagger_n(0, fredno) & 64)) {
                    /* Fred je otoceny vlevo, takze ho otocime doprava */
                    levelbagger_n(0, fredno) |= 64;
                    drawf_img(imgfred[fredno * 2], minx * 2, y * 2);
                    phasefred[fredno] = 0;
                  }
                  else {
                    /* Fred je otoceny vpravo, takze muzeme pochodovat */
                    if (minx + 16 < XSIZE_LEVELSTAT) {
                      /* jeste nejsme uplne vpravo */
                      /* nejdrive zjistime, zda vubec doprava muzeme */
                      bouldcnt = 0;
                      for (i1 = 0; i1 < 16; i1++) {
                        if ((levelstat_yx(y + i1, minx + 16) & 7) == OBJ_BOULD)
                          bouldcnt++;
                        else if (levelstat_yx(y + i1, minx + 16) & 7) break;
                      }
                      if (bouldcnt && i1 == 16) {
                        /* cesta je volna, az na balvany - zkusime je
                         * posunout */
                        if (!boulder_move(minx, y, 2)) i1 = 0;
                      }
                      if (i1 == 16) {
                        /* cesta je volna */
                        if (++phasefred[fredno] > 1) phasefred[fredno] = 0;
                        redrawfhoriz_img(imgfred[fredno * 2 + phasefred[fredno]],
                            (minx + 1) * 2, y * 2, minx * 2);
                        levelstat_yx(y, minx + 1) = lstat;
                          /* coz se rovna levelstat_yx(y, minx) */
                        for (i1 = 0; i1 < 16; i1++) {
                          levelstat_yx(y + i1, minx) = 0;
                          levelstat_yx(y + i1, minx + 16) = levelstat_yx(y + i1, minx + 15);
                        }
                        found->x++;
                        minx++;
                      }
                    }
                    else {
                      /* uz jsme uplne vpravo a chceme jeste vic, takze jsme
                       * vyhrali */
                      levelfinished = 1;
                      return;
                    }
                  }
                  break;
                case 2: /* vlevo */
                  if (levelbagger_n(0, fredno) & 64) {
                    /* Fred je otoceny vpravo, takze ho otocime doleva */
                    levelbagger_n(0, fredno) &= ~64;
                    drawf_img(imgfredr[fredno * 2], minx * 2, y * 2);
                    phasefred[fredno] = 0;
                  }
                  else {
                    /* Fred je otoceny vlevo, takze muzeme pochodovat */
                    if (minx > 0) {
                      /* jeste nejsme uplne vlevo */
                      /* nejdrive zjistime, zda vubec doleva muzeme */
                      bouldcnt = 0;
                      for (i1 = 0; i1 < 16; i1++) {
                        if ((levelstat_yx(y + i1, minx - 1) & 7) == OBJ_BOULD)
                          bouldcnt++;
                        else if (levelstat_yx(y + i1, minx - 1) & 7) break;
                      }
                      if (bouldcnt && i1 == 16) {
                        /* cesta je volna, az na balvany - zkusime je
                         * posunout */
                        if (!boulder_move(minx, y, 3)) i1 = 0;
                      }
                      if (i1 == 16) {
                        /* cesta je volna */
                        if (++phasefred[fredno] > 1) phasefred[fredno] = 0;
                        redrawfhoriz_img(imgfredr[fredno * 2 + phasefred[fredno]],
                            (minx - 1) * 2, y * 2, minx * 2);
                        for (i1 = 0; i1 < 16; i1++) {
                          levelstat_yx(y + i1, minx + 15) = 0;
                          levelstat_yx(y + i1, minx - 1) = levelstat_yx(y + i1, minx);
                        }
                        levelstat_yx(y, minx) = levelstat_yx(y, minx + 1);
                        found->x--;
                        minx--;
                      }
                    }
                  }
                  break;
                case 3: /* nahoru */
                  if (y == maxy && y + 16 < YSIZE_LEVELSTAT) {
                    /* jen pokud jsme neklesli (a muzeme mit pod sebou
                     * vytahovou plosinu) */
                    /* podivame se, jestli mame pod sebou vytah */
                    if (levelstat_yx(y + 16, minx) == (OBJ_UP | 128)) {
                      fredup[fredno] = 1;
                      fred_move_up(fredno, minx, &y, found);
                    }
                  }
              } /* switch */
          }
        }
        else {
          /* nebyla stisknuta zadna klavesa, takze uvedeme Freda do zakladni
           * faze */
          if (phasefred[fredno]) {
            if (levelbagger_n(0, fredno) & 64)
              drawf_img(imgfred[fredno * 2], minx * 2, y * 2);
            else drawf_img(imgfredr[fredno * 2], minx * 2, y * 2);
            phasefred[fredno] = 0;
          }
        }

        if (y == maxy && levelstat_yx(y + 16, minx) == (OBJ_UP | 128))
          /* mame pod sebou vytah, takze nakreslime indikator */
          drawf_img(imgupact, minx * 2, (y + 16) * 2);

        fredno = 0;                     /* kazdy pruchod predpoklada na zacatku fredno == 0 */
        break;
    } /* switch */
  } /* for (i_things) */

  redrawf();
  /* posun fazi */
  if (++phaserun > 5) phaserun = 0;
  if (phaseactbagg)
    if (++phasebagg > 1) phasebagg = 0;
  phaseactbagg = !phaseactbagg;
  /* vynulovani priznaku, ze baggery byly obslouzeny */
  for (i1 = 1; i1 < 16; i1++)
    levelbagger_n(i1, 1) &= ~128;

  /* delay */
  if (keyboard[keyscfg[8]] && !demoactive)
    /* jestlize je zmacknuta zrychlovaci klavesa, cekame, jen jako by byla
     * nejvyssi rychlost */
    timer += STEP_SPEED;
  else timer += gamespeed * STEP_SPEED;
  now = SDL_GetTicks();
  if (now < timer) SDL_Delay(timer - now);
}
