/*
 *  Copyright (c) 2002-2007 Jiri Benc <jbenc@upir.cz>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * keys.c
 *
 * Funkce pro praci s klavesnici.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include "SDL.h"
#include "params.h"
#include "keys.h"


/* Struktura dvojice klavesa-nazev */
typedef struct {
  SDLKey key;
  char name[12];
} keyname_pair;

/* Struktura prekladu klaves na znaky */
typedef struct {
  SDLKey key;
  char translate;
} keytranslate_pair;

#include "keytable.c"           /* tabulky znaku */

static keyname_pair *keynames_act;
static keytranslate_pair *xtab_act;
static keytranslate_pair *xtab_sh_act;

/* hodnota a modifikatory naposledy stisknute klavesy */
SDLKey lastkey = 0;
SDLMod lastmod = 0;

/* pole stisknutych klaves */
Uint8 *keyboard;

/* pocet milisekund pred akceptovanim dalsi klavesy pri vstupu textu */
#define KEY_DELAY       200


void reinit_keys(void)
{
	switch (param_lang) {
	case LANG_CZ:
		keynames_act = keynames_cz;
		xtab_act = xtab_cz;
		xtab_sh_act = xtab_sh_cz;
		break;
	case LANG_EN:
	default:
		keynames_act = keynames_en;
		xtab_act = xtab_en;
		xtab_sh_act = xtab_sh_en;
		break;
	}
}

/* Filtr na udalosti SDL */
int sdl_filter(const SDL_Event *event)
{
  switch (event->type) {
    case SDL_KEYDOWN:
      lastkey = event->key.keysym.sym;
      lastmod = event->key.keysym.mod;
      return 0;
    case SDL_KEYUP:
      if (lastkey == event->key.keysym.sym) {
        /*lastkey = 0;
        lastmod = 0;*/
      }
      return 0;
    default:
      return 1;
  }
}


/* Inicializace klavesnice (urceno pro volani z init_sdl()). */
void init_sdlkeys()
{
  reinit_keys();
  SDL_SetEventFilter(sdl_filter);       /* nastaveni filtru udalosti */
  keyboard = SDL_GetKeyState(NULL);
}


/* Delay v delce, jaka se ceka pred prijetim klavesy jako stisknute. */
void delay_before_key()
{
  Uint32 timer, now;

  timer = SDL_GetTicks();               /* ulozeni pocatecniho poctu tiku */
  /* delay */
  timer += KEY_DELAY;
  now = SDL_GetTicks();
  if (now < timer) SDL_Delay(timer - now);
}


/* Ceka na stisk klavesy a vrati ji. */
SDLKey wait_for_key()
{
  SDL_Event event;

  lastkey = lastmod = 0;
  SDL_SetEventFilter(NULL);
  delay_before_key();
  while (1) {
    SDL_WaitEvent(&event);
    if (event.type == SDL_KEYDOWN)
      break;
  }
  SDL_SetEventFilter(sdl_filter);
  lastkey = event.key.keysym.sym;
  lastmod = event.key.keysym.mod;
  return lastkey;
}



static char tmpkeyname[32];

/* Vrati nazev klavesy. */
char *key_name(SDLKey key)
{
  int i1;
  
  for (i1 = 0; i1 < (sizeof(keynames_cz) / sizeof(keyname_pair)); i1++) {
    if (keynames_act[i1].key == key) return keynames_act[i1].name;
  }
  sprintf(tmpkeyname, "#%X", key);
  return tmpkeyname;
}


/* Prelozi posledni stisknutou klavesu na znak. */
char key_translate()
{
  int i1;
  keytranslate_pair *p;

  if ((lastmod & KMOD_SHIFT)) {
    p = xtab_sh_act;
    i1 = sizeof(xtab_sh_cz) / sizeof(keytranslate_pair);
  }
  else {
    p = xtab_act;
    i1 = sizeof(xtab_cz) / sizeof(keytranslate_pair);
  }
  for (; i1 > 0; i1--) {
    if (p->key == lastkey) return p->translate;
    p++;
  }
  return 0;
}
