/*
 *  Copyright (c) 2002-2007 Jiri Benc <jbenc@upir.cz>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * graphics.h
 *
 * Funkce pro praci s grafikou (pomoci SDL).
 *
 */

#include "SDL.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "keys.h"
#include "readdata.h"
#include "params.h"
#include "graphics.h"

SDL_Surface *screen;            /* surface obrazovky */

/* surfacy obrazku (spritu): */
SDL_Surface 
  *(imgfred[4]),                /* Fred (hrac 1 & hrac 2) */
  *(imgfredr[4]),               /* Fred - reverzni */
  *(imgbagg[2]),                /* Bagger */
  *(imgbaggr[2]),               /* Bagger - reverzni */
  *(imgrun[4]),                 /* spoustec baggera */
  *(imgup[2]),                  /* vytah */
  *imgupact,                    /* indikator stani na vytahu */
  *(imgbrick[CNT_IMGBRICK]),    /* cihla */
  *(imgbould[CNT_IMGBOULD]),    /* balvan */
  *imgfont,                     /* font */
  *imgtitle,                    /* logo "Fred" */
  *(imgmainmenu[2]),            /* hlavni menu */
  *imgctrlsmenu,                /* menu pro nastaveni ovladani */
  *(imggamemenu[2]),            /* menu v prubehu hry */
  *(imggamespeed[2]),           /* box pro nastaveni rychlosti */
  *imgpasswd,                   /* napis "Heslo:" */
  /* miniverze obrazku (pro nahledy pri save/load): */
  *(imgmfred[2]),
  *(imgmfredr[2]),
  *imgmbagg,
  *imgmbaggr,
  *imgmrun,
  *imgmup,
  *(imgmbrick[CNT_IMGBRICK]),
  *(imgmbould[CNT_IMGBOULD]);

/* hodnota cerne barvy (pro zrychleni prekreslovani) */
Uint16 black;

/* buffer obdelniku pro rychle prekreslovani */
SDL_Rect *rectchanged;                  /* buffer */
int rectchangedcnt;                     /* aktualni velikost bufferu */
#define SIZE_RECTCHANGED        440     /* max. pocet obdelniku */


/* Vypsani chyby SDL (a ukonceni programu). */
void error_sdl(const char *msg)
{
  error("%s (%s)", msg, SDL_GetError());
}


/* Natazeni spritu do surfacu - vraci surface. Jako vstup je nazev spritu
 * (tak, jak je ulozen v baliku). */
SDL_Surface *load_img(const char *name)
{
  char *data;
  int w, h;
  SDL_Surface *img, *res;
  char path[32];

  strcpy(path, name);
  strcat(path, ".img");
  data = pack_fget(path, NULL);
  w = ((int *)(data))[0];
  h = ((int *)(data))[1];
  img = SDL_CreateRGBSurfaceFrom(data + 8, w, h, 24, w * 3,
      0xff0000, 0x00ff00, 0x0000ff, 0);
  /*img = SDL_LoadBMP(path);*/
  if (!img) error_sdl(ls(STR_GRAPHICS_ERR_IMG_LOAD));
  res = SDL_DisplayFormat(img);                 /* konverze na bitovou hloubku obrazovky */
  if (!res) error_sdl(ls(STR_GRAPHICS_ERR_IMG_CONV));
  SDL_FreeSurface(img);
  free(data);
  return res;
}


/* Natazeni serie spritu (se jmeny *#, kde # je cislo) do pole surfacu. */
void load_serieimg(const char *name, SDL_Surface **img, int cnt)
{
  char fullname[64], *ch;
  int i1;

  strcpy(fullname, name);
  ch = fullname + strlen(fullname);
  for (i1 = 0; i1 < cnt; i1++) {
    sprintf(ch, "%d", i1 + 1);
    img[i1] = load_img(fullname);
  }
}


void free_img(SDL_Surface **img)
{
	if (*img) {
		SDL_FreeSurface(*img);
		*img = NULL;
	}
}

void free_serieimg(SDL_Surface **img, int cnt)
{
	int i;

	for (i = 0; i < cnt; i++)
		free_img(img + i);
}


/* Zapise dany sprite na obrazovku na dane souradnice (ale nevola
 * SDL_UpdateRect(), tedy neprekresluje obrazovku). */
void draw_img(SDL_Surface *img, Sint16 x, Sint16 y)
{
  SDL_Rect rect;

  rect.x = x;
  rect.y = y;
  SDL_BlitSurface(img, NULL, screen, &rect);
}


/* Prekresli sprite na obrazovce - tj. na puvodnim miste vykresli cerny
 * obdelnik, a pote vykresli novy sprite. */
void redraw_img(SDL_Surface *img, Sint16 newx, Sint16 newy, Sint16 oldx, Sint16 oldy)
{
  SDL_Rect rect;

  rect.x = oldx;
  rect.y = oldy;
  rect.w = img->w;
  rect.h = img->h;
  SDL_FillRect(screen, &rect, black);
  rect.x = newx;
  rect.y = newy;
  SDL_BlitSurface(img, NULL, screen, &rect);
}


/* Zapise cast daneho spritu na obrazovku. Cast ma sirku spritu a udany
 * svisly ofset a vysku. */
void draw_part_img(SDL_Surface *img, Sint16 ofs_y, Uint16 h, Sint16 x, Sint16 y)
{
  SDL_Rect srcrect, dstrect;
  
  srcrect.w = img->w;
  srcrect.h = h;
  srcrect.x = 0;
  srcrect.y = ofs_y;
  dstrect.x = x;
  dstrect.y = y;
  SDL_BlitSurface(img, &srcrect, screen, &dstrect);
}


/* Zapise na obrazovku na dane souradnice obdelnik v dane velikosti a barve
 * (zadane pres RGB slozky). */
void draw_bar(Sint16 x, Sint16 y, Uint16 w, Uint16 h, Uint8 r, Uint8 g, Uint8 b)
{
  SDL_Rect rect;

  rect.x = x;
  rect.y = y;
  rect.w = w;
  rect.h = h;
  SDL_FillRect(screen, &rect, SDL_MapRGB(screen->format, r, g, b));
}


/* Vykresli nevyplneny obdelnik v zadane barve. */
void draw_rect(Sint16 x, Sint16 y, Uint16 w, Uint16 h, Uint8 r, Uint8 g, Uint8 b)
{
  SDL_Rect rect;
  Uint16 color;

  rect.x = x;
  rect.y = y;
  rect.w = w;
  color = SDL_MapRGB(screen->format, r, g, b);
  rect.h = 1;
  SDL_FillRect(screen, &rect, color);
  rect.y += h - 1;
  SDL_FillRect(screen, &rect, color);
  rect.y = y;
  rect.h = h;
  rect.w = 1;
  SDL_FillRect(screen, &rect, color);
  rect.x += w - 1;
  SDL_FillRect(screen, &rect, color);
}


/* Prekresli obrazovku. */
void redraw()
{
  SDL_UpdateRect(screen, 0, 0, 0, 0);
  rectchangedcnt = 0;                   /* pro pripad, ze bylo volano i drawf_*() */
}


/* Varianta funkce draw_img(), ktera navic uklada zmeneny obdelnik pro
 * pozdejsi rychlejsi prekresleni. */
void drawf_img(SDL_Surface *img, Sint16 x, Sint16 y)
{
  SDL_Rect rect;

  rect.x = x;
  rect.y = y;
  SDL_BlitSurface(img, NULL, screen, &rect);
  memcpy(rectchanged + rectchangedcnt++, &rect, sizeof(SDL_Rect));
}


/* Varianta funkce redraw_img() umoznujici pouze vodorovne prekresleni,
 * ktera uklada zmeneny obdelnik. */
void redrawfhoriz_img(SDL_Surface *img, Sint16 newx, Sint16 newy, Sint16 oldx)
{
  SDL_Rect rect;

  rect.x = oldx;
  rect.y = newy;
  rect.w = img->w;
  rect.h = img->h;
  SDL_FillRect(screen, &rect, black);
  rect.x = newx;
  rect.y = newy;
  SDL_BlitSurface(img, NULL, screen, &rect);
  if (oldx < rect.x) {
    rect.w += rect.x - oldx;
    rect.x = oldx;
  }
  else rect.w += oldx - rect.x;
  memcpy(rectchanged + rectchangedcnt++, &rect, sizeof(SDL_Rect));
}


/* Varianta funkce redraw_img() umoznujici pouze svisle prekresleni,
 * ktera uklada zmeneny obdelnik. */
void redrawfvert_img(SDL_Surface *img, Sint16 newx, Sint16 newy, Sint16 oldy)
{
  SDL_Rect rect;

  rect.x = newx;
  rect.y = oldy;
  rect.w = img->w;
  rect.h = img->h;
  SDL_FillRect(screen, &rect, black);
  rect.x = newx;
  rect.y = newy;
  SDL_BlitSurface(img, NULL, screen, &rect);
  if (oldy < rect.y) {
    rect.h += rect.y - oldy;
    rect.y = oldy;
  }
  else rect.h += oldy - rect.y;
  memcpy(rectchanged + rectchangedcnt++, &rect, sizeof(SDL_Rect));
}


/* Varianta funkce draw_bar(), ktera pouziva jen cernou barvu a uklada
 * zmeneny obdelnik. */
void drawf_blackbar(Sint16 x, Sint16 y, Uint16 w, Uint16 h)
{
  SDL_Rect rect;

  rect.x = x;
  rect.y = y;
  rect.w = w;
  rect.h = h;
  SDL_FillRect(screen, &rect, black);
  memcpy(rectchanged + rectchangedcnt++, &rect, sizeof(SDL_Rect));
}


/* Varianta funkce draw_rect(), ktera uklada zmeneny obdelnik. */
void drawf_rect(Sint16 x, Sint16 y, Uint16 w, Uint16 h, Uint8 r, Uint8 g, Uint8 b)
{
  SDL_Rect rect;
  Uint16 color;

  rect.x = x;
  rect.y = y;
  rect.w = w;
  rect.h = h;
  color = SDL_MapRGB(screen->format, r, g, b);
  memcpy(rectchanged + rectchangedcnt++, &rect, sizeof(SDL_Rect));
  rect.h = 1;
  SDL_FillRect(screen, &rect, color);
  rect.y += h - 1;
  SDL_FillRect(screen, &rect, color);
  rect.y = y;
  rect.h = h;
  rect.w = 1;
  SDL_FillRect(screen, &rect, color);
  rect.x += w - 1;
  SDL_FillRect(screen, &rect, color);
}


/* Varianta funkce redraw(), ktera prekresluje jen zmenene obdelniky. */
void redrawf()
{
  SDL_UpdateRects(screen, rectchangedcnt, rectchanged);
  rectchangedcnt = 0;
}


/* Vypis textu. */
void draw_text(int x, int y, char *text)
{
  SDL_Rect srcrect, dstrect;
  
  srcrect.w = imgfont->w;
  srcrect.h = Y_FONT;
  for (; *text; text++) {
    dstrect.x = x;
    dstrect.y = y;
    srcrect.x = 0;
    srcrect.y = ((uchar)*text - ' ') * Y_FONT;
    SDL_BlitSurface(imgfont, &srcrect, screen, &dstrect);
    x += imgfont->w;
  }
}


void reinit_imgs(void)
{
	free_serieimg(imgmainmenu, 2);
	load_serieimg("titmenu", imgmainmenu, 2);
	free_img(&imgctrlsmenu);
	imgctrlsmenu = load_img("controls");
	free_img(&imgpasswd);
	imgpasswd = load_img("passwd");

	free_serieimg(imggamemenu, 2);
	load_serieimg("menu", imggamemenu, 2);
	free_serieimg(imggamespeed, 2);
	load_serieimg("speed", imggamespeed, 2);
	SDL_SetColorKey(imggamemenu[0], SDL_SRCCOLORKEY, black);
	SDL_SetColorKey(imggamemenu[1], SDL_SRCCOLORKEY, black);
	SDL_SetColorKey(imggamespeed[0], SDL_SRCCOLORKEY, black);
	SDL_SetColorKey(imggamespeed[1], SDL_SRCCOLORKEY, black);
}

/* Inicializace vseho potrebneho pro SDL a natazeni spritu */
void init_sdl()
{
  /* alokace pameti pro buffer obdelniku */
  rectchanged = emalloc(sizeof(SDL_Rect) * SIZE_RECTCHANGED);
  rectchangedcnt = 0;

  /* inicializace SDL */
  if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0)
    error_sdl(ls(STR_GRAPHICS_ERR_SDL_INIT));

  /* nastaveni exitovaci funkce pro SDL */
  atexit(SDL_Quit);

  /* nastaveni odfiltrovani udalosti */
  SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);  /* mys nas nezajima */
  SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
  SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE);
  SDL_EventState(SDL_SYSWMEVENT, SDL_IGNORE);   /* nestandardni udalosti WM uz vubec ne */

  /* inicializace klavesnice */
  init_sdlkeys();

  /* nastaveni titulku okna na "Fred" */
  SDL_WM_SetCaption("Fred", "Fred");

  /* prepnuti do rezimu 640x400x16bit fullscreen */
  screen = SDL_SetVideoMode(640, 400, 16, /* SDL_HWSURFACE | */ SDL_FULLSCREEN);
      /* SDL_HWSURFACE pod Win DX5 na S3 ViRGE nefunguje, jinde je untested */
  if (!screen)
    error_sdl(ls(STR_GRAPHICS_ERR_FULLSCR));

  /* vypnuti kurzoru */
  SDL_ShowCursor(0);

  /* zcachovani cerne barvy (pro redraw_img()) */
  black = SDL_MapRGB(screen->format, 0, 0, 0);

  /* natazeni spritu */
  load_serieimg("fred", imgfred, 4);
  load_serieimg("fredr", imgfredr, 4);
  load_serieimg("bagger", imgbagg, 2);
  load_serieimg("baggerr", imgbaggr, 2);
  load_serieimg("run", imgrun, 4);
  load_serieimg("up", imgup, 2);
  load_serieimg("brick", imgbrick, CNT_IMGBRICK);
  load_serieimg("bould", imgbould, CNT_IMGBOULD);

  load_serieimg("mfred", imgmfred, 2);
  load_serieimg("mfredr", imgmfredr, 2);
  imgmbagg = load_img("mbagger");
  imgmbaggr = load_img("mbaggerr");
  imgmrun = load_img("mrun");
  imgmup = load_img("mup");
  load_serieimg("mbrick", imgmbrick, CNT_IMGBRICK);
  load_serieimg("mbould", imgmbould, CNT_IMGBOULD);

  imgfont = load_img("font");
  imgtitle = load_img("title");

  imgupact = load_img("upactive");
  SDL_SetColorKey(imgupact, SDL_SRCCOLORKEY, black);

  reinit_imgs();
}

