#ifndef DANCING_H
#define DANCING_H

/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  The source code of this module is based on java code by
  Stan Chesnutt.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <assert.h>

#include <iostream>
#include <list>
#include <vector>

class Column;

class Node
{
private:
  Node   *left;
  Node   *right;
  Node   *up;
  Node   *down;
  Column *column;
  int    rowNumber;
  int    label;

public:
  /*
  * Create a self-referential node
  */
  Node();
  Node(int rowNumber, int label, Column *column);
  virtual ~Node();

  inline void setLeft(Node *node)
  {
    left=node;
  }

  inline Node* getLeft() const
  {
    return left;
  }

  inline void setRight(Node *node)
  {
    right=node;
  }

  inline Node* getRight() const
  {
    return right;
  }

  inline void setUp(Node *node)
  {
    up=node;
  }

  inline Node* getUp() const
  {
    return up;
  }

  inline void setDown(Node *node)
  {
    down=node;
  }

  inline Node* getDown() const
  {
    return down;
  }

  inline void setColumn(Column *node)
  {
    column=node;
  }

  inline Column* getColumn() const
  {
    return column;
  }

  inline int getLabel() const
  {
    return label;
  }

  inline int getRowNumber() const
  {
    return rowNumber;
  }
};

class Column : public Node
{
private:
  int  size; //! The number of rows which have a "1" in this column

  /*
   * An "optional" column represents a constraint that doesn't have to
   * be satisfied, but can only be satisfied once.  Columns that are
   * not "optional" must be satisfied once and only once.
   */
  bool optional;

public:
  Column();
  Column(int label, bool optional);

  void addAtEnd(Node *node);

  inline bool isOptional() const
  {
    return optional;
  }

  inline int getCount() const
  {
    return size;
  }

  inline void incrementSize()
  {
    ++size;
  }

  inline void decrementSize()
  {
    --size;
  }
};

class NodeSnapshot
{
public:
  Node* pointer;
  Node  value;
};

class ColumnSnapshot
{
public:
  Column* pointer;
  Column  value;
};

class DancingLinksArena
{
private:
  Column             *firstColumn;
  int                rowCount;
  std::vector<Node*> solution;
  size_t             solutionIndex;
  size_t             startingCount;
  Node               *traveller;
  std::list<Column*> columnUsed;
  std::list<Node*>   nodeUsed;

  std::list<NodeSnapshot>   nodeSnapshot;
  std::list<ColumnSnapshot> columnSnapshot;
  Column                    *firstColumnSnapshot;

private:
  Column* getNextNonOptionalColumn(Node *node);
  void solveNonRecurse(std::vector<int>& result);
  Column* newColumn(int label, bool optional);
  Node* newNode(int rowNumber, int label, Column *column);

public:
  DancingLinksArena();
  ~DancingLinksArena();

  void assignLabels(int *labels, bool *optional, size_t length);
  void assignLabels(int *labels, size_t length);
  Node* addInitialRow(int *labels, size_t length);
  void snapshot();
  void restore();

  void removeColumn(Column *columnHead);
  void reinsertColumn(Column *columnNode);
  void removeRow(Node *rowHead);
  void reinsertRow(Node *rowHead);

  void solve(std::vector<int>& solution);
  void removeInitialSolutionSet(const std::list<Node*>& solutions);
};

class DancingLinksSudoku
{
private:
  DancingLinksArena dla;
  int               labels[324];
  int               rowData[9][9][9][4];
  Node              *rows[9][9][9];

private:
  bool handleSolution(const std::vector<int>& rowIndex, std::vector<int>& solution);

public:
  DancingLinksSudoku();

  void assignRiddle(size_t *puzzle);
  void assignRiddle(const std::vector<size_t>& puzzle);

  bool solveit(std::vector<int>& solution);
};
#endif
