/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Statistics.h"

#include <Lum/Base/L10N.h>

#include <Lum/Model/DataStream.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Space.h>
#include <Lum/Text.h>
#include <Lum/WindowGroup.h>

static Lum::Object* GetPercentBar(size_t count)
{
  Lum::PercentBar                 *bar;
  Lum::Model::DoubleDataStreamRef stream;

  stream=new Lum::Model::DoubleDataStream();
  stream->SetChannels(1);
  stream->Set(0,(1.0*count)/9);

  bar=new Lum::PercentBar();
  bar->DisplayVertical();
  bar->SetHeight(Lum::Base::Size::stdCharHeight,5);
  bar->SetModel(stream);

  return bar;
}

static Lum::Object* GetText(const std::wstring& label)
{
  Lum::Text *text;

  text=new Lum::Text();
  text->SetFlex(true,false);
  text->SetAlignment(Lum::Text::centered);
  text->SetText(label);

  return text;
}

StatisticsDialog::StatisticsDialog(const Sudoku& game)
 : game(game)
{
  SetExitAction(GetClosedAction());
}

StatisticsDialog::~StatisticsDialog()
{
  // No code
}

void StatisticsDialog::PreInit()
{
  Lum::Button      *button;
  Lum::ButtonRow   *row;
  Lum::Grid        *grid;
  Lum::Panel       *vert;
  Lum::WindowGroup *wGroup;

  wGroup=new Lum::WindowGroup();

  vert=new Lum::VPanel();

  grid=new Lum::Grid();
  grid->SetSpace(true,true);
  grid->SetSize(9,2);
  grid->SetEqualDimensions(true,false);

  for (size_t i=1; i<=9; i++) {
    size_t count=0;

    for (size_t j=0; j<Sudoku::size; j++) {
      Sudoku::Bitset allowed;

      if (game.Get(j)==i) {
        game.GetPotential(j,allowed);

        if (allowed.test(i)) {
          count++;
        }
      }
    }

    grid->SetObject(i-1,0,GetPercentBar(count));
  }

  grid->SetObject(0,1,GetText(L"1"));
  grid->SetObject(1,1,GetText(L"2"));
  grid->SetObject(2,1,GetText(L"3"));
  grid->SetObject(3,1,GetText(L"4"));
  grid->SetObject(4,1,GetText(L"5"));
  grid->SetObject(5,1,GetText(L"6"));
  grid->SetObject(6,1,GetText(L"7"));
  grid->SetObject(7,1,GetText(L"8"));
  grid->SetObject(8,1,GetText(L"9"));
  vert->Add(grid);

  vert->Add(new Lum::VSpace());

  row=new Lum::ButtonRow();
  row->SetFlex(true,false);

  button=new Lum::Button();
  button->SetFlex(true,true);
  button->RequestFocus();
  button->SetText(_ld(dlgButtonOk));
  button->SetType(Lum::Button::typeDefault);
  button->SetModel(GetClosedAction());
  row->Add(button);

  vert->Add(row);

  wGroup->SetMain(vert);

  SetTop(wGroup);

  Dialog::PreInit();
}

void StatisticsDialog::Show(Lum::OS::Window* parent, const Sudoku& game)
{
  StatisticsDialog *dialog;

  dialog=new StatisticsDialog(game);
  dialog->SetParent(parent);
  dialog->SetTitle(L"Statistics...");

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  delete dialog;
}

