/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Model/Boolean.h>

#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Panel.h>
#include <Lum/PopupGroup.h>
#include <Lum/Toggle.h>

static GameArea::Prefs *prefs=new GameArea::Prefs();

class SetSelector : public Lum::Dialog
{
private:
  Lum::Model::ActionRef models[10]; // Values 0..9, where 0 means empty
  bool                  selected;
  size_t                value;

public:
  SetSelector()
  : selected(false),
    value(10)
  {
    for (size_t i=0; i<=9; i++) {
      models[i]=new Lum::Model::Action();
      AttachModel(models[i]);
    }

    SetType(Lum::OS::Window::typePopup);
  }

  ~SetSelector()
  {
    for (size_t i=0; i<=9; i++) {
      UnattachModel(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Button     *button;
    Lum::Grid       *grid;
    Lum::Panel      *panel;
    Lum::PopupGroup *container;

    panel=new Lum::VPanel();

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(3,3);

    for (size_t i=1; i<=9; i++) {
      wchar_t label[2];

      label[0]=L'0'+i;
      label[1]=L'\0';

      button=new Lum::Button();
      button->SetFlex(true,true);
      button->SetText(label);
      button->SetModel(models[i]);

      grid->SetObject((i-1)%3,(i-1)/3,button);
    }
    panel->Add(grid);

    button=new Lum::Button();
    button->SetFlex(true,false);
    button->SetText(_(L"GAMEAREA_CLEAR",L"Clear"));
    button->SetModel(models[0]);

    panel->Add(button);

    container=new Lum::PopupGroup();
    container->SetMain(panel);

    SetTop(container);

    Dialog::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=9; ++i) {
      if (model==models[i] && models[i]->IsFinished()) {
        selected=true;
        value=i;
        Exit();
        break;
      }
    }

    Dialog::Resync(model,msg);
  }

  bool HasSelection() const
  {
    return selected;
  }

  size_t GetValue() const
  {
    return value;
  }
};

class MarkSelector : public Lum::Dialog
{
private:
  Lum::Model::BooleanRef models[9];
  size_t                 value;

public:
  MarkSelector(size_t value)
  : value(value)
  {
    for (size_t i=0; i<=8; i++) {
      models[i]=new Lum::Model::Boolean(value & (1 << i));
      AttachModel(models[i]);
    }

    SetType(Lum::OS::Window::typePopup);
  }

  ~MarkSelector()
  {
    for (size_t i=0; i<=8; i++) {
      UnattachModel(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Grid       *grid;
    Lum::Panel      *panel;
    Lum::PopupGroup *container;
    Lum::Toggle     *toggle;

    panel=new Lum::VPanel();

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(3,3);

    for (size_t i=0; i<=8; i++) {
      toggle=new Lum::Toggle();
      toggle->SetModel(models[i]);
      toggle->SetLabel(Lum::Base::NumberToWString(i+1));

      grid->SetObject(i%3,i/3,toggle);
    }
    panel->Add(grid);

    container=new Lum::PopupGroup();
    container->SetMain(panel);

    SetTop(container);

    Dialog::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=8; ++i) {
      if (model==models[i]) {
        if (models[i]->Get()) {
          value|=(1 << i);
        }
        else {
          value=value & ~(1 << i);
        }

        return;
      }
    }

    Dialog::Resync(model,msg);
  }

  size_t GetValue() const
  {
    return value;
  }
};

GameArea::GameArea()
 : minFont(Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                  Lum::OS::Display::fontScaleNormal)),
   markFont(Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,7)),
   font(Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                  Lum::OS::Display::fontScaleCaption1)),
   presetColor(0.5,0.5,0.5,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   focusBackgroundColor(0.95,0.95,0.95,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   markBackgroundColor(0.95,0.95,0.95,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   errorColor(1.0,0,0,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   correctColor(0,0.5,0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   boxSize(0),
   focusedCell(0),
   finished(new Lum::Model::Boolean())
{
  SetPrefs(::prefs);

  SetCanFocus(true);

  finished->Set(true);
}

int GameArea::GetCellOffset(size_t index) const
{
  return (index+1)*1+(index/3+1)*1+index*boxSize;
}

void GameArea::SetModel(const Sudoku &riddle, const Sudoku &solution)
{
  this->solution=solution;
  this->riddle=riddle;
  this->game=riddle;

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  finished->Set(game==solution);

  Redraw();
}

void GameArea::SetGame(const Game& game)
{
  riddle.SetAreaAsString(game.riddle);
  solution.SetAreaAsString(game.solution);
  this->game.SetAreaAsString(game.game);

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  if (game.marks.length()==9*Sudoku::size) {
    for (size_t i=0; i<9*Sudoku::size; i++) {
      if (game.marks[i]!=L' ') {
        marks[i/9]|=(1 << (i%9));
      }
    }
  }

  finished->Set(this->game==solution);
}

void GameArea::GetGame(Game& game) const
{
  riddle.GetAreaAsString(game.riddle);
  solution.GetAreaAsString(game.solution);
  this->game.GetAreaAsString(game.game);

  game.marks=L"";

  for (size_t i=0; i<Sudoku::size; i++) {
    for (size_t j=0; j<=8; j++) {
      if (marks[i] & (1 << j)) {
        game.marks.append(L"*");
      }
      else {
        game.marks.append(L" ");
      }
    }
  }
}

void GameArea::GetGame(Sudoku& game) const
{
  game=this->game;
}

void GameArea::CalcSize()
{
  size_t markBoxSize=0;
  size_t minBoxSize=0;

  for (size_t i=1; i<=9; i++) {
    Lum::OS::FontExtent extent;

    minFont->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

    minBoxSize=std::max(minBoxSize,extent.width-extent.left-extent.right);
    minBoxSize=std::max(minBoxSize,extent.height);
  }

  minBoxSize+=2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder);

  for (size_t i=1; i<=9; i++) {
    Lum::OS::FontExtent extent;

    markFont->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::normal);

    markBoxSize=std::max(markBoxSize,extent.width-extent.left-extent.right);
    markBoxSize=std::max(markBoxSize,extent.height);
  }

  markBoxSize*=3;

  minWidth=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);
  minHeight=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);

  boxSize=0;
  for (size_t i=1; i<=9; i++) {
    Lum::OS::FontExtent extent;

    font->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

    boxSize=std::max(boxSize,extent.width-extent.left-extent.right);
    boxSize=std::max(boxSize,extent.height);
  }

  boxSize+=2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder);

  size=(3+1)*1+(9+1)*1+9*std::max(markBoxSize,boxSize);

  width=size;
  height=size;

  Object::CalcSize();
}

Lum::OS::Font* GameArea::FindFont(size_t size)
{
  Lum::OS::FontRef font;
  bool             fits;
  size_t           fontSize;

  font=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                 Lum::OS::Display::fontScaleNormal);
  fontSize=font->pixelHeight+1;

  do {
    Lum::OS::FontRef tmpFont;
    size_t           boxSize=0;

    tmpFont=Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,fontSize);

    if (tmpFont.Valid()) {
      for (size_t i=1; i<=9; i++) {
        Lum::OS::FontExtent extent;

        tmpFont->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

        boxSize=std::max(boxSize,extent.width-extent.left-extent.right);
        boxSize=std::max(boxSize,extent.height);
      }

      fits=(boxSize<=size) && (fontSize<=size);
    }
    else {
      fits=fontSize<=size;
    }

    if (fits) {
      font=tmpFont;
      fontSize++;
    }

  } while (fits);

  return font.Get();
}

void GameArea::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();
  size_t            offset;

  boxSize=std::min((width-(3+1)*1-(9+1)*1)/9,(height-(3+1)*1-(9+1)*1)/9);

  size=(3+1)*1+(9+1)*1+9*boxSize;

  // Fill region outside game area with background
  if (size<width) {
    DrawBackground(this->x+size,this->y,width-size,size);
  }
  if (size<height) {
    DrawBackground(this->x,this->y+size,size,height-size);
  }
  if (size<width && size<height) {
    DrawBackground(this->x+size,this->y+size,width-size,height-size);
  }

  draw->PushForeground(Lum::OS::display->blackColor);

  offset=0;
  for (size_t i=0; i<10; i++) {
    // Vertical
    draw->DrawLine(this->x+offset,this->y,
                   this->x+offset,this->y+size-1);

    // Horizontal
    draw->DrawLine(this->x,this->y+offset,
                   this->x+this->size-1,this->y+offset);

    if (i%3==0) {
      offset++;

      // Vertical
      draw->DrawLine(this->x+offset,this->y,
                     this->x+offset,this->y+this->size-1);

      // Horizontal
      draw->DrawLine(this->x,this->y+offset,
                     this->x+this->size-1,this->y+offset);
    }
    offset=offset+boxSize+1;
  }
  draw->PopForeground();

  font=FindFont(boxSize-2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder));

  draw->PushFont(font,Lum::OS::Font::bold);
  for (size_t i=0; i<Sudoku::dimension*Sudoku::dimension; i++) {
    if (riddle.Get(i)!=Sudoku::empty) { // Preset fields
      std::wstring        text(Lum::Base::NumberToWString(riddle.Get(i)));
      Lum::OS::FontExtent extent;

      if (HasFocus() && !finished->Get() && i==focusedCell) { // Draw focused
        draw->PushForeground(focusBackgroundColor);
      }
      else {
        draw->PushForeground(Lum::OS::display->whiteColor);
      }
      draw->FillRectangle(this->x+GetCellOffset(i%Sudoku::dimension),
                          this->y+GetCellOffset(i/Sudoku::dimension),
                          boxSize,boxSize);
      draw->PopForeground();

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      draw->PushForeground(presetColor);
      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                       text);
      draw->PopForeground();
    }
    else {
      // Draw background
      if (HasFocus() && !finished->Get() && i==focusedCell) {
        draw->PushForeground(focusBackgroundColor);
      }
      else {
        draw->PushForeground(Lum::OS::display->whiteColor);
      }
      draw->FillRectangle(this->x+GetCellOffset(i%Sudoku::dimension),
                          this->y+GetCellOffset(i/Sudoku::dimension),
                          boxSize,boxSize);
      draw->PopForeground();

      if (game.Get(i)!=Sudoku::empty || finished->Get()) { // manually set fields
        std::wstring         text;
        Lum::OS::FontExtent  extent;
        Sudoku::Bitset       allowed;


        if (finished->Get()) {
          text=Lum::Base::NumberToWString(solution.Get(i));

          if (solution.Get(i)!=game.Get(i)) {
            draw->PushForeground(errorColor);
          }
          else {
            draw->PushForeground(correctColor);
          }
        }
        else {
          text=Lum::Base::NumberToWString(game.Get(i));

          game.GetPotential(i,allowed);

          font->StringExtent(text,extent);

          if (!allowed.test(game.Get(i))) {
            draw->PushForeground(errorColor);
          }
          else {
            draw->PushForeground(Lum::OS::display->blackColor);
          }
        }

        draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                         this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                         text);
        draw->PopForeground();
      }
      else {
        for (size_t j=0; j<=8; j++) {
          if (marks[i] & (1 << j)) {
            std::wstring         text(Lum::Base::NumberToWString(j+1));
            Lum::OS::FontExtent  extent;

            markFont->StringExtent(text,extent);

            draw->PushFont(markFont);
            draw->PushForeground(Lum::OS::display->blackColor);
            draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+
                             (boxSize/3)*(j%3)+
                             (boxSize/3-extent.width+extent.left+extent.right)/2-extent.left,
                             this->y+GetCellOffset(i/Sudoku::dimension)+
                             (boxSize/3)*(j/3)+
                             (boxSize/3-extent.height)/2+extent.ascent,
                             text);
            draw->PopForeground();
            draw->PopFont();
          }
        }
      }
    }
  }
  draw->PopFont();

  /*
  if (HasFocus() && !finished->Get()) {
    draw->PushForeground(Lum::OS::display->blackColor);
    draw->PushDash("\002\002",2,::Lum::OS::DrawInfo::fDash);
    draw->DrawRectangle(this->x+GetCellOffset(focusedCell%Sudoku::dimension)+Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceIntraObject),
                        this->y+GetCellOffset(focusedCell/Sudoku::dimension)+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceIntraObject),
                        boxSize-2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceIntraObject),
                        boxSize-2*Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceIntraObject));
    draw->PopDash();
    draw->PopForeground();
  }*/

}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (finished->Get()) {
    return false;
  }

  if ((event.type==Lum::OS::MouseEvent::down && PointIsIn(event) &&
       event.button==Lum::OS::MouseEvent::button1) ||
      (event.type==Lum::OS::MouseEvent::move && PointIsIn(event) &&
       event.IsGrabed() && event.qualifier==Lum::OS::qualifierButton1)) {
    size_t tmp=0;

    while (tmp<Sudoku::dimension*Sudoku::dimension) {
      if (event.x>=this->x+GetCellOffset(tmp%Sudoku::dimension) &&
          event.x<this->x+GetCellOffset(tmp%Sudoku::dimension)+(int)boxSize &&
          event.y>=this->y+GetCellOffset(tmp/Sudoku::dimension) &&
          event.y<this->y+GetCellOffset(tmp/Sudoku::dimension)+(int)boxSize) {
        break;
      }

      tmp++;
    }

    if (tmp<Sudoku::dimension*Sudoku::dimension && tmp!=focusedCell) {
      focusedCell=tmp;
      Redraw();
    }

    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) &&
           event.IsGrabed() && event.button==Lum::OS::MouseEvent::button1) {

    size_t tmp=0;

    while (tmp<Sudoku::dimension*Sudoku::dimension) {
      if (event.x>=this->x+GetCellOffset(tmp%Sudoku::dimension) &&
          event.x<this->x+GetCellOffset(tmp%Sudoku::dimension)+(int)boxSize &&
          event.y>=this->y+GetCellOffset(tmp/Sudoku::dimension) &&
          event.y<this->y+GetCellOffset(tmp/Sudoku::dimension)+(int)boxSize) {
        break;
      }

      tmp++;
    }

    if (tmp<Sudoku::dimension*Sudoku::dimension && riddle.Get(tmp)==Sudoku::empty) {

      if (tmp!=focusedCell) {
        focusedCell=tmp;
        Redraw();
      }

      if (event.x-(this->x+GetCellOffset(tmp%Sudoku::dimension))<=(int)boxSize/2 &&
          event.y-(this->y+GetCellOffset(tmp/Sudoku::dimension))<=(int)boxSize/2) {
        MarkSelector *selector;

        selector=new MarkSelector(marks[focusedCell]);
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x,GetWindow()->GetY()+event.y);

        if (selector->Open()) {
          selector->EventLoop();
          selector->Close();

          marks[focusedCell]=selector->GetValue();
          if (marks[focusedCell]!=0) {
            game.Set(focusedCell,Sudoku::empty);
          }

          Redraw();
        }

        delete selector;
      }
      else {
        SetSelector *selector;

        selector=new SetSelector();
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x,GetWindow()->GetY()+event.y);
        selector->Open();
        selector->EventLoop();
        selector->Close();

        if (selector->HasSelection()) {
          game.Set(focusedCell,selector->GetValue());
          marks[focusedCell]=0;
          finished->Set(game==solution);
          Redraw();
        }

        delete selector;
      }
    }
  }

  return false;
}

bool GameArea::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{
  if (finished->Get()) {
    return false;
  }

  size_t value;

  if (event.type==Lum::OS::KeyEvent::down) {
    switch (event.key) {
    case Lum::OS::keyLeft:
      if (focusedCell%Sudoku::dimension>0) {
        focusedCell--;
        Redraw();
      }
      return true;
    case Lum::OS::keyRight:
      if (focusedCell%Sudoku::dimension<Sudoku::dimension-1) {
        focusedCell++;
        Redraw();
      }
      return true;
    case Lum::OS::keyUp:
      if (focusedCell/Sudoku::dimension>0) {
        focusedCell-=Sudoku::dimension;
        Redraw();
      }
      return true;
    case Lum::OS::keyDown:
      if (focusedCell/Sudoku::dimension<Sudoku::dimension-1) {
        focusedCell+=Sudoku::dimension;
        Redraw();
      }
      return true;
    case Lum::OS::keySpace:
    case Lum::OS::keyBackspace:
    case Lum::OS::keyDelete:
      if (riddle.Get(focusedCell)==Sudoku::empty) {
        game.Set(focusedCell,Sudoku::empty);
        Redraw();
      }
      return true;
    default:
      if (Lum::Base::WStringToNumber(event.text,value)) {
        if (value>=1 && value<=Sudoku::dimension) {
          if (riddle.Get(focusedCell)==Sudoku::empty) {
            game.Set(focusedCell,value);
            finished->Set(game==solution);
            Redraw();
          }
        }

        return true;
      }
      break;
    }
  }

  return false;
}

Lum::Model::Boolean* GameArea::GetFinishedModel() const
{
  return finished.Get();
}

void GameArea::Solve()
{
  finished->Set(true);
  Redraw();
}

