#
# e_mail.py
# Easy API - e_mail module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""Email module is composed of email device manipulation functions.
"""
import smtplib
import os
from email.MIMEText import MIMEText
from email.MIMEBase import MIMEBase
from email.MIMEMultipart import MIMEMultipart
from email import Encoders

from easy.easy_exceptions import EasyError

__all__ = ["create_email", "add_attachment", "remove_attachment", "connect",
           "disconnect", "send_mail"]

class Email(object):
    """Email module is composed of email device manipulation functions.
    """

    def __init__(self):
        """Constructor for Email class.
        """
        self._to = None
        self._cc = None
        self._bcc = None
        self._subject = "(no subject)"
        self._message = None
        self._files = []

    def _get_to(self):
        """Get to's address mail.

        @rtype:  string, list or dict
        @return: to's address mail
        """
        return self._to

    def _set_to(self, to):
        """Set to's address mail.

        @type  to: string, list or dict
        @param to: to's address mail
        """
        self._to = to

    def _get_cc(self):
        """Get cc's address mail.

        @rtype:  string, list or dict
        @return: cc's address mail
        """
        return self._cc

    def _set_cc(self, cc):
        """Set cc's address mail.

        @type  cc: string, list or dict
        @param cc: cc's address mail
        """
        self._cc = cc

    def _get_bcc(self):
        """Get bcc's address mail.

        @rtype:  string, list or dict
        @return: bcc's address mail
        """
        return self._bcc

    def _set_bcc(self, bcc):
        """Set bcc's address mail.

        @type  bcc: string, list or dict
        @param bcc: bcc's address mail
        """
        self._bcc = bcc

    def _get_subject(self):
        """Get subject's mail.

        @rtype:  string
        @return: subject's mail
        """
        return self._subject

    def _set_subject(self, subject):
        """Set subject's mail.

        @type  subject: string
        @param subject: subject's mail
        """
        self._subject = subject

    def _get_message(self):
        """Get message's mail.

        @rtype:  string
        @return: mail's message
        """
        return self._message

    def _set_message(self, message):
        """Set message's mail.

        @type  message: string
        @param message: message's mail
        """
        self._message = message

    def _create_email(self):
        """Create an email.

        @rtype:  Email
        @return: the created email
        """
        return Email()

    def add_attachment(self, file_path):
        """Attach a file.

        @type  file_path: string
        @param file_path: file's path
        """
        self._files.append(file_path)

    def remove_attachment(self, file_path):
        """Remove a file.

        @type  file_path: string
        @param file_path: file's path
        """
        self._files.remove(file_path)

    to = property(_get_to, _set_to)
    cc = property(_get_cc, _set_cc)
    bcc = property(_get_bcc, _set_bcc)
    subject = property(_get_subject, _set_subject)
    message = property(_get_message, _set_message)

_TO = "to"
_CC = "cc"
_BCC = "bcc"
_SUBJECT = "subject"

_smtp = smtplib.SMTP()
_email = None
_multipart = None
_connected = False
_login = ""

_email = Email
create_email = _email
add_attachment = _email.add_attachment
remove_attachment = _email.remove_attachment


def connect(login, password, smtp_server):
    """Connect to a smtp server.

    @type        login: string
    @param       login: login to access smtp server
    @type     password: string
    @param    password: password to access smtp server
    @type  smtp_server: string
    @param smtp_server: smtp server destination
    """
    global _connected
    if not _connected:
        _login = login
        try:
            _smtp.connect(smtp_server)
            _smtp.ehlo(login)
            _smtp.starttls()
            _smtp.ehlo(login)
            _smtp.login(login, password)
            _connected = True
        except:
            raise EasyError("Error in connection")


def disconnect():
    """Disconnect the smtp server.
    """
    global _connected
    if _connected:
        _smtp.close()
    _connected = False


def send_mail(email):
    """Send an email to destination.

    @type  email: Email object
    @param email: email's destination
    """
    _multipart = MIMEMultipart()
    _multipart[_TO] = _to_str(email.to)
    _multipart[_CC] = _to_str(email.cc)
    _multipart[_BCC] = _to_str(email.bcc)
    _multipart[_SUBJECT] = _to_str(email.subject)

    _multipart.attach(MIMEText(email.message))

    for file_path in email._files:
        try:
            fopen = open(file_path, "rb")
            base = MIMEBase("application", "octet-stream")
            base.set_payload(fopen.read())
            Encoders.encode_base64(base)
            base.add_header("Content-Disposition", "attachment; filename= %s"
                             % os.path.basename(file_path))
            _multipart.attach(base)
            fopen.close()
        except:
            raise EasyError("Error in attachment")

    to_addrs = _to_list(email.to)
    cc_addrs = _to_list(email.cc)
    bcc_addrs = _to_list(email.bcc)
    addrs = to_addrs + cc_addrs + bcc_addrs

    _smtp.sendmail(_login, addrs, _multipart.as_string())


def _to_list(value):
    """Transform to list.

    @type  value: string, list ou dict
    @param value: the value to be transformed to string
    """
    emails_list = []
    if type(value) is str:
        emails_list = value.replace(" ", "").split(",")
    elif type(value) is list:
        emails = ""
        for each in value:
            emails += each + ","
        emails_list = emails.replace(" ", "").split(",")
    elif type(value) is dict:
        emails = ""
        for each in value:
            emails += "%s <%s>" % (value.get(each), each) + ","
        emails_list = emails.split(",")
    return emails_list


def _to_str(value):
    """Transform to string.

    @type  value: string, list ou dict
    @param value: the value to be transformed to string
    """
    emails_str = ""
    if type(value) is str:
        emails_str = value
    elif type(value) is list:
        emails_str = ""
        for each in value:
            emails_str += "%s ," % each
    elif type(value) is dict:
        for each in value:
            emails_str += "%s <%s>" % (value.get(each), each) + ","
    return emails_str