#
# playback_recorder.py
# Easy API - playback_recorder module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Abraao de Morais Marinho dos Santos <abraao@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""This module encapsulates a set of functions to record and reproduce
media streams. Some examples of features include seek, file time,
current time position and volume control.
"""
from easy.easy_exceptions import EasyError

import pygst
pygst.require("0.10")
import gst
import os
import platform
import thread
import time



__all__ = ["is_playing", "is_paused", "is_recording", "is_stopped", "play",
           "pause", "stop", "record", "record_async", "get_length",
           "get_current_position", "seek", "get_volume", "set_volume"]

class PlaybackRecorder(object):
    
    def __init__(self):
        """Constructor for PlaybackRecorder class.
        """
        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False

        self._init_playback_pipeline()
        self._init_recorder_pipeline()
        self.set_volume(10)
    
    def _init_playback_pipeline(self):
        """Initializes the Playback pipeline
        """
        self._playback_pipeline = gst.parse_launch("playbin name=sink")
        
    def _init_recorder_pipeline(self):
        """Initializes the Recorder pipeline
        """        
        #To be implemented by child classes
        self._recorder_pipeline = None

    def is_playing(self):
        """Check if device is playing.

        @rtype:  boolean
        @return: True, if device is playing
        """
        return self._PLAYING

    def is_paused(self):
        """Check if device is paused.

        @rtype:  boolean
        @return: True, if device is paused
        """
        return self._PAUSED

    def is_recording(self):
        """Check if device is recording.

        @rtype:  boolean
        @return: True, if device is recording
        """
        return self._RECORDING

    def is_stopped(self):
        """Check if device is stopped.

        @rtype:  boolean
        @return: True, if device is stopped
        """
        return not (self.is_playing() or self.is_paused() or \
                    self.is_recording())

    def play(self, filename=None):
        """Start playing audio file.

        @type  filename: string
        @param filename: file name
        """
        if filename and os.path.exists(filename):
            self.stop()            
            # We need to create a new playback pipeline every time the play
            # function is called, because there was troubles when we try to 
            # play a *.wav and after a *.mp3 file (and vice versa) using the 
            # same pipeline using playbin element.
            #
            # When a playback pipeline is created its default volume value is
            # 1, so we need store the actual volume value before create a new 
            # playback pipeline and after set its volume to the stored value. 
            # Otherwise every time we create a new playback pipeline the 
            # volume will be set to 1.
            
            volume = self.get_volume()
            self._init_playback_pipeline()
            self.set_volume(volume)
            
            filename = os.path.abspath(filename)
            self._playback_pipeline.set_property("uri", "file://" + filename)
            self._playback_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True
            thread.start_new_thread(self._catch_EOS, ())
        else:
            raise EasyError("File does not exist.")

    def pause(self):
        """Pause or resume the current playing audio file.
        """
        if self.is_playing():
            self._playback_pipeline.set_state(gst.STATE_PAUSED)
            self._PAUSED = True
            self._PLAYING = False
        elif self.is_paused():
            self._playback_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True
            self._PAUSED = False

    def stop(self):
        """Stop the current audio stream.
        """
        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False
        
        if self._playback_pipeline:
            self._playback_pipeline.set_state(gst.STATE_NULL)
        if self._recorder_pipeline:
            self._recorder_pipeline.set_state(gst.STATE_NULL)
        
    def record(self, filename, record_time=-1, callback=None):
        """Records an media file according to the recorder pipeline.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file. If '-1' the 'stop' method must be called to break recording.
        @type     callback: function
        @param    callback: function that will be called after recording
        """
        if record_time:
            try:
                record_time = int(record_time)
            except Exception, e:
                raise EasyError("The given recording length's is not a valid integer.")
            if record_time < -1:
                raise EasyError("The given recording length's is not a positive integer or -1.")

        filename = os.path.abspath(filename)
        self._set_recorder_file(filename)
        
        self._recorder_pipeline.set_state(gst.STATE_PLAYING)
        self._RECORDING = True
        
        if record_time > 0:
            thread.start_new_thread(self._time_counter, (record_time, callback))
    
    def _time_counter(self, _length, callback=None):
        """Create a counter to record an audio/video with a specific length.

        @type  _length: int
        @param _length: Recorded file length (in seconds).
        @type  callback: function
        @param callback: function that will be called after recording
        """
        time.sleep(_length)
        self.stop()
        if callable(callback):
            callback()

    def get_length(self):
        """Returns the audio file's length in nano-seconds.

        @rtype:  long integer
        @return: audio file's length in nano-seconds
        """
        # TODO - The playbin gst element returns -1 to all videos.
        if self.is_playing() or self.is_paused():
            query = self._playback_pipeline.query_duration(gst.FORMAT_TIME,
                                                         None)
            if query:
                return query[0]

    def get_current_position(self):
        """Returns the current position (in nano-seconds) of the playing file.

        @rtype:  long integer
        @return: current audio position in nano-seconds
        """
        if self.is_playing() or self.is_paused():
            query = self._playback_pipeline.query_position(gst.FORMAT_TIME,
                                                         None)
            if query:
                return query[0]

    def seek(self, value=None):
        """Seek for the 'value' position in the audio file.

        @type  value: number
        @param value: position number that will be seeked in the audio file
        """
        if self.is_playing() or self.is_paused():
            self._playback_pipeline.seek_simple(gst.FORMAT_TIME,
                                        gst.SEEK_FLAG_FLUSH, value)

    def get_volume(self):
        """Return the volume level of the played audio file.

        @rtype:  long integer
        @return: volume level of the played audio file
        """
        return self._playback_pipeline.get_property("volume")

    def set_volume(self, volume=None):
        """Set the volume level of the played audio file. The values range is
        from 0.0 to 10.0.

        @type  volume: double
        @param volume: number that set volume level, ranged from 0.0 to 10.0
        """
        if volume is None:
            raise EasyError("Volume not specified.")
        
        if volume <= 10.0 and volume >= 0.0:
            self._playback_pipeline.set_property("volume", volume)
        else:
            raise EasyError("The volume specified is out of range.")
        
    def _set_recorder_file(self, filename):
        """Set the name of the recorded file.

        @type  filename: string
        @param filename: name of the recorded file
        """
        self.sink_element = self._recorder_pipeline.get_by_name("sink")
        self.sink_element.set_property("location", filename)

    def _catch_EOS(self):
        """Get bus EOS(end of streamer) message (or error) and and stops the 
        pipeline. This method works without requires GLib/Gtk+ main loop.
        """
        message = None
        while not message:
            time.sleep(1)
            bus = self._playback_pipeline.get_bus()
            if bus:
                message = bus.poll(gst.MESSAGE_ERROR|gst.MESSAGE_EOS,
                                        timeout=1)
        self.stop()
        