#
# camera.py
# Easy API - camera module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""Camera module is composed of manipulation functions for the camera
device. These functions include recording, playing videos and taking
snapshots. Some examples of features include seek, file time, current time
position and changing video properties (brightness, contrast, hue and
saturation).
"""
from _playback_recorder import PlaybackRecorder
from easy.easy_exceptions import EasyError
import gst
import os
import pygst
import thread
import time
pygst.require("0.10")


__all__ = ["RESOLUTION_176x144", "RESOLUTION_320x240", "RESOLUTION_352x288",
           "RESOLUTION_640x480", "is_playing", "is_paused", "is_recording",
           "is_displaying", "is_stopped", "play", "display", "pause", "record",
           "stop", "stop_display", "click", "get_length", 
           "get_current_position", "seek", "configure", 
           "set_window_id", "get_volume", "set_volume"]

class Camera(PlaybackRecorder):
    """Camera module is composed of manipulation functions for the camera
    device. These functions include recording, playing videos and taking
    snapshots. Some examples of features include seek, file time, current time
    position and changing video properties (brightness, contrast, hue and
    saturation).

    The Camera module has two pipelines. The first one, the display pipeline,
    has the capability of displaying captured video stream from the camera
    device on the screen, recording stream as a video file. The display
    pipeline also allows taking snapshots. The second one, the playback
    pipeline, has the capability of playing videos. All those operations can be
    done in a synchronous or asynchronous way.
    """

    RESOLUTION_176x144 = (176,144)
    RESOLUTION_320x240 = (320,240)
    RESOLUTION_352x288 = (352,288)
    RESOLUTION_640x480 = (640,480)

    def __init__(self):
        """Constructor for Camera class.
        """
        PlaybackRecorder.__init__(self)
        self._DISPLAYING = False
        self._window_id = None
        
    def _init_playback_pipeline(self):
        self._vsink = gst.element_factory_make("xvimagesink")
        self._playback_pipeline = gst.parse_launch("playbin")
        self._playback_pipeline.set_property("video-sink", self._vsink)

    def _init_recorder_pipeline(self, resolution=RESOLUTION_640x480):
        """Build a GStreamer pipeline.
        
        @type  resolution: tuple
        @param resolution: tuple with the resolution's values
        """
        self._recorder_pipeline = gst.Pipeline("source_pipeline")

        source = gst.element_factory_make("gconfv4l2src", "source")
        filter = gst.element_factory_make("capsfilter", "source_filter")
        self._tee = gst.element_factory_make("tee", "tee")

        caps = gst.Caps("video/x-raw-yuv,width=%i,height=%i,framerate=8/1"
                        % resolution)

        self._recorder_pipeline.add(source)
        self._recorder_pipeline.add(filter)
        self._recorder_pipeline.add(self._tee)

        filter.set_property("caps", caps)

        source.link(filter)
        filter.link(self._tee)
        
    def _make_recorder_bin(self):
        """Build a GStreamer pipeline to record a video file.
        """
        self._recorder_bin = gst.Bin("recorder_bin")
        queue = gst.element_factory_make("queue", "queue")
        encoder = gst.element_factory_make("hantro4200enc", "encoder")
        muxer = gst.element_factory_make("avimux", "muxer")
        filter = gst.element_factory_make("capsfilter", "filter")
        self._filesink = gst.element_factory_make("gnomevfssink", "filesink")

        self._filesink.connect("allow-overwrite", lambda *x: True)

        caps = gst.Caps("video/x-raw-yuv")

        self._recorder_bin.add(queue)
        self._recorder_bin.add(filter)
        self._recorder_bin.add(encoder)
        self._recorder_bin.add(muxer)
        self._recorder_bin.add(self._filesink)

        queue.link(encoder)
        encoder.link(muxer)
        muxer.link(filter)
        filter.link(self._filesink)

        filter.set_property("caps", caps)

        pad = queue.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._recorder_bin.add_pad(ghostpad)        

    def _make_display_bin(self):
        """Build a GStreamer bin to display camera's image in the screen.
        """
        self._display_bin = gst.Bin("display_bin")

        self._sink = gst.element_factory_make("xvimagesink", "sink")
        self._display_bin.add(self._sink)

        pad = self._sink.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._display_bin.add_pad(ghostpad)

        self._recorder_pipeline.add(self._display_bin)

    def _make_snapshot_bin(self):
        """Build a GStreamer pipeline to take photographs.
        """
        self._snapshot_bin = gst.Bin("snapshot_bin")
        self.caps = gst.element_factory_make("ffmpegcolorspace", "caps")
        jpegenc = gst.element_factory_make("jpegenc", "jpegenc")
        self._filesink_photo = gst.element_factory_make("filesink",
                                                       "filesink_photo")

        self._snapshot_bin.add(self.caps)
        self._snapshot_bin.add(jpegenc)
        self._snapshot_bin.add(self._filesink_photo)

        self.caps.link(jpegenc)
        jpegenc.link(self._filesink_photo)

        pad = self.caps.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._snapshot_bin.add_pad(ghostpad)

        self._recorder_pipeline.add(self._snapshot_bin)

    def play(self, filename="file.avi"):
        """Start playing video file.

        @type   filename: string
        @param  filename: name of the recorded file
        """
        if filename and os.path.exists(filename):
            self.stop()            

            if self._window_id:
                self._vsink.set_xwindow_id(self._window_id)
                
            filename = os.path.abspath(filename)
            self._playback_pipeline.set_property("uri", "file://" + filename)
            self._playback_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True
            thread.start_new_thread(self._catch_EOS, ())
        else:
            raise EasyError("File does not exist.")
        
    def is_displaying(self):
        """Check if device is displaying.

        @rtype:  boolean
        @return: True, if device is displaying
        """
        return self._DISPLAYING

    def display(self, resolution=RESOLUTION_352x288):
        """Start showing camera display.

        @type  resolution: tuple
        @param resolution: resolution that camera will be started (176x144,
                           320x240, 352x288 or 640x480)
        """
        self._init_recorder_pipeline(resolution)
        self._make_display_bin()
        self._tee.link(self._display_bin)

        if self._window_id:
            self._sink.set_xwindow_id(self._window_id)

        self._recorder_pipeline.set_state(gst.STATE_PLAYING)

        self._DISPLAYING = True

    def stop_display(self):
        """Stop the current camera display.
        """
        if self._playback_pipeline:
            self._playback_pipeline.set_state(gst.STATE_NULL)
        if self._recorder_pipeline:
            self._recorder_pipeline.set_state(gst.STATE_NULL)
        try:
            self._tee.unlink(self._display_bin)
            self._DISPLAYING = False
        except: pass
            
    def stop(self):
        """Stop the current audio stream.
        """
        if self._playback_pipeline:
            self._playback_pipeline.set_state(gst.STATE_NULL)
        
        self._stop_recording()
        self.stop_display()
        
        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False
        self._DISPLAYING = False

    def _stop_recording(self):
        """Stop recording video.
    
        @type  callback: function
        @param callback: function that will be called as a callback after record
        """
        if self.is_recording():
            self._tee.unlink(self._recorder_bin)
            self._recorder_bin.set_state(gst.STATE_NULL)
            self._recorder_pipeline.remove(self._recorder_bin)
            self._RECORDING = False

    def record(self, filename="file.avi", record_time=-1, 
               resolution=RESOLUTION_352x288, callback=None):
        """Records a video captured by the camera

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file. If '-1' the 'stop' method must be called to break recording.
        @type   resolution: tuple
        @param  resolution: resolution that camera will record (176x144,
                            320x240, 352x288 or 640x480)
        @type     callback: function
        @param    callback: function that will be called as a callback after
                            record
        """
        self.stop()
        
        if not self.is_displaying():
            self.display(resolution=resolution)

        self._make_recorder_bin()
        self._recorder_pipeline.add(self._recorder_bin)
        self._tee.link(self._recorder_bin)
        
        # Needed time to record correctly
        time.sleep(0.4)
        
        PlaybackRecorder.record(self, filename, record_time, callback)
        
    def click(self, filename="pic.jpg"):
        """Capture an image via camera.

        @type  filename: string
        @param filename: name of the captured picture
        """
        if not self.is_displaying():
            self.display()

        self._make_snapshot_bin()
        self._set_photo_file(filename)
        self._recorder_pipeline.set_state(gst.STATE_PLAYING)
        
        # Needed time to record correctly
        time.sleep(0.4)

        self._tee.link(self._snapshot_bin)
        time.sleep(1)
        self._tee.unlink(self._snapshot_bin)

        self._snapshot_bin.set_state(gst.STATE_NULL)
        self._recorder_pipeline.remove(self._snapshot_bin)

    def configure(self, brightness=None, contrast=None, hue=None,
                  saturation=None):
        """Set image brightness, constrast, hue, saturation.

        @type  brightness: number
        @param brightness: integer number ranged from -1000 to 1000 that set
                           image brightness. Default: 0
        @type    contrast: number
        @param   contrast: integer number ranged from -1000 to 1000 that set
                           image contrast. Default: 0
        @type         hue: number
        @param        hue: integer number ranged from -1000 to 1000 that set
                           image hue. Default: 0
        @type  saturation: number
        @param saturation: integer number ranged from -1000 to 1000 that set
                           image saturation. Default: 0
        """
        if brightness != None:
            self._set_color_property("brightness", brightness)

        if contrast != None:
            self._set_color_property("contrast", contrast)

        if hue != None:
            self._set_color_property("hue", hue)

        if saturation != None:
            self._set_color_property("saturation", saturation)

    def set_window_id(self, window_id=None):
        """Set the window id to define the place where the video should be
        displayed.

        @type   window_id: integer
        @param  window_id: windows identifier that the image will be showed
        """
        self._window_id = window_id

    def _set_color_property(self, property, value):
        #TODO - maemo gstreamer doesn't allows this function
        """Set image color property.

        @type  property: string
        @param property: the property name (brightness, contrast, hue or sat-
                         uration)
        @type     value: number
        @param    value: integer number ranged from -1000 to 1000 that set ima-
                         ge color attribute. Default: 0
        """
        if value and (value < -1000 or value > 1000):
            raise EasyError("Value must is in range from -1000 to 1000.")

        self._playback_pipeline.set_state(gst.STATE_PAUSED)
        self._vsink.set_property(property, value)
        self._playback_pipeline.set_state(gst.STATE_PLAYING)

    def _set_recorder_file(self, filename):
        """Set the name of the recorded file.

        @type  filename: string
        @param filename: name of the recorded file
        """
        if not filename.lower().endswith('.avi'):
            filename += '.avi'
        
        self._filesink.set_property("location", filename)

    def _set_photo_file(self, filename):
        """Set the name of the recorded photo.

        @type  filename: string
        @param filename: name of the recorded file
        """
        if not (filename.lower().endswith('.jpg') or \
                filename.lower().endswith('.jpeg')):
            filename += '.jpg'
            
        self._filesink_photo.set_property("location", filename)

_camera = Camera()

# Camera functions/attributes
RESOLUTION_176x144 = _camera.RESOLUTION_176x144
RESOLUTION_320x240 = _camera.RESOLUTION_320x240
RESOLUTION_352x288 = _camera.RESOLUTION_352x288
RESOLUTION_640x480 = _camera.RESOLUTION_640x480
is_displaying = _camera.is_displaying
display = _camera.display
stop_display = _camera.stop_display
click = _camera.click
configure = _camera.configure
set_window_id = _camera.set_window_id

# Inherited/Overrided functions
is_playing = _camera.is_playing
is_paused = _camera.is_paused
is_recording = _camera.is_recording
is_stopped = _camera.is_stopped
play = _camera.play
pause = _camera.pause
stop = _camera.stop
record = _camera.record
get_length = _camera.get_length
get_current_position = _camera.get_current_position
seek = _camera.seek
get_volume = _camera.get_volume
set_volume = _camera.set_volume