#
# contact.py
# Easy API - contact module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""This module provides functions to manipulate contacts saved in maemo using the
evolution.ebook.EContact class. Changes in Contact are reflected in the
maemo contact data server.
"""
import evolution.ebook as evo
import re

class Contact(object):
    """This class manipulates contacts saved in maemo using the
    evolution.ebook.EContact class. Changes in Contact are reflected in the
    maemo contact data server.
    """

    def __init__(self, name="", econtact=None):
        """Constructor for Conctact class.

        @type      name: string
        @param     name: contact's name
        @type  econtact: object EContact
        @param econtact: ebook contact
        """
        self._book = ContactBook.get_instance()

        if not econtact:
            econtact = evo.EContact()

        self._econtact = econtact

        if not self._econtact.get_uid():
            self._book.add(self)

        if name:
            self._set_name(name)

    def get_econtact(self):
        """Get econtact.

        @rtype:  object EContact
        @return: ebook contact
        """
        return self._econtact

    def get_uid(self):
        """Get econtact UID.

        @rtype:  string
        @return: econtact UID
        """
        return self._econtact.get_uid()

    def _get_name(self):
        """Get contact's given name.

        @rtype:  string
        @return: contact's given name
        """
        return self._econtact.props.given_name

    def _set_name(self, given_name_):
        """Set contact's given name.

        @type  given_name_: string
        @param given_name_: contact's given name
        """
        self._econtact.props.given_name = given_name_
        self._book._update(self)

    def _get_family_name(self):
        """Get contact's family name.

        @rtype:  string
        @return: contact's family name
        """
        return self._econtact.props.family_name

    def _set_family_name(self, family_name_):
        """Set contact's family name.

        @type  family_name_: string
        @param family_name_: contact's family name
        """
        self._econtact.props.family_name = family_name_
        self._book._update(self)

    def _get_email(self):
        """Get contact's email.

        @rtype:  string
        @return: contact's email
        """
        return self._econtact.props.email_1

    def _set_email(self, email_):
        """Set contact's email.

        @type  email_: string
        @param email_: contact's email
        """
        self._econtact.props.email_1 = email_
        self._book._update(self)

    def _get_nickname(self):
        """Get contact's nickname.

        @rtype:  string
        @return: contact's nickname
        """
        return self._econtact.props.nickname

    def _set_nickname(self, nickname_):
        """Set contact's nickname.

        @type  nickname_: string
        @param nickname_: contact's nickname
        """
        self._econtact.props.nickname = nickname_
        self._book._update(self)

    def _get_gtalk(self):
        """Get contact's gtalk.

        @rtype:  string
        @return: contact's gtalk
        """
        return self._econtact.props.im_jabber_home_1

    def _set_gtalk(self, gtalk_):
        """Set contact's gtalk.

        @type  gtalk_: string
        @param gtalk_: contact's gtalk
        """
        self._econtact.props.im_jabber_home_1 = gtalk_
        self._book._update(self)

    name = property(_get_name, _set_name)
    family_name = property(_get_family_name, _set_family_name)
    email = property(_get_email, _set_email)
    nickname = property(_get_nickname, _set_nickname)
    gtalk = property(_get_gtalk, _set_gtalk)

class ContactBook(object):
    """This class manipulates the maemo contact data server through
    evolution.ebook.EContact class. ContactBook implements the Singleton
    pattern through the get_instance method. Changes in ContactBook are
    reflected in the maemo contact data server.
    """

    instance = None

    def __init__(self):
        """Constructor for ContactBook class.
        """
        self._book = evo.open_addressbook("default")
        self._contacts = {} # cache

        query = evo.EBookQuery.field_exists(evo.CONTACT_FULL_NAME)
        self._book_view = self._book.get_book_view(query)
        self._book_view.start()
        self._book_view.set_contacts_added_cb(self._contacts_added_cb)
        self._book_view.set_contacts_changed_cb(self._contacts_changed_cb)
        self._book_view.set_contacts_removed_cb(self._contacts_removed_cb)

        self._contacts_added_cb = None
        self._contacts_changed_cb = None
        self._contacts_removed_cb = None

    @classmethod

    def get_instance(cls):
        """Get ContactBook instance (singleton).

        @rtype:  object ContactBook
        @return: ContactBook
        """
        if cls.instance is None:
            cls.instance = ContactBook()
        return cls.instance

    def all(self):
        """Returns a list of all contacts.

        @rtype:  list
        @return: list of contacts
        """
        econtacts = self._book.get_all_contacts()
        self._contacts = {}

        for econtact in econtacts:
            contact = Contact(econtact=econtact)
            self._contacts[contact.get_uid()] = contact

        return self._contacts.values()

    def all_as_dict(self):
        """Returns a dictionary of all contacts in the book. Keys are the
        Contact UIDs.

        @rtype:  dictionary
        @return: dict of all contacts
        """
        self.all()
        return self._contacts

    def cache(self):
        """Returns a list of all contacts in cache.

        @rtype:  list
        @return: list of contacts
        """
        return self._contacts.values()

    def cache_as_dict(self):
        """Returns a dictionary of all contacts in the cache. Keys are the
        Contact UIDs.

        @rtype:  dictionary
        @return: dict of all contacts
        """
        return self._contacts

    def get(self, id_):
        """Get Contact by UID.

        @type  id_: string
        @param id_: Unique ID that identify a Contact

        @rtype:  object Contact
        @return: Contact
        """
        if id_ not in self._contacts:
            econtact = self._book.get_contact(id_)
            if econtact:
                self._contacts[id_] = Contact(econtact=econtact)
            else:
                return
        return self._contacts[id_]

    def add(self, contact_):
        """Add contact to ContactBook.

        @type  contact_: object Contact
        @param contact_: contact to add in ContactBook
        """
        econtact = contact_.get_econtact()
        id = self._book.add_contact(econtact)
        self._contacts[id] = contact_

    def _update(self, contact_):
        """Update contact.

        @type  contact_: object Contact
        @param contact_: contact to update
        """
        econtact = contact_.get_econtact()
        self._book.commit_contact(econtact)

    def exists(self, contact_):
        """Verify if the contact exist in ContactBook.

        @type  contact_: object Contact
        @param contact_: contact to verify

        @rtype:  boolean
        @return: True if contacts exists in ContactBook
        """
        econtact = contact_.get_econtact()
        return self._book.contact_exists(econtact)

    def remove(self, contact_):
        """Remove contact from contactBook.

        @type  contact_: object Contact
        @param contact_: contact to remove.
        """
        id = contact_.get_uid()
        self._book.remove_contact_by_id(id)

    def remove_by_id(self, id_):
        """Remove contact by UID.

        @type  id_: string
        @param id_: contact UID
        """
        self._book.remove_contact_by_id(id_)
        self._contacts.pop(id_)

    def find_by_name(self, name_):
        """Find contacts by name.

        @type  name_: string
        @param name_: contact name

        @rtype:  list
        @return: contact list
        """
        contact_list = self.all()
        result_list = []

        for contact in contact_list:
            name = contact.get_econtact().props.given_name
            if name and re.search(name_, name):
                result_list.append(contact)

        return result_list

    def _contacts_added_cb(self, bookview, econtacts):
        """Callback function called when a contact is added.

        @type  bookview:  object EBookView
        @param bookview:  book view
        @type  econtacts: list
        @param econtacts: EContact list
        """
        contacts = []
        for econtact in econtacts:
            contact = Contact(econtact=econtact)
            self._contacts[econtact.get_uid()] = contact
            contacts.append(contact)

        if callable(self._contacts_added_cb):
            self._contacts_added_cb(contacts)

    def _contacts_changed_cb(self, bookview, econtacts):
        """Callback function called when a contact is changed.

        @type  bookview:  object EBookView
        @param bookview:  book view
        @type  econtacts: list
        @param econtacts: EContact list

        """
        contacts = []
        for econtact in econtacts:
            contact = Contact(econtact=econtact)
            self._contacts[econtact.get_uid()] = contact
            contacts.append(contact)

        if callable(self._contacts_changed_cb):
            self._contacts_changed_cb(contacts)

    def _contacts_removed_cb(self, bookview, econtacts):
        """Callback function called when a contact is removed.

        @type  bookview:  object EBookView
        @param bookview:  book view
        @type  econtacts: list
        @param econtacts: list with the UIDs of removed contacts.
        """
        contacts = []
        for contact_id in econtacts:
            self._contacts.pop(contact_id)
            contacts.append(contact_id)

        if callable(self._contacts_removed_cb):
            self._contacts_removed_cb(contacts)

    def set_contacts_added_cb(self, callback):
        """Set callback function that will be called when a contact is added.

        @type  callback: function
        @param callback: function
        """
        self._contacts_added_cb = callback

    def set_contacts_changed_cb(self, callback):
        """Set callback function that will be called when a contact is changed.

        @type  callback: function
        @param callback: function
        """
        self._contacts_changed_cb = callback

    def set_contacts_removed_cb(self, callback):
        """Set callback function that will be called when a contact is removed.

        @type  callback: function
        @param callback: function
        """
        self._contacts_removed_cb = callback

create_contact = Contact