#
# bluetooth.py
# Easy API - bluetooth module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

""" Bluetooth module is composed  of bluetooth's device manipulation. This
module uses LightBlue a cross-platform Python Bluetooth API which provides
simple access to Bluetooth operations (http://lightblue.sourceforge.net/).
"""
import lightblue as bluetooth

from easy.easy_exceptions import EasyError

__all__ = ["finddevices", "findservices", "finddevicename", "gethostaddr", 
           "gethostclass", "socket", "stopadvertise", "obex", "sendfile",
           "selectdevice", "selectservice"]


def sendfile(address, channel, source):
    """Sends a file to a remote device.Raises BluetoothError if an error
    occurs.

    @type  address: string
    @param address: the address of the remote device
    @type  channel: number
    @param channel: the RFCOMM channel of the remote OBEX service.
    @type  channel: string
    @param source: a filename or file-like object, containing the data to be
        sent. If a file object is given, it must be opened for reading.

    """
    try:
        bluetooth.obex.sendfile(address, channel, source)
    except Exception, e:
        raise EasyError("Check if the Bluetooth is activated or if the " \
        "remote device is still available")
        
def finddevices(getnames=True, length=10):
    """Performs a device discovery and returns the found devices as a list of 
    (address, name, class-of-device) tuples. Raises BluetoothError if an error
    occurs.


    @type  getnames: boolean
    @param getnames: True if device names should be retrieved during 
        discovery. If false, None will be returned instead of the device
        name.
    @type  length: number
    @param length: the number of seconds to spend discovering devices. 

       
    @rtype:  list of tuples
    @return: returns the found devices as a list of 
        (address, name, class-of-device) tuples.
    """
    try:
        return bluetooth.finddevices(getnames, length)
    except Exception, e:
        raise EasyError("Check if the Bluetooth is activated.")
        
def findservices(addr=None, name=None, servicetype=None):
    """Performs a service discovery and returns the found services as a list
    of (device-address, service-port, service-name) tuples. Raises 
    BluetoothError if an error occurs.

    @type  addr: string
    @param addr: a device address, to search only for services on a specific 
        device
    @type  name: unicode
    @param name: a service name string, to search only for a service with a
        specific name
    @type  servicetype: unicode
    @param servicetype: can be RFCOMM or OBEX to search only for RFCOMM or
        OBEX-type services. (OBEX services are not returned from an RFCOMM
        search)

       
    @rtype:  list of tuples
    @return: returns the found services as a list
        of (device-address, service-port, service-name) tuples.
    """
    try:
        return bluetooth.findservices(addr, name, servicetype)
    except Exception, e:
        raise EasyError("Check if the Bluetooth is activated or if the " \
        "remote device is still available")
    
finddevicename = bluetooth.finddevicename
gethostaddr = bluetooth.gethostaddr
gethostclass = bluetooth.gethostclass
obex = bluetooth.obex
selectdevice = bluetooth.selectdevice
selectservice = bluetooth.selectservice
socket = bluetooth.socket
stopadvertise = bluetooth.stopadvertise