/*
 * Copyright (C) 2008, 2009 Andrew Sichevoi.
 *
 * This file is part of Conler (http://thekondor.net/conler).
 *
 * Conler is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Conler is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Conler. If not, see <http://www.gnu.org/licenses/>.
 */

#include "ui-utils.h"
#include "debug.h"

#include <stdarg.h>

#include <gtk/gtk.h>
#include <glib/glist.h>

gboolean get_current_tree_iter(GtkTreeView* tree_view,
			       GtkTreeModel** model,
			       GtkTreeIter* iter)
{
    GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

    *model = gtk_tree_view_get_model(tree_view);

    if (!gtk_tree_selection_get_selected(selection, model, iter))
	creturn(FALSE, "Can't get selection!");

    return TRUE;
}

gboolean is_next_tree_iter_sibling(GtkTreeModel* model, GtkTreeIter* iter)
{
    GtkTreeIter* next_iter = gtk_tree_iter_copy(iter);
    gboolean next = gtk_tree_model_iter_next(model, next_iter);

    gtk_tree_iter_free(next_iter);

    return next;
}

gboolean is_prev_tree_iter_sibling(GtkTreeModel* model, GtkTreeIter* iter)
{
    GtkTreePath* path = gtk_tree_model_get_path(model, iter);
    gboolean prev = gtk_tree_path_prev(path);

    gtk_tree_path_free(path);

    return prev;
}

GtkTreeIter* get_next_tree_iter(GtkTreeModel* model, GtkTreeIter* iter)
{
    GtkTreeIter* next_iter = gtk_tree_iter_copy(iter);
    gboolean next = gtk_tree_model_iter_next(model, next_iter);

    if (next)
        return next_iter;

    gtk_tree_iter_free(next_iter);
    return NULL;
}

GtkTreeIter* get_prev_tree_iter(GtkTreeModel* model, GtkTreeIter* iter)
{
    GtkTreePath* path = gtk_tree_model_get_path(model, iter);
    gboolean prev = gtk_tree_path_prev(path);
    GtkTreeIter* next_iter;

    if (!prev) {
        gtk_tree_path_free(path);
        return NULL;
    }

    // Looks dirty because of stupid API
    next_iter = gtk_tree_iter_copy(iter);
    if (!gtk_tree_model_get_iter(model, next_iter, path)) {
        gtk_tree_iter_free(next_iter);
        gtk_tree_path_free(path);
        return NULL;
    }

    gtk_tree_path_free(path);

    return next_iter;
}

gchar* tree_iter_to_path_string(GtkTreeModel* model, GtkTreeIter* iter)
{
    GtkTreePath* path = gtk_tree_model_get_path(model, iter);
    gchar* path_string = gtk_tree_path_to_string(path);

    gtk_tree_path_free(path);

    return path_string;
}

gboolean path_string_to_tree_iter(GtkTreeModel* model, gchar* path_string,
                                                       GtkTreeIter* iter)
{
    GtkTreePath* path = gtk_tree_path_new_from_string(path_string);
    gboolean result = TRUE;

    if (!gtk_tree_model_get_iter(model, iter, path))
        result = FALSE;

    gtk_tree_path_free(path);

    return result;
}

/* if prev is FALSE, swapping is performed in the back direction */
gboolean swap_list_store_items(GtkTreeModel* tree_model, GtkTreeIter* iter,
                                                         const gboolean prev)
{
    GtkTreeIter* neighbour_iter;

    if (prev)
        neighbour_iter = get_prev_tree_iter(tree_model, iter);
    else
        neighbour_iter = get_next_tree_iter(tree_model, iter);

    if (!neighbour_iter)
        creturn(FALSE, "No neighbour iterator");

    gtk_list_store_swap(GTK_LIST_STORE(tree_model), iter, neighbour_iter);

    gtk_tree_iter_free(neighbour_iter);

    return TRUE;
}

GtkWidget* show_add_or_edit_entity_basic_dialog(GtkWindow* parent,
						gchar* title,
						gchar* ok_button_label,
						gchar* cancel_button_label,
						GtkWidget* input_hbox,
						...)
{
    GtkWidget* dialog;
    GtkWidget* input_vbox = gtk_vbox_new(FALSE, 0);
    va_list ap;

    dialog = gtk_dialog_new_with_buttons(
                              title, parent,
                              GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			      ok_button_label, GTK_RESPONSE_OK,
			      cancel_button_label, GTK_RESPONSE_CANCEL, NULL);

    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
				input_hbox);

    va_start(ap, input_hbox);
    while (TRUE) {
	GtkWidget* input_item = va_arg(ap, GtkWidget *);
	if (!input_item)
	    break;

	gtk_box_pack_start_defaults(GTK_BOX(input_vbox), input_item);
    }
    va_end(ap);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), input_vbox);

    return dialog;
}


void resize_window_by_delta(GtkWindow* window, gint delta_w, gint delta_h)
{
    gint width, height;

    gtk_window_get_size(window, &width, &height);
    gtk_window_resize(window, width + delta_w, height + delta_h);
}

GtkWidget* make_label_with_markup(const gchar* text)
{
    GtkWidget* label = gtk_label_new(NULL);

    gtk_label_set_markup(GTK_LABEL(label), text);

    return label;
}

GtkWidget* make_check_button(const gchar* label, gboolean toggled)
{
    GtkWidget* check_button;

    check_button = gtk_check_button_new_with_mnemonic(label);
    g_object_set(G_OBJECT(check_button), "active", toggled, NULL);

    return check_button;
}
