/*
 * Copyright (C) 2008, 2009 Andrew Sichevoi.
 *
 * This file is part of Conler (http://thekondor.net/conler).
 *
 * Conler is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Conler is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Conler. If not, see <http://www.gnu.org/licenses/>.
 */

#include "slots.h"
#include "types.h"
#include "common.h"
#include "const.h"
#include "ui.h"
#include "ui-utils.h"
#include "debug.h"

#include <glib-object.h>
#include <glib/gstdio.h>

#include <gtk/gtk.h>

#ifdef MAEMO
  #include <hildon/hildon.h>
  #include <hildon/hildon-help.h>
  #include <libosso.h>
  #include <gdk/gdkkeysyms.h>
#endif

#define W(wdg)  app_data->wdg


// static gboolean set_tree_view_cursor(gpointer);

static void connect_toolbutton_signals(AppData_s*);
static void close_application(GtkWidget*, GdkEvent*, gpointer);
#ifdef MAEMO
  static gboolean key_pressed(GtkWidget*, GdkEventKey*, gpointer);
  static void help_activated(GtkWidget*, gpointer);
#endif
static void quit_app_toolbutton_clicked(GtkWidget*, gpointer);
static void apply_changes(GtkWidget*, gpointer);
static void ap_disabled_state_changed(GtkCellRenderer*, gchar*, gpointer);
static void ap_name_changed(GtkCellRenderer*, gchar*, gchar*, gpointer);
static void ap_selection_changed(GtkWidget*, gpointer);
static void cmd_disabled_state_changed(GtkCellRenderer*, gchar*, gpointer);
static void cmd_superuser_env_state_changed(GtkCellRenderer*, gchar*, gpointer);
static void cmdline_changed(GtkCellRenderer*, gchar*, gchar*, gpointer);
static void cmd_selection_changed(GtkWidget*, gpointer);

static void add_ap_toolbutton_clicked(GtkWidget*, gpointer);
static void disable_ap_toolbutton_clicked(GtkWidget*, gpointer);
static void del_ap_toolbutton_clicked(GtkWidget*, gpointer);
static void rename_ap_toolbutton_clicked(GtkWidget*, gpointer);
static void add_cmd_toolbutton_clicked(GtkWidget*, gpointer);
static void del_cmd_toolbutton_clicked(GtkWidget*, gpointer);
static void move_up_cmd_toolbutton_clicked(GtkWidget*, gpointer);
static void move_down_cmd_toolbutton_clicked(GtkWidget*, gpointer);
static void edit_cmd_toolbutton_clicked(GtkWidget*, gpointer);


// TODO: move to utils?
static void update_ap_model_commands(AppData_s*, GtkTreeIter*);


void connect_signals(AppData_s* app_data)
{
    g_signal_connect(W(mainWindow), "delete_event",
                     G_CALLBACK(close_application), app_data);

    g_signal_connect(W(mainWindow), "destroy_event",
                     G_CALLBACK(close_application), app_data);

#ifdef MAEMO
    g_signal_connect(W(mainWindow), "key_press_event",
                     G_CALLBACK(key_pressed), app_data);
    g_signal_connect(W(helpToolButton), "clicked",
		     G_CALLBACK(help_activated), app_data);
#endif

    /* AP */
    g_object_set(W(ap.enabledCellRndr), "mode",
                 GTK_CELL_RENDERER_MODE_ACTIVATABLE, NULL);
    g_signal_connect(W(ap.enabledCellRndr), "activate",
                     G_CALLBACK(ap_disabled_state_changed), app_data);

    g_object_set(W(ap.nameCellRndr), "editable", TRUE, NULL);
    g_signal_connect(W(ap.nameCellRndr), "edited",
                     G_CALLBACK(ap_name_changed), app_data);

    GtkTreeSelection* apListSelection = gtk_tree_view_get_selection(
                                            GTK_TREE_VIEW(W(apListView)));
    g_signal_connect(apListSelection, "changed",
                     G_CALLBACK(ap_selection_changed), app_data);

    /* Command */
    g_object_set(W(cmd.disabledCellRndr), "mode",
		 GTK_CELL_RENDERER_MODE_ACTIVATABLE, NULL);
    g_signal_connect(W(cmd.disabledCellRndr), "activate",
		     G_CALLBACK(cmd_disabled_state_changed), app_data);
    g_object_set(W(cmd.superuserCellRndr), "mode",
		 GTK_CELL_RENDERER_MODE_ACTIVATABLE, NULL);
    g_signal_connect(W(cmd.superuserCellRndr), "activate",
		     G_CALLBACK(cmd_superuser_env_state_changed), app_data);

    g_object_set(W(cmd.cmdlineCellRndr), "editable", TRUE, NULL);
    g_signal_connect(W(cmd.cmdlineCellRndr), "edited",
                     G_CALLBACK(cmdline_changed), app_data);

    GtkTreeSelection* cmdListSelection = gtk_tree_view_get_selection(
                                            GTK_TREE_VIEW(W(commandListView)));
    g_signal_connect(cmdListSelection, "changed",
                     G_CALLBACK(cmd_selection_changed), app_data);

    connect_toolbutton_signals(app_data);
}

static void connect_toolbutton_signals(AppData_s* app_data)
{
    g_signal_connect(W(quitToolButton), "clicked",
                     G_CALLBACK(quit_app_toolbutton_clicked), app_data);
    g_signal_connect(W(applyToolButton), "clicked",
                     G_CALLBACK(apply_changes), app_data);

    g_signal_connect(W(ap.disableToolButton), "clicked",
                     G_CALLBACK(disable_ap_toolbutton_clicked), app_data);
    g_signal_connect(W(ap.addToolButton), "clicked",
                     G_CALLBACK(add_ap_toolbutton_clicked), app_data);
    g_signal_connect(W(ap.delToolButton), "clicked",
                     G_CALLBACK(del_ap_toolbutton_clicked), app_data);
    g_signal_connect(W(ap.renameToolButton), "clicked",
                     G_CALLBACK(rename_ap_toolbutton_clicked), app_data);

    g_signal_connect(W(cmd.addToolButton), "clicked",
                     G_CALLBACK(add_cmd_toolbutton_clicked), app_data);
    g_signal_connect(W(cmd.delToolButton), "clicked",
                     G_CALLBACK(del_cmd_toolbutton_clicked), app_data);
    g_signal_connect(W(cmd.moveUpToolButton), "clicked",
                     G_CALLBACK(move_up_cmd_toolbutton_clicked), app_data);
    g_signal_connect(W(cmd.moveDownToolButton), "clicked",
                     G_CALLBACK(move_down_cmd_toolbutton_clicked), app_data);
    g_signal_connect(W(cmd.editToolButton), "clicked",
                     G_CALLBACK(edit_cmd_toolbutton_clicked), app_data);
}

static void ap_disabled_state_changed(GtkCellRenderer* cell_renderer,
                                      gchar* path_string, gpointer data)
{
    GET_APP_DATA(data);
    gboolean state;

    inverse_ap_disabled_field(path_string, data, &state);
    SET_STOCK_ICON(W(ap.disableToolButton), state, NO, YES);

    W(changed) = TRUE;
}

static void ap_name_changed(GtkCellRenderer* cell_renderer, gchar* path_string,
                            gchar* new_ap_name, gpointer data)
{
    GtkTreeIter iter;
    GET_APP_DATA(data)
    GtkTreeModel* model = gtk_tree_view_get_model(GTK_TREE_VIEW(W(apListView)));
    GtkTreePath* path = gtk_tree_path_new_from_string(path_string);

    if (!gtk_tree_model_get_iter(model, &iter, path)) 
        ccreturn("Can't get an iter for current\n");

    AccessPoint_s* ap;
    gtk_tree_model_get(model, &iter, 0, &ap, -1);
    g_free(ap->name);
    ap->name = g_strdup(new_ap_name);
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, ap, -1);

    free_access_point(ap);
    gtk_tree_path_free(path);

    W(changed) = TRUE;
}

static void ap_selection_changed(GtkWidget* selection, gpointer data)
{
// TODO: don't process selection if ap-state pixbuf clicked
    GtkTreeModel* model;
    GtkTreeIter iter;
    GET_APP_DATA(data)
    GtkTreeModel* cmd_model = gtk_tree_view_get_model(
                                    GTK_TREE_VIEW(W(commandListView)));
    AccessPoint_s* ap;

    if (W(cmd.changed)) {
        W(changed) = TRUE;
        update_ap_model_commands(app_data, W(ap.currentIter));
    }

    /* Use-Case: all APs are deleted */
    if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection),
                                                         &model, &iter)) {
        enable_ap_general_toolbuttons(app_data, FALSE);
        enable_cmd_toolbuttons(app_data, FALSE);
        return;
    }
    enable_ap_general_toolbuttons(app_data, TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(W(cmd.addToolButton)), TRUE);
   
 
//    enable_cmd_toolbuttons(app_data, TRUE);
//    gtk_widget_set_sensitive(gtk_widget(w(cmd.addtoolbutton)), true);
//    enable_ap_general_toolbuttons(app_data, TRUE);
    // TODO: make an alias for gtk_widget_set_sensitive(), e.g.
    //       @define enable_tool_button(toolbutton, enable)

    gtk_tree_model_get(model, &iter, 0, &ap, -1);
    DBG("ap = %s\n", ap->name);

    SET_STOCK_ICON(W(ap.disableToolButton), ap->disabled, NO, YES);
    fill_cmd_list_view(GTK_LIST_STORE(cmd_model), ap->commands);

    free_access_point(ap);
    // TODO: what about g_object_unref to the model?

    W(ap.currentIter) = gtk_tree_iter_copy(&iter);
    W(cmd.changed) = FALSE;
}

// TODO: copy&paste of 'ap_disabled_state_changed()"
static void cmd_disabled_state_changed(GtkCellRenderer* cell_renderer,
				       gchar* path_string, gpointer data)
{
    GET_APP_DATA(data);
    inverse_cmd_disabled_field(path_string, data, NULL);

    W(cmd.changed) = TRUE;
}

static void cmd_superuser_env_state_changed(GtkCellRenderer* cell_renderer,
					    gchar* path_string, gpointer data)
{
    GET_APP_DATA(data);
    inverse_cmd_superuser_env_field(path_string, data, NULL);

    W(cmd.changed) = TRUE;
}

static void cmd_selection_changed(GtkWidget* selection, gpointer data)
{
    GtkTreeModel* model;
    GtkTreeIter iter;
    GET_APP_DATA(data);
    gboolean next_cmd, prev_cmd;

    if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection),
                                         &model, &iter)) {
        enable_cmd_toolbuttons(app_data, FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(W(cmd.addToolButton)), TRUE);
        return;
    }

    enable_cmd_toolbuttons(app_data, TRUE);

    next_cmd = is_next_tree_iter_sibling(model, &iter);
    prev_cmd = is_prev_tree_iter_sibling(model, &iter);

    gtk_widget_set_sensitive(GTK_WIDGET(W(cmd.moveDownToolButton)), next_cmd);
    gtk_widget_set_sensitive(GTK_WIDGET(W(cmd.moveUpToolButton)), prev_cmd);
}


static void cmdline_changed(GtkCellRenderer* cell_renderer, gchar* path_string,
                            gchar* new_cmd, gpointer data)
{
    GtkTreeIter iter;
    GET_APP_DATA(data);
    GtkTreeModel* model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                    W(commandListView)));
    GtkTreePath* path = gtk_tree_path_new_from_string(path_string);

    if (!gtk_tree_model_get_iter(model, &iter, path)) {
	gtk_tree_path_free(path);
        ccreturn("Can't get an iter for current\n");
    }

    Command_s* cmd;
    gtk_tree_model_get(model, &iter, 0, &cmd, -1);
    g_free(cmd->command);

    cmd->command = g_strdup(new_cmd);
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, cmd, -1);

    free_command(cmd);
    gtk_tree_path_free(path);

    W(cmd.changed) = TRUE;
}


/* Makes synchronization between commands in commandListView and AP model */
static void update_ap_model_commands(AppData_s* app_data, GtkTreeIter* ap_iter)
{
    GtkTreeModel* ap_model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                        W(apListView)));
    GtkTreeModel* cmd_model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                        W(commandListView)));
    AccessPoint_s* ap = NULL;

    gtk_tree_model_get(ap_model, ap_iter, 0, &ap, -1);
    free_command_list(ap->commands);
    ap->commands = get_cmd_gslist_from_model(cmd_model);

#ifdef DEBUG
    for (GSList* iter = ap->commands; iter; iter = g_slist_next(iter))
        DBG("* cmd = %s", CMD_LIST_CAST(iter)->command);
#endif

    gtk_list_store_set(GTK_LIST_STORE(ap_model), ap_iter, 0, ap, -1);

    free_access_point(ap);

    DBG2("Sync AP Model <-> commandListView");
}

static void add_ap_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                    W(apListView)));
    GtkTreeIter iter;
    AccessPoint_s* ap;
    gboolean to_append = FALSE;

    if (!show_add_ap_dialog(GTK_WINDOW(W(mainWindow)), FALSE, &ap))
	return;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(apListView)), &model, &iter))
	to_append = TRUE;

    if (to_append)
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
    else
    {
	GtkTreeIter new_iter;

	gtk_list_store_insert_after(GTK_LIST_STORE(model), &new_iter, &iter);
	iter = new_iter;
    }

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, ap, -1);
    select_treeview_row(W(apListView), &iter);

    free_access_point(ap);

    W(changed) = TRUE;
}

static void del_ap_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeModel* cmd_model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                        W(commandListView)));
    GtkTreeIter iter;
    GtkTreeIter* neighbour_iter;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(apListView)), &model, &iter))
	ccreturn("Can't get current iter!");

    neighbour_iter = get_next_tree_iter(model, &iter);
    if (!neighbour_iter)
        neighbour_iter = get_prev_tree_iter(model, &iter);

    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_list_store_clear(GTK_LIST_STORE(cmd_model));

    if (neighbour_iter) {
        select_treeview_row(W(apListView), neighbour_iter);
        gtk_tree_iter_free(neighbour_iter);
    }

    W(changed) = TRUE;
}

static void disable_ap_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeIter iter;
    gchar* path_string;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(apListView)), &model, &iter))
	return;

    path_string = tree_iter_to_path_string(model, &iter);

    g_signal_emit_by_name(G_OBJECT(W(ap.enabledCellRndr)), "activate",
                                                  path_string, data);

    g_free(path_string);
}

static void rename_ap_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeIter iter;
    AccessPoint_s* ap;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(apListView)), &model, &iter))
	return;

    gtk_tree_model_get(model, &iter, 0, &ap, -1);
    if (!show_add_ap_dialog(GTK_WINDOW(W(mainWindow)), TRUE, &ap)) {
	free_access_point(ap);
	return;
    }

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, ap, -1);

    free_access_point(ap);

    W(changed) = TRUE;
}


static void add_cmd_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeIter iter;
    Command_s* cmd;
    gboolean to_append = FALSE;

    if (!show_add_command_dialog(GTK_WINDOW(W(mainWindow)), FALSE, &cmd))
	return;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(commandListView)),
			       &model, &iter))
	to_append = TRUE;

    if (to_append)
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
    else
    {
	GtkTreeIter new_iter;

	gtk_list_store_insert_after(GTK_LIST_STORE(model), &new_iter, &iter);
	iter = new_iter;
    }

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, cmd, -1);
    select_treeview_row(W(commandListView), &iter);

    free_command(cmd);

    W(cmd.changed) = TRUE;
}

// TODO: remove copy&paste; similar code 'edit_ap_name_toolbutton_clicked'
static void edit_cmd_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeIter iter;
    Command_s* cmd;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(commandListView)),
			       &model, &iter))
	ccreturn("Can't get current iter!");

    gtk_tree_model_get(model, &iter, 0, &cmd, -1);
    if (!show_add_command_dialog(GTK_WINDOW(W(mainWindow)), TRUE, &cmd)) {
	free_command(cmd);
	return;
    }

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, cmd, -1);

    free_command(cmd);
    W(cmd.changed) = TRUE;
}

// TODO: remove copy&paste; similar code in 'del_ap_toolbutton_clicked'
static void del_cmd_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeModel* model;
    GtkTreeIter iter;
    GtkTreeIter* next_iter;

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(commandListView)),
			       &model, &iter))
        ccreturn("Can't get current iter!");

    next_iter = get_next_tree_iter(model, &iter);
    if (!next_iter)
        next_iter = get_prev_tree_iter(model, &iter);

    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);

    if (next_iter) {
        select_treeview_row(W(commandListView), next_iter);
        gtk_tree_iter_free(next_iter);
    }

    W(cmd.changed) = TRUE;
}

// TODO: don't like this implementation (w/a). should be revised
static void move_up_cmd_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeIter iter;
    GtkTreeModel* model;
    GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(
 							  W(commandListView)));

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(commandListView)),
			       &model, &iter))
	ccreturn("Can't get current iter!");

    (void)swap_list_store_items(model, &iter, TRUE);
    cmd_selection_changed((GtkWidget *)selection, data);

    W(cmd.changed) = TRUE;
}

static void move_down_cmd_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    GtkTreeIter iter;
    GtkTreeModel* model;
    GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(
 							  W(commandListView)));

    if (!get_current_tree_iter(GTK_TREE_VIEW(W(commandListView)),
			       &model, &iter))
	ccreturn("Can't get current iter!");

    (void)swap_list_store_items(model, &iter, FALSE);
    cmd_selection_changed((GtkWidget *)selection, data);

    W(cmd.changed) = TRUE;
}

#ifdef MAEMO
static gboolean key_pressed(GtkWidget* widget, GdkEventKey* event, gpointer data)
{
    GET_APP_DATA(data);

    switch (event->keyval) {
        case GDK_F6:
                    W(fullscreen) = !(W(fullscreen));
                    if (!(W(fullscreen)))
                        gtk_window_unfullscreen(GTK_WINDOW(W(mainWindow)));
                    else
                        gtk_window_fullscreen(GTK_WINDOW(W(mainWindow)));

                    return TRUE;
    }

    return FALSE;
}

static void help_activated(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    // NOTE: using undocumented flag == 0, which launch
    //       help in the system's help browser instead
    //       topic help dialog
    (void)hildon_help_show(W(osso_context), HELP_TOPIC_ID, 0);
}
#endif

static void quit_app_toolbutton_clicked(GtkWidget* widget, gpointer data)
{
    close_application(NULL, NULL, data);
}

static void close_application(GtkWidget* widget, GdkEvent* event, gpointer data)
{
    GET_APP_DATA(data);

    if (W(changed) || W(cmd.changed)) {
        // TODO: make Ok, No, Cancel buttons
        gint response;
        GtkWidget* confirm_save_dialog = gtk_message_dialog_new(
                                        GTK_WINDOW(W(mainWindow)),
                                        GTK_DIALOG_MODAL,
                                        GTK_MESSAGE_QUESTION,
                                        GTK_BUTTONS_YES_NO,
                                        "Save changes before exit?");
        gtk_window_set_title(GTK_WINDOW(confirm_save_dialog), "Save Changes?");

        response = gtk_dialog_run(GTK_DIALOG(confirm_save_dialog));
        if (GTK_RESPONSE_YES == response)
            apply_changes(NULL, data);

        gtk_widget_destroy(confirm_save_dialog);
    }

    gtk_main_quit();
}

static void apply_changes(GtkWidget* widget, gpointer data)
{
    GET_APP_DATA(data);

    ConlerCfg_s cfg;
    GtkTreeModel* ap_model = gtk_tree_view_get_model(GTK_TREE_VIEW(
                                                        W(apListView)));

    if (!W(changed) && !W(cmd.changed))
        ccreturn("NO SAVE is required");

    /* Use-Case: commands has been updated:
     *   - No other AP has not been selected
     *   - Save/Quit button has been pressed after changes
     */
//   if (W(cmd.changed) && !W(changed))
    // TODO: not optimal, research
    if (W(cmd.changed))
        update_ap_model_commands(app_data, W(ap.currentIter));

    cfg.version = g_strdup(PROGRAM_VERSION);
    cfg.access_points = get_ap_gslist_from_model(ap_model);

#ifdef DEBUG
//    for (GSList* iter = cfg.access_points; iter; iter = g_slist_next(iter))
//	dump_ap("To save:", AP_LIST_CAST(iter), TRUE);
#endif

    DBG2("Action: b/writing config out");
    write_cfg(CONFIG_PATH, &cfg);
    DBG2("Action: a/writing config out");
    free_cfg(&cfg);

#ifdef MAEMO
    hildon_banner_show_information(GTK_WIDGET(W(mainWindow)), NULL,
                                   "Configuration saved");
#endif

    DBG("*** SAVE CONFIGURATION FILE ***");

    W(changed) = FALSE;
    W(cmd.changed) = FALSE;
}


