// (c) 2008, Andrew V. Sichevoi
// http://thekondor.net/conler

#include <glib.h>

#include <libxml/tree.h>
#include <libxml/xmlwriter.h>

#include <string.h>
#include <stdio.h>

#include "common.h"
#include "const.h"
#include "types.h"
#include "debug.h"

/* Continues iteration if it is not a desired node */
#define SKIP_NODE_PROCESSING(node, node_name) \
            if (xmlStrcmp((xmlChar *)node->name, BAD_CAST(node_name)))\
                continue;

#define SKIP_NODETYPE_PROCESSING(node, node_type) \
            if (node_type != node->type)\
                continue;

static xmlNodePtr get_element_node(const xmlNodePtr const, const char* const);
static AccessPoint_s* node_as_ap(const xmlDocPtr const, const xmlNodePtr const);
static gboolean write_ap_cfg(xmlTextWriterPtr, GSList*);


MaemoVersion get_maemo_version()
{
    FILE* version_file = g_fopen(OSSO_SOFTWARE_VERSION, "r");
    static const int BUF_SIZE = 64;
    char contents[BUF_SIZE];
    MaemoVersion version = UNKNOWN_VERSION;

    if (version_file) {
        if (fgets(contents, BUF_SIZE - 1, version_file)) {
            contents[BUF_SIZE] = 0;
            version = g_strrstr(contents, DIABLO_IDENT) ? DIABLO : CHINOOK;
        }

        fclose(version_file);
    }
    
    return version;
}


// TODO: return error code
// TODO: s/CFG_RETURN/R_CFG_RETURN/g
gboolean read_cfg(const gchar* const filename, ConlerCfg_s* cfg)
{
#undef CFG_RETURN
#define CFG_RETURN(reason)  { debug_msg("err = %s\n", reason);\
                              xmlFreeDoc(doc);  return FALSE; }
    xmlNodePtr cur_node;
    xmlDocPtr doc;

    doc = xmlParseFile(filename);
    if (!doc)
        return FALSE;

    cur_node = xmlDocGetRootElement(doc);
    if (!cur_node)
        CFG_RETURN("Empty document");

    if (xmlStrcmp(cur_node->name, BAD_CAST(ROOT_CFG_ELEMENT)))
        CFG_RETURN("Document wrong type");

    /* Get version of configuration */
    cfg->version = (gchar *)xmlGetProp(cur_node, BAD_CAST(VERSION_CFG_ATTR));

    cur_node = get_element_node(cur_node, APS_CFG_ELEMENT);
    if (!cur_node)
        CFG_RETURN("No APs are defined");

    cfg->access_points = NULL;
 
    /* Collect APs */
    for (cur_node = cur_node->xmlChildrenNode; cur_node; cur_node = cur_node->next) {
        SKIP_NODE_PROCESSING(cur_node, AP_CFG_ELEMENT);

        AccessPoint_s* ap = node_as_ap(doc, cur_node); // TODO: don't use 'doc'
        if (ap)
            APPEND2SLIST(cfg->access_points, ap);
    }

    xmlFreeDoc(doc);
    return TRUE;
}


// TODO: code error return
gboolean write_cfg(const gchar* const filename, const ConlerCfg_s* const cfg)
{
#undef W_CFG_RETURN_IF_FAIL
#define W_CFG_RETURN_IF_FAIL(rc) if (rc < 0) { xmlFreeTextWriter(writer);\
                                               xmlBufferFree(buf); return FALSE;}
    xmlTextWriterPtr writer;
    xmlBufferPtr buf;
    int rc;

    buf = xmlBufferCreate();
    g_return_val_if_fail(buf != NULL, FALSE);

    writer = xmlNewTextWriterMemory(buf, 0);
    if (!writer) {
        xmlBufferFree(buf);
        return FALSE;
    }

    rc = xmlTextWriterStartDocument(writer, NULL, "utf-8", NULL); 
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterWriteComment(writer,
                          BAD_CAST("This file is generated automatically. "
                                   "Please don't edit it manually, use a "
                                   "special gui wizard"));

    rc = xmlTextWriterStartElement(writer, BAD_CAST(ROOT_CFG_ELEMENT));
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterWriteAttribute(writer, BAD_CAST(VERSION_CFG_ATTR),
                                             BAD_CAST(cfg->version));
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterStartElement(writer, BAD_CAST(APS_CFG_ELEMENT));
    W_CFG_RETURN_IF_FAIL(rc);

    if (!write_ap_cfg(writer, cfg->access_points))
        W_CFG_RETURN_IF_FAIL(-1);

    /* Close 'access-points' element */
    rc = xmlTextWriterEndElement(writer);
    W_CFG_RETURN_IF_FAIL(rc);
   
    /* Close 'conler-config' element */
    rc = xmlTextWriterEndElement(writer);
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterEndDocument(writer);
    W_CFG_RETURN_IF_FAIL(rc);

    xmlFreeTextWriter(writer);

    // TODO: use xmlSaveFormatFile() or similar
    FILE* config = fopen(filename, "w");
    if (!config)
        W_CFG_RETURN_IF_FAIL(-1);

    xmlBufferDump(config, buf);
    fclose(config);

    xmlBufferFree(buf);

    return TRUE;
}

static gboolean write_ap_cfg(xmlTextWriterPtr writer, GSList* ap_list)
{
    int rc;

    for (GSList* iter = ap_list; iter; iter = g_slist_next(iter)) {
        // TODO: set name to 'unknown', set 'disabled' property to True
        AccessPoint_s* ap = AP_LIST_CAST(iter);
        if (!ap->name || !g_utf8_strlen(ap->name, -1)) {
            debug_msg("AP name must be set. Skipping.");
            continue;
        }

        rc = xmlTextWriterStartElement(writer, BAD_CAST(AP_CFG_ELEMENT));
        g_return_val_if_fail(rc >= 0, FALSE);

        rc = xmlTextWriterWriteAttribute(writer, BAD_CAST(NAME_CFG_ATTR),
                                                 BAD_CAST(ap->name));
        g_return_val_if_fail(rc >= 0, FALSE);

        if (ap->disabled) {
            rc = xmlTextWriterWriteAttribute(writer,
                                             BAD_CAST(DISABLED_CFG_ATTR),
                                             BAD_CAST(TRUE_BOOL));
            g_return_val_if_fail(rc >= 0, FALSE);
        }

        rc = xmlTextWriterStartElement(writer, BAD_CAST(CMDS_CFG_ELEMENT));
        g_return_val_if_fail(rc >= 0, FALSE);

        /* Write commands */
        for (GSList* cmd_iter = ap->commands; 
                                        cmd_iter;
                                        cmd_iter = g_slist_next(cmd_iter)) {
            
            gchar* command = (gchar *)cmd_iter->data;
            rc = xmlTextWriterWriteElement(writer, BAD_CAST(CMD_CFG_ELEMENT),
                                                   BAD_CAST(command));
        }

        /* Close 'commands' element */
        rc = xmlTextWriterEndElement(writer);
        g_return_val_if_fail(rc >= 0, FALSE);

        /* Close 'access-point' element */
        rc = xmlTextWriterEndElement(writer);
        g_return_val_if_fail(rc >= 0, FALSE);
    }

    return TRUE;
}


void free_cfg(ConlerCfg_s* cfg)
{

    if (cfg->access_points) {
        g_slist_foreach(cfg->access_points, (GFunc)free_access_point, NULL);
//        g_slist_foreach(cfg->access_points, (GFunc)g_free, NULL);
        g_slist_free(cfg->access_points);
    }

    if (cfg->version)
        g_free(cfg->version);
}

static AccessPoint_s* node_as_ap(const xmlDocPtr const doc,\
                                 const xmlNodePtr const node)
{
    AccessPoint_s* ap = NULL;
    xmlNodePtr cur_node;
    static guint unknown_id = 0;

    gchar* name = (gchar *)xmlGetProp(node, BAD_CAST(NAME_CFG_ATTR));

    /* AccessPoint is not valid if:
     *  - no children
     *  - next is not a CMDS_CFG_ELEMENT
     */
    cur_node = get_element_node(node, CMDS_CFG_ELEMENT);
    if (!cur_node || xmlStrcmp(cur_node->name, BAD_CAST(CMDS_CFG_ELEMENT))) 
        return NULL;

    ap = g_new(AccessPoint_s, 1);
    g_assert(ap != NULL);
   
    if (!name) {
        ap->name = g_strdup_printf(UNKNOWN_AP_FMT, unknown_id++);
        ap->disabled = TRUE;
    }
    else
    {
        ap->name = name;
        ap->disabled = FALSE;
    }

    ap->commands = NULL;

    /* Check if AP is disabled */
    xmlChar* disabled = xmlGetProp(node, BAD_CAST(DISABLED_CFG_ATTR));
    if (disabled) {
        ap->disabled = !xmlStrcmp(disabled, BAD_CAST(TRUE_BOOL));
        xmlFree(disabled);
    }

    /* Collect AP's commands */
    for (cur_node = cur_node->xmlChildrenNode; cur_node;\
                                               cur_node = cur_node->next) {
        SKIP_NODE_PROCESSING(cur_node, CMD_CFG_ELEMENT);

        // TODO: disabled command attribute
        gchar* cmd = g_strdup((const gchar *)xmlNodeGetContent(cur_node));
        APPEND2SLIST(ap->commands, cmd);
    }

    return ap;
}




static xmlNodePtr get_element_node(const xmlNodePtr const node,\
                                   const char* const name)
{
    xmlNodePtr cur_node = NULL; 

    for (cur_node = node->children; cur_node; cur_node = cur_node->next) {
        SKIP_NODETYPE_PROCESSING(cur_node, XML_ELEMENT_NODE);

        if (!xmlStrcmp(cur_node->name, BAD_CAST(name)))
            return cur_node;
    }

    return NULL;
}

void free_command_list(GSList* cmd_list)
{
    if (!cmd_list)
        return;

    g_slist_foreach(cmd_list, (GFunc)g_free, NULL);
    g_slist_free(cmd_list);
}

void free_access_point(AccessPoint_s* ap)
{
    if (!ap)
        return; 

    if (ap->name)
        g_free(ap->name);

    free_command_list(ap->commands);
    g_free(ap);
}

AccessPoint_s* copy_access_point(AccessPoint_s* ap)
{
    AccessPoint_s* new_ap = g_new(AccessPoint_s, 1);
    g_assert(new_ap != NULL);

    new_ap->name = g_strdup(ap->name);
    new_ap->disabled = ap->disabled;
    new_ap->commands = NULL;

    for (GSList* iter = ap->commands; iter; iter = g_slist_next(iter))
        APPEND2SLIST(new_ap->commands, g_strdup((gchar *)(iter->data)));

    return new_ap;
}

// TODO: dump_config(), dump_AP()
